;; src/pod/core.clj
(ns pod.core
  (:require [bencode.core :refer [read-bencode write-bencode]]
            [cheshire.core :as json]
            [clojure.string :as s])
  (:import [java.io PushbackInputStream]))

;; ——————————————————————————————————
;; Helpers
;; ——————————————————————————————————

(defn bytes->str [^bytes b]
  (String. b "UTF-8"))

(defn normalize-msg [m]
  (into {}
        (for [[k v] m]
          [(if (bytes? k) (bytes->str k) k)
           (cond
             (map? v) (normalize-msg v)
             (bytes? v) (bytes->str v)
             :else v)])))

;; ——————————————————————————————————
;; I/O
;; ——————————————————————————————————

(def stdin (PushbackInputStream. System/in))

(defn read-msg []
  (normalize-msg (read-bencode stdin)))

(defn write-msg [msg]
  (write-bencode System/out msg)
  (.flush System/out))

;; ——————————————————————————————————
;; Pod Protocol Helpers
;; ——————————————————————————————————

(defn make-describe [ns-name fns]
  {"format" "json"
   "namespaces"
   [{"name" ns-name
     "vars" (vec (for [fname (keys fns)]
                   {"name" (last (s/split fname #"/"))}))}]
   "ops" {"shutdown" {}}})

(defn safe-invoke [f args]
  (let [parsed-args (cond
                      (string? args)
                      (try
                        (json/parse-string args)
                        (catch Exception _ [args]))

                      (sequential? args) args
                      :else [args])]
    (json/generate-string (apply f parsed-args))))

;; ——————————————————————————————————
;; Main Loop (configurable)
;; ——————————————————————————————————

(defn pod-main
  "Start a pod main loop.
   - `describe-fn`: () -> describe map
   - `invoke-fn`: (var-name args) -> JSON string result"
  [describe-fn invoke-fn]
  (binding [*out* *err*]
    (println "Pod started. Waiting for messages from babashka..."))
  (try
    (loop []
      (let [msg (try
                  (read-msg)
                  (catch java.io.EOFException _ nil))]
        (when msg
          (binding [*out* *err*]
            (println "Received msg:" msg))
          (let [op (get msg "op")
                id (get msg "id")]
            (case op
              "describe"
              (do
                (write-msg (describe-fn))
                (binding [*out* *err*]
                  (println "Sent describe response.")))

              "invoke"
              (try
                (let [var (get msg "var")
                      args (get msg "args")
                      value (invoke-fn var args)]
                  (write-msg {"value" value
                              "id" id
                              "status" ["done"]}))
                (catch Throwable e
                  (write-msg {"ex-message" (.getMessage e)
                              "id" id
                              "status" ["done" "error"]})))

              "shutdown"
              (do
                (binding [*out* *err*]
                  (println "Pod shutting down."))
                (System/exit 0))

              (write-msg {"ex-message" (str "Unknown op: " op)
                          "id" id
                          "status" ["done" "error"]})))
          (recur))))
    (catch Throwable e
      (binding [*out* *err*]
        (println "Fatal error:" (.getMessage e))
        (.printStackTrace e))
      (System/exit 1))))
