(ns com.timezynk.useful.mongo.queue
  (:require
   [clojure.tools.logging :as log]
   [com.timezynk.useful.prometheus.core :as prom]
   [com.timezynk.useful.time :as t]
   [com.timezynk.useful.timed-queue :refer [TimedQueue]]
   [somnium.congomongo :as mongo]))

(def ^:const DEFAULT_NUM_WORKERS 1)
(def ^:const DEFAULT_MIN_INTERVAL 250)
(def ^:const DEFAULT_MIN_SLEEP 100)

(def ^:private ^:const JOIN_TIMEOUT
  "Maximum number of milliseconds to wait for worker threads to finish."
  20000)

(defprotocol WorkQueue
  (start-workers! [this thread-name] "Starts all worker threads.")
  (stop-workers! [this] "Stops all worker threads."))

(defn- pop-job! [queue]
  (mongo/fetch-and-modify (.collection queue)
                          {:run-at {:$lte (System/currentTimeMillis)}}
                          nil
                          :remove? true
                          :sort {:run-at 1}))

(defn- report-job-stats
  "Reports to Prometheus on the processing of `job`.
   Expects time in milliseconds."
  [queue job run-time wait-time]
  (let [job-type (:type job)]
    (log/debug "Job" (:type job) "in" (.collection queue) "is ready to run."
               "Latency" wait-time "ms")
    (-> queue (.run-counter) (prom/inc-by! (/ run-time 1000.0) job-type))
    (-> queue (.run-gauge) (prom/gauge-with-labels job-type) (.set run-time))
    (-> queue (.wait-gauge) (prom/gauge-with-labels job-type) (.set wait-time))
    (-> queue (.job-counter) (prom/inc! job-type))))

(defn- mark-drained
  "Reports to Prometheus that the queue has been drained."
  [queue]
  (-> queue .run-gauge .clear)
  (-> queue .wait-gauge .clear))

(defmacro ^:private report
  "Wraps `handle-form` such as to log the event and record statistics."
  [queue job handle-form]
  `(if ~job
     (let [latency# (- (System/currentTimeMillis) (:run-at ~job))
           start-time# (System/nanoTime)
           _# ~handle-form
           diff-ns# (-> (System/nanoTime) (- start-time#) (double))
           diff-ms# (/ diff-ns# 1000000.0)]
       (report-job-stats ~queue ~job diff-ms# latency#))
     (mark-drained ~queue)))

(defn- run-job! [queue job]
  (when job
    ((.handler queue) job)))

(defn- process-job! [queue job]
  (->> job (run-job! queue) (report queue job)))

(defn- add-payload-prefix [acc [k v]]
  (assoc acc (str "payload." (name k)) v))

(defn- processing-loop
  "Processes `queue` until the termination condition is met."
  [queue]
  (let [collection (.collection queue)
        go-on? (.go-on? queue)
        min-interval (.min-interval queue)
        min-sleep (.min-sleep queue)]
    (log/info "Starting workers for queue" collection)
    (while @go-on?
      (try
        (->> (pop-job! queue)
             (process-job! queue)
             (t/sleep-pad min-interval min-sleep))
        (catch Exception e
          (log/error e "Exception in queue" collection))))
    (log/info "Queue" collection "finished")))

(deftype MongoQueue
         [collection handler num-workers thread-priority min-interval
          min-sleep go-on? worker-threads
          run-counter job-counter run-gauge wait-gauge]

  WorkQueue
  (start-workers! [this thread-name]
    (reset! go-on? true)
    (dotimes [i num-workers]
      (swap! worker-threads
             conj
             (doto (Thread. (bound-fn [] (processing-loop this))
                            (if (= 1 num-workers)
                              thread-name
                              (str thread-name "-" i)))
               (.setDaemon false)
               (.setPriority thread-priority)
               (.start)))))

  (stop-workers! [_this]
    (reset! go-on? false)
    (when (seq @worker-threads)
      (run! #(.join % JOIN_TIMEOUT) @worker-threads)
      (reset! worker-threads [])
      (prom/unregister run-counter)
      (prom/unregister job-counter)
      (prom/unregister run-gauge)
      (prom/unregister wait-gauge)))

  TimedQueue
  (push-job! [_this type run-at payload]
    (mongo/insert! collection
                   {:run-at run-at
                    :type type
                    :payload payload}
                   :write-concern :unacknowledged))

  (upsert-job! [_this type run-at selector update]
    (mongo/update! collection
                   (merge
                    {:type type}
                    (reduce add-payload-prefix {} selector))
                   (merge
                    {:$set
                     {:run-at run-at
                      :type type}}
                    update)
                   :upsert true
                   :write-concern :unacknowledged)))

(defn create [& {:as params}]
  (let [{:keys [id collection handler num-workers
                thread-priority min-interval min-sleep]} params]
    (MongoQueue. collection
                 handler
                 (or num-workers DEFAULT_NUM_WORKERS)
                 (or thread-priority Thread/MIN_PRIORITY)
                 (or min-interval DEFAULT_MIN_INTERVAL)
                 (or min-sleep DEFAULT_MIN_SLEEP)
                 (atom true)
                 (atom [])
                 (prom/counter (keyword (str (name id) "_user_time_seconds"))
                               (str "Total user time of " id)
                               :type)
                 (prom/counter (keyword (str (name id) "_messages_total"))
                               (str "Messages processed in " id)
                               :type)
                 (prom/gauge (keyword (str (name id) "_milliseconds_executing"))
                             (str "Time to run a single " (name id) " job (ms)")
                             :type)
                 (prom/gauge (keyword (str (name id) "_milliseconds_waiting"))
                             (str "Latency of a single " (name id) " job (ms)")
                             :type))))
