(ns com.timezynk.useful.time
  (:require [clojure.test :refer [is]]))

(defmacro sleep-pad
  "Makes `body` last at least as long as `min-duration` by sleeping after.
   Measures `min-duration` and `min-sleep` in milliseconds."
  [min-duration min-sleep & body]
  `(let [next-ts# (+ (System/currentTimeMillis) ~min-duration)]
     ~@body
     (Thread/sleep (max ~min-sleep
                        (- next-ts# (System/currentTimeMillis))))))

(defn exponential-backoff-interval
  "Length of time to wait for when following the EBO strategy."
  [num-failures base-interval max-interval]
  (try
    (->> (or num-failures 0)
         (dec)
         (Math/pow 2)
         (long)
         (Math/max 1)
         (* base-interval)
         (long))
    (catch Exception _
      max-interval)))

(defmacro wait-till
  "Puts the calling thread to sleep until `pred` becomes true.
   Wakes up to evaluate `pred` every `interval` (default 250) milliseconds.
   Asserts failure after crossing the attempt threshold (default 5).
   Intended for use within `clojure.test` tests."
  ([pred interval max-attempts]
   `(loop [attempt# 1]
      (if (> attempt# ~max-attempts)
        (is false "wait condition not fulfilled")
        (when-not ~pred
          (Thread/sleep ~interval)
          (recur (inc attempt#))))))
  ([pred]
   `(wait-till ~pred 250 5)))

(defn current-time-ms
  "Wraps `System/currentTimeMillis` to make it `redef`-able."
  []
  (System/currentTimeMillis))
