(ns com.timezynk.useful.logger
  (:refer-clojure :exclude [name])
  (:require [clojure.tools.logging.impl :as impl]
            [com.timezynk.useful.env :as env]
            [com.timezynk.useful.string :as string])
  (:import [com.slack.api Slack]
           [com.slack.api.methods.request.chat ChatPostMessageRequest]))

(defn- post-on-slack
  "Posts `message` on the #backend-errors Slack channel.
   Returns `ChatPostMessageResponse` on success, nil on failure.

   Expects the `ERROR_RELAY_SLACK_TOKEN` environment variable to hold a
   sufficiently authorized Slack token."
  [message]
  (when-let [token (System/getenv "ERROR_RELAY_SLACK_TOKEN")]
    (try
      (let [slack-methods (-> (Slack/getInstance) (.methods token))
            request (-> (ChatPostMessageRequest/builder)
                        (.channel "#backend-errors")
                        (.text message)
                        (.build))]
        (.chatPostMessage slack-methods request))
      (catch Exception _))))

(defrecord BotWrapper [inner]
  impl/Logger

  (enabled? [_ level]
    (impl/enabled? inner level))

  (write! [_ level throwable message]
    (when (and env/production? (= :error (keyword level)))
      (post-on-slack (cond-> message
                       throwable (str "\n" (string/from-throwable throwable)))))
    (impl/write! inner level throwable message)))

(defrecord BotWrapperFactory [inner]
  impl/LoggerFactory

  (name [_]
    (impl/name inner))

  (get-logger [_ logger-ns]
    (-> inner
        (impl/get-logger logger-ns)
        (->BotWrapper))))

(defn factory
  "Creates and wraps a SLF4J factory so that it posts ERROR messages on Slack."
  []
  (->BotWrapperFactory (impl/slf4j-factory)))
