(ns com.timezynk.useful.hash
  (:require [com.timezynk.useful.base64 :refer [base64-str base64-str-urlsafe]])
  (:import [org.apache.commons.codec.binary Hex]))

(defn digest [^String algo]
  (let [md (java.security.MessageDigest/getInstance algo)]
    (fn [bytes]
      (.digest md bytes))))

(def sha256 (digest "SHA-256"))

(def md5 (digest "MD5"))

(defn hmac [^String algo ^"[B" bytes ^"[B" key-bytes]
  (let [mac-key (javax.crypto.spec.SecretKeySpec. key-bytes algo)
        mac (doto (javax.crypto.Mac/getInstance algo) (.init mac-key))]
    (.doFinal mac bytes)))

(def hmac-sha256 (partial hmac "HmacSHA256"))

(defn b64-hmac-sha256 [^String data ^String hmac-key]
  (base64-str (hmac-sha256 (.getBytes data "UTF-8") (.getBytes hmac-key "UTF-8"))))

(def hmac-sha1 (partial hmac "HmacSHA1"))

(defn b64-hmac-sha1 [^String data ^String hmac-key]
  (base64-str (hmac-sha1 (.getBytes data "UTF-8") (.getBytes hmac-key "UTF-8"))))

(defn b64-md5 [^String data]
  (base64-str (md5 (.getBytes data "UTF-8"))))

(defn b64-urlsafe-md5 [^String data]
  (base64-str-urlsafe (md5 (.getBytes data "UTF-8"))))

(defn hex-md5 [^String data]
  (String. (Hex/encodeHex (md5 (.getBytes data "UTF-8")))))

(defn b64-sha256 [^String data]
  (base64-str (sha256 (.getBytes data "UTF-8"))))
