(ns com.timezynk.useful.channel
  (:require
   [clojure.tools.logging :as log :refer [debug info warn]]
   [com.timezynk.useful.channel.subscriber :as subscriber
                                           :refer [->RequestResponseSubscriber
                                                   ->BroadcastSubscriber]]
   [com.timezynk.useful.channel.subscriber.hook :as hook]
   [com.timezynk.useful.channel.task :as task]
   [com.timezynk.useful.env :as env]
   [com.timezynk.useful.prometheus.core :as prometheus])
  (:import [java.util.concurrent LinkedBlockingQueue
            PriorityBlockingQueue
            BlockingQueue
            TimeUnit]
           [java.util UUID]))

(defonce ^{:dynamic true} *debug* true)

(def ^:const NUM_WORKERS 2)

(def ^:const ^:private WAIT_THRESHOLD
  "Maximum number of milliseconds to spend in queue before emitting a warning."
  2000)

(def ^:const ^:private RUN_THRESHOLD
  "Maximum number of milliseconds to spend running before emitting a warning."
  1000)

(defonce current-message-id (atom 0))

(def queue-size
  (prometheus/gauge :channel_queue_size
                    "Number of actions waiting in the channel queue"
                    :queue_id))

(def processed-messages
  (prometheus/counter :channel_processed_total
                      "Number of actions processed by the channel queue"
                      :queue_id))

(def timed-out-messages
  (prometheus/counter :channel_timed_out_total
                      "Number of actions exceeding the wait-time threshold"))

(def time-in-queue
  (prometheus/gauge :channel_time_in_queue
                    "Time waiting in the channel queue (ms)"
                    :queue_id))

(def time-executing
  (prometheus/gauge :channel_time_executing
                    "Time executing the task (ms)"
                    :queue_id :collection :action :hook))

(defn publish! ^BlockingQueue
  ([^BlockingQueue channel context topic cname messages]
   (when (seq messages)
     (let [reply-channel (LinkedBlockingQueue.)]
       (doseq [m messages
               s (subscriber/eligible-for topic cname)]
         (try
           (.put channel
                 (subscriber/publish s
                                     topic
                                     cname
                                     m
                                     (or task/*reply-channel* reply-channel)
                                     context))
           (catch Exception e
             (warn e topic cname "failed to publish" m))))
       (.put reply-channel [:queued-message-tasks])
       reply-channel)))
  ([^BlockingQueue channel topic cname messages]
   (publish! channel nil topic cname messages)))

(defn- on-empty [_]
  (log/trace "completed. No tasks to wait for")
  true)

(defn- on-finished [_]
  (log/trace "completed. All tasks finished")
  true)

(defn- on-timeout [tasks]
  (prometheus/inc! timed-out-messages)
  (info "timeout. Still waiting for" tasks)
  false)

(defn wait-for [timeout-ms reply-channel]
  (when reply-channel
    (let [poll #(.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)]
      (loop [[event id payload] (poll)
             tasks #{}]
        (log/trace "event" event id "waiting for" tasks)
        (case event
          :queued-message-tasks (if (empty? tasks)
                                  (on-empty tasks)
                                  (recur (poll) tasks))

          :queued               (recur (poll) (conj tasks id))

          :started              (recur (poll) tasks)

          :finished             (let [new-tasks (disj tasks id)]
                                  (if (empty? new-tasks)
                                    (on-finished new-tasks)
                                    (recur (poll) new-tasks)))

          :exception            (throw payload)

          (if (seq tasks)
            (on-timeout tasks)
            (recur (poll) tasks)))))))

(defn- wrapv
  "Turns `x` into `[x]`, unless it is sequential."
  [x]
  (cond-> x
    (not (sequential? x)) (vector)))

(defn- subscribe [topic cname f factory]
  (doseq [t (wrapv topic)
          c (wrapv cname)]
    (subscriber/add t (factory c f))))

(defn subscribe-broadcast [topic collection-name f]
  (when (and topic f)
    (debug topic collection-name "new broadcast subscriber")
    (subscribe topic collection-name f ->BroadcastSubscriber)))

(defn subscribe-request-response [topic collection-name f]
  (when (and topic f)
    (debug topic collection-name "new request-response subscriber")
    (subscribe topic collection-name f ->RequestResponseSubscriber)))

(defn unsubscribe-all []
  (subscriber/remove-all))

(defn- log-wait-time [message queue-id millis]
  (when millis
    (-> time-in-queue (prometheus/gauge-with-labels queue-id) (.set millis))
    (log/logp (if (> millis WAIT_THRESHOLD) :warn :debug)
              (format "%s, W:%d" message millis))))

(defn- log-run-time [message queue-id millis]
  (when millis
    (let [{:keys [topic cname subscriber]} (:task message)
          ->label (comp name keyword)]
      (-> time-executing
          (prometheus/gauge-with-labels queue-id
                                        (->label cname)
                                        (->label topic)
                                        (hook/pretty-print (:f subscriber)))
          (.set millis))
      (log/logp (if (> millis RUN_THRESHOLD) :warn :debug)
                (format "%s, R:%d" message millis)))))

(defn route-message [message message-counter queue-id]
  (when-let [t (:task message)]
    (let [[start-at end-at] (task/process t)
          wait-time (and start-at (- start-at (:enqueued-at message)))
          run-time (and start-at end-at (- end-at start-at))]
      (log-wait-time message queue-id wait-time)
      (log-run-time message queue-id run-time)))
  (.inc message-counter))

(defn broker-loop [^BlockingQueue channel queue-id]
  (fn []
    (info "starting message broker")
    (let [size-gauge (prometheus/gauge-with-labels queue-size queue-id)
          message-counter (prometheus/counter-with-labels processed-messages queue-id)]
      (while true
        (try
          (route-message (.take channel) message-counter queue-id)
          (.set size-gauge (.size channel))
          (when-not env/test?
            (Thread/sleep 50))
          (catch Exception e
            (warn e "Exception in channel broker")
            (Thread/sleep 100)))))))

(defn create-broker! ^BlockingQueue [^BlockingQueue channel queue-id]
  (dotimes [i NUM_WORKERS]
    (doto (Thread. (broker-loop channel queue-id) (str "mchan-" i))
      (.setDaemon true)
      (.setPriority Thread/MIN_PRIORITY)
      (.start)))
  channel)

(defn start-channel! ^BlockingQueue []
  (create-broker! (PriorityBlockingQueue.) (str (UUID/randomUUID))))

