(ns com.timezynk.useful.channel.task
  (:require [clojure.tools.logging :as log]
            [com.timezynk.useful.channel.subscriber.hook :as hook]
            [com.timezynk.useful.prometheus.core :as prometheus]))

(def ^{:dynamic true} *reply-channel* nil)

(defonce task-errors
  (prometheus/counter
    :channel_errors_total
    "Number of hooks which have thrown an uncaught exception"))

(defprotocol MessageTask
  (process [task] "Processes the task. Returns a vector of start and finish timestamps."))

(defn- run-subscriber [task subscriber]
  (let [{:keys [context topic cname message]} task
        start-at (System/currentTimeMillis)
        result (hook/call (:f subscriber) topic cname context message)
        end-at (System/currentTimeMillis)]
    [result start-at end-at]))

(defrecord RequestResponseTask [subscriber task-id topic cname message reply-channel context]
  MessageTask
  (process [task]
    (.put reply-channel [:started task-id])
    (log/debug topic cname "running request-response task" task-id)
    (try
      ; Bind dynamic reply channel so that recursive tasks are collected in the outermost
      ; wait-for
      (binding [*reply-channel* reply-channel]
        (let [[result start-at end-at] (run-subscriber task subscriber)]
          (.put reply-channel [:finished task-id result])
          [start-at end-at]))
      (catch Exception e
        (log/error e topic cname "request-response failed to run")
        (.inc task-errors)
        (.put reply-channel [:exception task-id e])))))

(defrecord BroadcastTask [subscriber topic cname message context]
  MessageTask
  (process [task]
    (log/debug topic cname "running broadcast task")
    (try
      (subvec (run-subscriber task subscriber) 1)
      (catch Exception e
        (log/error e topic cname "broadcast failed to run")
        (.inc task-errors)))))
