(ns com.timezynk.useful.prometheus.middleware
  (:require
   [clojure.string :as string]
   [clojure.tools.logging :as log]
   [com.timezynk.useful.prometheus.core :as metrics]
   [com.timezynk.useful.measured-future :refer [thread-cpu-time]]
   [clojure.test :refer [deftest is testing]]
   [com.timezynk.useful.date :refer [iso-pattern]]))

(def ^:const UA_PRODUCTS #"[^ ()/]+/[^ ()/]+")
(def ^:const SLOW_LOG_LIMIT 0.5)

(defonce request-counter (metrics/counter :http_requests_total
                                          "A counter of the total number of HTTP requests processed"
                                          :method :status :user_agent :version))

(defonce request-seconds (metrics/counter :http_requests_seconds
                                          "A counter of the total seconds spent processing HTTP requests"
                                          :method :uri))

(defonce histogram (metrics/histogram :http_request_latency_seconds
                                      "A histogram of the response latency for HTTP requests in seconds."
                                      [0.001, 0.005, 0.010, 0.020, 0.050, 0.100, 0.200, 0.300, 0.500, 0.750, 1, 2, 4, 8]
                                      :method :status_class))

(defn get-products [user-agent]
  (string/join " "
               (re-seq UA_PRODUCTS
                       (string/replace user-agent "Mozilla/5.0" ""))))

(defn get-client [{:keys [headers]}]
  (let [user-agent (or (get headers "user-agent") "")]
    (get-products user-agent)))

(defn get-uri [{:keys [uri]}]
  (-> uri
      (string/replace #"(?<=\/settings\/v\d+\/)[^/]+\/[^/]+(?=\/)?" ":target/:data")
      (string/replace #"[0-9a-f]{64}" ":reg-id")
      (string/replace #"[0-9a-f]{24}" ":oid")
      (string/replace #"\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}(\.\d{3})?" ":datetime")
      (string/replace #"2\d{3}/[1-9][0-2]?" ":year/:month")
      (string/replace iso-pattern ":date")
      (string/replace #"[a-zA-Z0-9\-_%]{100,}" ":fcm-token")
      (string/replace #"(?<=username\/)[^/]+(?=\/)?" ":un")
      (string/replace #"(?<=permissions\/)[^/]+(?=\/)?" ":per")
      (string/replace #"(?<=\/)\d+(?=\/)?" ":no")
      (string/replace #"(?<=\/registry-fields\/)[^/]+(?=\/)?" ":static")))

(defn wrap-metrics [handler version]
  (fn [request]
    (let [request-method (:request-method request)
          start-cpu-time (thread-cpu-time)
          start-time (System/currentTimeMillis)
          response (handler request)
          finish-time (System/currentTimeMillis)
          finish-cpu-time (thread-cpu-time)
          cpu-time-diff (/ (double (- finish-cpu-time start-cpu-time)) 1000000000.0)
          response-status (get response :status 404)
          request-time (/ (double (- finish-time start-time)) 1000.0)
          status-class (str (int (/ response-status 100)) "XX")
          method-label (string/upper-case (name request-method))
          client (get-client request)
          uri (get-uri request)]
      (apply metrics/observe! histogram request-time [method-label status-class])
      (apply metrics/inc! request-counter [method-label (str response-status) client version])
      (apply metrics/inc-by! request-seconds cpu-time-diff [method-label uri])
      (when (> request-time SLOW_LOG_LIMIT)
        (log/warn "Slow request" request-time "seconds:" (.toUpperCase (name request-method)) (:uri request)))
      response)))

(deftest anonymize-uri
  (testing "replacing object id"
    (is (= "/api/flexitime/:oid" (get-uri {:uri "/api/flexitime/617a641e364296cd6f83f7c8"})))
    (is (= "/api/shifts/:oid/booked-users" (get-uri {:uri "/api/shifts/617a6a6557261bb052a00fda/booked-users"}))))
  (testing "replacing dates"
    (is (= "/api/flexitime/:oid/diff/:date/" (get-uri {:uri "/api/flexitime/617a641e364296cd6f83f7c8/diff/2021-09-10/"}))))
  (testing "replacing datetime"
    (is (= "/api/wat/:datetime/summary" (get-uri {:uri "/api/wat/2021-09-10T08:29:30/summary"}))))
  (testing "replacing year/month"
    (is (= "/api/wat/:year/:month" (get-uri {:uri "/api/wat/2021/9"})))
    (is (= "/api/wat/:year/:month" (get-uri {:uri "/api/wat/2021/10"}))))
  (testing "replacing fcm token"
    (is (= "/api/device/fcm/:oid/:fcm-token" (get-uri {:uri "/api/device/fcm/617a641e364296cd6f83f7c8/eYWHDjnI0ik%3AAPA91bG-3O1LTFCp0smdwJ8z3G0SVqO3aFgnQxq7ZcvyHj3q3tRtzD9cNYS2-VHaQnirhwHF2lqu9Yb9p3TMvtREdhKmjWMnFk0SbISWXrhDC2lyJmqzJJVViFGKuCx2viIgfz8j_mbJ"}))))
  (testing "replacing company settings"
    (is (= "/api/settings/v1/:target/:data" (get-uri {:uri "/api/settings/v1/company/eaccounting-integration"})))
    (is (= "/api/settings/v1/:target/:data/archive" (get-uri {:uri "/api/settings/v1/company/eaccounting-integration/archive"}))))
  (testing "replacing personal settings"
    (is (= "/api/settings/v1/:target/:data" (get-uri {:uri "/api/settings/v1/personal/filtertabs%2Ftimestamps-view"})))
    (is (= "/api/settings/v1/:target/:data/archive" (get-uri {:uri "/api/settings/v1/personal/filtertabs%2Ftimestamps-view/archive"}))))
  (testing "replacing apple device id"
    (is (= "/api/device/ios/:oid/:reg-id" (get-uri {:uri "/api/device/ios/617a641e364296cd6f83f7c8/617a641e364296cd6f83f7c8617a641e364296cd6f83f7c86b71c9053daa37b8"}))))
  (testing "replacing username"
    (is (= "/api/username/:un" (get-uri {:uri "/api/username/user%40example.com"})))
    (is (= "/api/username/:un/promote" (get-uri {:uri "/api/username/user%40example.com/promote"}))))
  (testing "replacing permissions id"
    (is (= "/api/permissions/:per" (get-uri {:uri "/api/permissions/4%2520srskilt"})))
    (is (= "/api/permissions/:per/users" (get-uri {:uri "/api/permissions/4%2520srskilt/users"}))))
  (testing "replacing employee number"
    (is (= "/api/validate-employee-number/:no" (get-uri {:uri "/api/validate-employee-number/718"}))))
  (testing "replacing invoice number"
    (is (= "/api/usage-invoice/:no/reload" (get-uri {:uri "/api/usage-invoice/12345/reload"}))))
  (testing "replacing static registry field ID"
    (is (= "/api/registry-fields/:static" (get-uri {:uri "/api/registry-fields/shift-breaks"})))
    (is (= "/api/registry-fields/:static" (get-uri {:uri "/api/registry-fields/shift-duplicity"})))))
