(ns com.timezynk.useful.channel
  (:require
   [clojure.tools.logging :as log :refer [debug info warn]]
   [com.timezynk.useful.channel.message :as message]
   [com.timezynk.useful.channel.subscriber :as subscriber
                                           :refer [->RequestResponseSubscriber
                                                   ->BroadcastSubscriber]]
   [com.timezynk.useful.channel.task :as task]
   [com.timezynk.useful.prometheus.core :as prometheus])
  (:import [java.util.concurrent LinkedBlockingQueue
            PriorityBlockingQueue
            BlockingQueue
            TimeUnit]
           [java.util UUID]))

(defonce ^{:dynamic true} *debug* true)

(def ^:const NUM_WORKERS 2)

(def ^:const ^:private WAIT_THRESHOLD
  "Maximum number of milliseconds to spend in queue before emitting a warning."
  2000)

(def ^:const ^:private RUN_THRESHOLD
  "Maximum number of milliseconds to spend running before emitting a warning."
  1000)

(defonce current-message-id (atom 0))

(def queue-size (prometheus/gauge :channel_queue_size "Number of actions waiting in the channel queue" :queue_id))
(def processed-messages (prometheus/counter :channel_processed_total "Number of actions processed by the channel queue" :queue_id))

(defn publish! ^BlockingQueue
  ([^BlockingQueue channel context topic cname messages]
   (when (seq messages)
     (let [reply-channel (LinkedBlockingQueue.)]
       (doseq [m messages
               s (subscriber/eligible-for topic cname)]
         (try
           (.put channel
                 (subscriber/publish s
                                     topic
                                     cname
                                     m
                                     (or task/*reply-channel* reply-channel)
                                     context))
           (catch Exception e
             (warn e topic cname "failed to publish" m))))
       (.put reply-channel [:queued-message-tasks])
       reply-channel)))
  ([^BlockingQueue channel topic cname messages]
   (publish! channel nil topic cname messages)))

(defn wait-for [timeout-ms reply-channel]
  (when reply-channel
    (loop [[event id payload] (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
           tasks #{}]
      (debug "event" event id "waiting for" tasks)
      (case event
        :queued-message-tasks (if (empty? tasks)
                                (do
                                  (debug "completed. No tasks to wait for")
                                  true)
                                (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                                       tasks))

        :queued               (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                                     (conj tasks id))

        :started              (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                                     tasks)

        :finished             (let [new-tasks (disj tasks id)]
                                (if (empty? new-tasks)
                                  (do
                                    (debug "completed. All tasks finished")
                                    true)
                                  (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                                         new-tasks)))

        :exception            (throw payload)

        (if (seq tasks)
          (do
            (info "timeout. Still waiting for" tasks)
            false)
          (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                 tasks))))))

(defn- subscribe [topic cname f factory]
  (let [topics (cond-> topic
                 (not (sequential? topic)) (vector))]
    (doseq [topic topics]
      (subscriber/add topic (factory cname f)))))

(defn subscribe-broadcast [topic collection-name f]
  (when (and topic f)
    (debug topic collection-name "new broadcast subscriber")
    (subscribe topic collection-name f ->BroadcastSubscriber)))

(defn subscribe-request-response [topic collection-name f]
  (when (and topic f)
    (debug topic collection-name "new request-response subscriber")
    (subscribe topic collection-name f ->RequestResponseSubscriber)))

(defn unsubscribe-all []
  (subscriber/remove-all))

(defn route-message [message message-counter]
  (when-let [t (:task message)]
    (let [[start-at end-at] (task/process t)
          wait-time (- start-at (:enqueued-at message))
          run-time (- end-at start-at)]
      (when (or (> wait-time WAIT_THRESHOLD)
                (> run-time RUN_THRESHOLD))
        (warn (message/report message wait-time run-time)))))
  (.inc message-counter))

(defn broker-loop [^BlockingQueue channel queue-id]
  (fn []
    (info "starting message broker")
    (let [size-gauge (prometheus/gauge-with-labels queue-size queue-id)
          message-counter (prometheus/counter-with-labels processed-messages queue-id)]
      (while true
        (try
          (route-message (.take channel) message-counter)
          (.set size-gauge (.size channel))
          (catch Exception e
            (warn e "Exception in channel broker")
            (Thread/sleep 100)))))))

(defn create-broker! ^BlockingQueue [^BlockingQueue channel queue-id]
  (dotimes [i NUM_WORKERS]
    (doto (Thread. (broker-loop channel queue-id) (str "mchan-" i))
      (.setDaemon true)
      (.start)))
  channel)

(defn start-channel! ^BlockingQueue []
  (create-broker! (PriorityBlockingQueue.) (str (UUID/randomUUID))))

