(ns com.timezynk.useful.prometheus.middleware
  (:require
    [clojure.string :as string]
    [clojure.tools.logging :as log]
    [com.timezynk.useful.prometheus.core :as metrics]
  ))

(def ^:const UA_PRODUCTS #"[^ ()/]+/[^ ()/]+")
(def ^:const SLOW_LOG_LIMIT 0.5)

(defonce request-counter (metrics/counter :http_requests_total
                                      "A counter of the total number of HTTP requests processed"
                                      :method :status :user_agent :version))

(defonce histogram (metrics/histogram :http_request_latency_seconds
                                  "A histogram of the response latency for HTTP requests in seconds."
                                  [0.001, 0.005, 0.010, 0.020, 0.050, 0.100, 0.200, 0.300, 0.500, 0.750, 1, 2, 4, 8]
                                  :method :status_class))

(defn get-products [user-agent]
  (clojure.string/join " "
    (re-seq UA_PRODUCTS
      (clojure.string/replace user-agent "Mozilla/5.0" ""))))

(defn get-client [{:keys [headers query-params]}]
  (let [user-agent (or (get headers "user-agent") "")]
    (get-products user-agent)))

(defn wrap-metrics [handler version]
  (fn [request]
    (let [request-method (:request-method request)
          start-time (System/currentTimeMillis)
          response (handler request)
          finish-time (System/currentTimeMillis)
          response-status (get response :status 404)
          request-time (/ (double (- finish-time start-time)) 1000.0)
          status-class (str (int (/ response-status 100)) "XX")
          method-label (string/upper-case (name request-method))
          client (get-client request)]
      (apply metrics/observe! histogram request-time [method-label status-class])
      (apply metrics/inc! request-counter [method-label (str response-status) client version])
      (when (> request-time SLOW_LOG_LIMIT)
        (log/warn "Slow request" request-time "seconds:" (.toUpperCase (name request-method)) (:uri request)))
      response)))
