(ns com.timezynk.useful.rest.middleware
  (:use [slingshot.slingshot :only [try+]])
  (:require [clojure.spec.alpha :refer [check-asserts check-asserts?]]
            [clojure.tools.logging :as log]
            [cheshire.core :as json]
            [ring.util.response :as res]))

(def rng (java.util.Random. (System/currentTimeMillis)))
(def CLJ_ENV (or (System/getenv "CLJ_ENV") "development"))

(defn get-client-id [{:keys [headers]}]
  (or (get headers "x-tz-client")
      (get headers :x-tz-client)
      (get headers "user-agent")
      (get headers :user-agent)))

(defn ^Throwable get-cause [^Throwable t]
  (when t
    (let [^Throwable cause (.getCause t)]
      (if (or (nil? cause) (= cause t))
        t
        (recur cause)))))

(defn wrap-exceptions [handler]
  (fn [request]
    (try+
      (when-not (check-asserts?)
        (check-asserts true))
      (handler request)
      (catch [:clojure.spec.alpha/failure :assertion-failed] e
        (log/warn (get-cause (:throwable &throw-context)) "Spec validation exception 400" (:request-method request) (:uri request) ", client:" (get-client-id request))
        {:status 400
         :headers {"Content-Type" "application/json"
                   "Cache-Control" "no-cache"}
         :body (json/encode e)})
      (catch [:type :validation-error] e
        (log/warn (get-cause (:throwable &throw-context)) "Validation exception 400" (:request-method request) (:uri request) ", client:" (get-client-id request))
        {:status  400
         :headers {"Content-Type" "application/json"
                   "Cache-Control" "no-cache"}
         :body    (json/encode (:errors e))})
      (catch map? e
        (let [status (:code e 500)]
          (log/warn (get-cause (:throwable &throw-context)) "Caught slingshot exception" status (:request-method request) (:uri request) ", client:" (get-client-id request))
          {:status status
           :headers {"Content-Type" "application/json"
                     "Cache-Control" "no-cache"}
           :body    (json/encode e)}))
      (catch Exception e
        (log/error (get-cause e) "Caught exception 500" (:request-method request) (:uri request)
                   ", client:" (get-client-id request))
        {:status 500
         :headers {"Content-Type" "application/json"
                   "Cache-Control" "no-cache"}
         :body (json/encode (merge
                 {:error (.getMessage e)}
                 (when (= CLJ_ENV "development")
                   {:trace (map str (.getStackTrace e))})))}))))

(defn wrap-nocache [handler]
  (fn [request]
    (when-let [result (handler request)]
      (cond
        (= :post (:request-method request)) (res/header result "Cache-Control" "no-cache")
        (.endsWith ^String (:uri request) ".js") (res/header result "Cache-Control" "max-age=1,must-revalidate,public")
        :else result))))

(defn wrap-cors [handler]
  (fn [request]
    (if (= :options (:request-method request))
      (-> {:status 200
           :body ""}
        (res/header "Access-Control-Allow-Origin" "*")
        (res/header "Access-Control-Max-Age" "86400")
        (res/header "Access-Control-Allow-Headers" "content-type, accept, origin, authorization, x-tz-authorization, x-tz-client, user-agent, accept-encoding")
        (res/header "Access-Control-Allow-Methods" "GET, POST, PUT, PATCH, DELETE, OPTIONS"))
      (when-let [result (handler request)]
        (-> result
          (res/header "Access-Control-Allow-Origin" "*"))))))
