(ns com.timezynk.useful.cipher
  "AES cipher. Implementation inspired by
  http://stackoverflow.com/questions/992019/java-256bit-aes-encryption/"
  (:use [com.timezynk.useful.core :only [concat-bytes split-bytes]])
  (:import [javax.crypto SecretKey SecretKeyFactory Cipher]
           [javax.crypto.spec SecretKeySpec PBEKeySpec IvParameterSpec]))

(defn aes-key
  "Generate AES key from the given password and salt. Salt should be eight
  characters long. Bits can be 128, 192 or 256."
  [^String password ^String salt bits]
  (let [factory (SecretKeyFactory/getInstance "PBKDF2WithHmacSHA1")
        spec (PBEKeySpec. (.toCharArray password) (.getBytes salt "UTF-8") 1024 bits)
        secret (.generateSecret factory spec)]
    (SecretKeySpec. (.getEncoded secret) "AES")))

(defn ^"[B" aes-encrypt
  "Encrypt the byte array with the given key. Returns a byte-array
  with the first 16-bytes being the IV, suitable for decryption with
  aes-decrypt"
  [^SecretKey aes-key ^"[B" bytes]
  (let [cipher (doto (Cipher/getInstance "AES/CBC/PKCS5Padding")
                 (.init Cipher/ENCRYPT_MODE aes-key))]
    (concat-bytes (.getIV cipher) (.doFinal cipher bytes))))

(defn ^"[B" aes-decrypt
  "Decrypt the byte array with the given key. The first 16 bytes of the
  byte array must be the IV."
  [^SecretKey aes-key ^"[B" bytes]
  (let [[iv data] (split-bytes bytes 16)
        cipher (doto (Cipher/getInstance "AES/CBC/PKCS5Padding")
                 (.init Cipher/DECRYPT_MODE aes-key (IvParameterSpec. iv)))]
    (.doFinal cipher data)))
