(ns com.timezynk.useful.locale
  (:require [clojure.java.io :as io]
            [cheshire.core :as json])
  (:import [java.util Locale]
           [org.joda.time.format DateTimeFormatter DateTimeFormat]))

(def ^{:dynamic true} *locale* (delay nil))

(def ^:private translations-path (atom nil))

(defn set-translations-path! [path]
  (reset! translations-path path))

(def ^:private translations
  (delay (-> (io/resource @translations-path)
             (io/reader)
             (json/parse-stream true))))

(defn current-locale ^Locale []
  (when *locale*
    @*locale*))

(defn locale ^Locale [req]
  (if-let [user-lang (get-in req [:user :lang])]
    (Locale. (.substring user-lang 0 2))
    (if-let [servlet-req (:servlet-request req)]
      (.getLocale servlet-req)
      (let [^String accept-language (or (:lang req) (get-in req [:headers "accept-language"]) "en")]
        (Locale. (.substring accept-language 0 2))))))

(defmacro with-locale [req & body]
  `(let [l# (delay (locale ~req))]
     (binding [*locale* l#]
       ~@body)))

(defmacro with-locale* [lang & body]
  `(let [l# (delay (Locale. ~lang))]
     (binding [*locale* l#]
       ~@body)))

(defn user-locale ^Locale [user]
  (when-let [lang (:lang user)]
    (Locale. lang)))

(defmacro with-user-locale [user & body]
  `(let [l# (delay (user-locale ~user))]
     (binding [*locale* l#]
       ~@body)))

(defn wrap-locale [handler]
  (fn [request]
    (with-locale request
      (handler request))))

(defn localized-resource [basename extension]
  (let [path #(str basename % "." extension)
        ^ClassLoader loader (.getClassLoader ^Class (type localized-resource))]
    (or (when (current-locale)
          (.getResource loader (path (str  "_" (.getLanguage (current-locale))))))
        (.getResource loader (path "")))))

(defn translator
  ([] (translator nil))
  ([base-name]
   (let [lang (when-let [locale (current-locale)] (keyword (.getLanguage locale)))
         default-index (get-in @translations [:locales :default])
         index (or (get-in @translations [:locales lang]) default-index)]
     (fn [k]
       (when k
         (let [k (if base-name (keyword (str (name base-name) "-" (name k))) k)]
           (or
            (get-in @translations [:translations k index])
            (get-in @translations [:translations k default-index]))))))))

(defn date-pattern-formatter ^DateTimeFormatter [^String pattern]
  (.withLocale
   (DateTimeFormat/forPattern pattern)
   (current-locale)))

(defn date-formatter [^String style]
  (.withLocale
   (DateTimeFormat/forStyle style)
   (current-locale)))
