(ns ring.middleware.format-params
  (:require [cheshire.core :as json]))

(defn get-charset
  "Extracts charset from Content-Type header. utf-8 by default."
  [{:keys [content-type]}]
  (let [default-charset "utf-8"]
    (if content-type
      (or (second (re-find #";\s*charset=([^\s;]+)" content-type)) default-charset)
      default-charset)))

(defn make-type-request-pred
  "Function that returns a predicate fn checking if *Content-Type*
   request header matches a specified regexp and body is set."
  [regexp]
  (fn [{:keys [body] :as req}]
    (when-let [#^String type (get req :content-type
                                (get-in req [:headers "Content-Type"]
                                        (get-in req [:headers "content-type"])))]
      (and body (not (empty? (re-find regexp type)))))))

(defn wrap-format-params
    "Wraps a handler such that requests body are deserialized from to the right format, added in a :body-params key and merged in :params. It takes 3 args:
:predicate is a predicate taking the request as sole argument to test if deserialization should be used.
:decoder specifies a fn taking the body String as sole argument and giving back a hash-map.
:charset can be either a string representing a valid charset or a fn taking the req as argument and returning a valid charset."
  [handler & {:keys [predicate decoder charset]}]
  (fn [req]
    (if (predicate req)
      (let [body (:body req)
            char-enc (if (string? charset) charset (charset req))
            bstr (slurp body :encoding char-enc)
            fmt-params (decoder bstr)
            req* (assoc req
                   :body-params fmt-params
                   :params (merge (:params req) (when (map? fmt-params) fmt-params)))]
        (handler req*))
      (handler req))))

(def json-request?
  (make-type-request-pred #"^application/(vnd.+)?json"))

(defn json-parse-string [s]
  (json/parse-string s true))

(defn wrap-json-params
  "Handles body params in JSON format. See wrap-format-params for details."
  [handler & {:keys [predicate decoder charset]
              :or {predicate json-request?
                   decoder json-parse-string
                   charset get-charset}}]
  (wrap-format-params handler :predicate predicate :decoder decoder :charset charset))
