package com.timezynk;

import java.io.DataOutputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;
import java.text.SimpleDateFormat;
import java.util.Date;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Contains class structure for Pipedrive API integration.
 * Takes Map or JSON as input and returns the results in the same format.
 * Runs the requested API calls and returns relevant information.
 */
public class Pipedrive
{

	/*
	 *  Class variables
	 */

	private static final Log log = LogFactory.getLog("com.timezynk.pipedrive");
	private static final String SWEDISH_A_LOWER = "[åä]";
	private static final String SWEDISH_A_UPPER = "[ÅÄ]";
	private static final String SWEDISH_O_LOWER = "[ö]";
	private static final String SWEDISH_O_UPPER = "[Ö]";
	private static final String SUPPORTED_CHARACTERS = "[^ !\"#$%&'()*\\^\\+\\-\\.,:/\\<=>\\?@_a-zåäöA-ZÅÄÖ0-9{|}~]{1,254}";
	private static final String ID_SUPPORTED_CHARACTERS = "[^a-f0-9]"; // 52e65263e4b0e67464310cc2
	private static final String HASH_VALID_CHARACTERS = "^[a-f0-9]{40}$";
	private static final String PHONE_SUPPORTED_CHARACTERS = "[^0-9\\+\\- ]{1,254}";
	private static final long SIX_WEEKS = 3628800000L;
	private static final SimpleDateFormat ISO_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");

	private String apiKey;
	private String signupLinkKey;
	private ObjectMapper objMapper = new ObjectMapper();

	/*
	 * Constructor
	 */

	public Pipedrive(String api, String dealHash)
	{

		if (!api.matches(HASH_VALID_CHARACTERS))
		{

			log.error("Invalid API Key, must be 40 characters in lenght and contain numbers: 0 to 9 and letters from lowercase: a-z and UPPERCASE: A-Z.");
			throw new InvalidKey(api);

		}
		else if (!dealHash.matches(HASH_VALID_CHARACTERS))
		{

			log.error("Invalid signupHash Key, must be 40 characters in lenght and contain numbers: 0 to 9 and letters from lowercase: a-z and UPPERCASE: A-Z.");
			throw new InvalidKey(api);

		}

		signupLinkKey = dealHash;
		apiKey = api;

	}

	private String getAPIKey()
	{

		return apiKey;

	}

	private String getSignupKey()
	{

		return signupLinkKey;

	}

	/*
	 * Conversion tools
	 */

	// This is a bit dangerous, but it was the only way Eclipse would allow unchecked values to be read into a HashMap. Unchecked is intended as Object is the target type, any object is allowed.
	@SuppressWarnings("unchecked")
	private Map<String, Object> parseToMap(JsonNode jsonNode) throws JsonParseException, JsonMappingException, IOException
	{

		Map<String, Object> map = new HashMap<String, Object>();
		map = objMapper.readValue(jsonNode.toString(), Map.class);
		return map;

	}

	/*
	 * Connection tools
	 */

	private HttpURLConnection prepareHttpCon(String urlString) throws MalformedURLException, IOException
	{

		URL uri = new URL(urlString);
		HttpURLConnection con = (HttpURLConnection) uri.openConnection();
		con.setRequestProperty("Content-Type", "application/json");
		con.setRequestProperty("Accept", "application/json");
		return con;

	}

	/* Needs working with and testing.
    private JsonNode sendPipeDriveApiCallPut(String url, String arguments) throws MalformedURLException, IOException
    {

    	HttpURLConnection con = prepareHttpCon("https://api.pipedrive.com/v1" + url + getAPIKey());
        con.setDoInput(true);
        DataOutputStream out = new DataOutputStream(con.getOutputStream());
        //out.writeBytes(json);
        out.flush();
    	JsonNode httpPutResponseNode = mapper.readTree(con.getInputStream());
        out.close();
        con.disconnect();
        return httpPutResponseNode;

    }*/

	private JsonNode sendPipeDriveApiCallGet(String url, String arguments) throws MalformedURLException, IOException
	{

		HttpURLConnection con = prepareHttpCon("https://api.pipedrive.com/v1" + url + getAPIKey() + arguments);
		con.setDoInput(true);
		JsonNode httpGetResponseNode = objMapper.readTree(con.getInputStream());
		con.disconnect();
		return httpGetResponseNode;

	}

	private JsonNode sendPipeDriveApiCallDelete(String url, String arguments) throws MalformedURLException, IOException
	{

		HttpURLConnection con = prepareHttpCon("https://api.pipedrive.com/v1" + url + arguments + "?api_token=" + getAPIKey());
		con.setDoOutput(true);
		con.setRequestMethod("DELETE");
		JsonNode httpDeleteResponseNode = objMapper.readTree(con.getInputStream());
		con.disconnect();
		return httpDeleteResponseNode;

	}


	private JsonNode sendPipeDriveApiCallPost(String url, String json) throws MalformedURLException, IOException
	{

		HttpURLConnection con = prepareHttpCon("https://api.pipedrive.com/v1" + url + getAPIKey());
		con.setDoOutput(true);
		con.setDoInput(true);
		DataOutputStream out = new DataOutputStream(con.getOutputStream());
		out.writeBytes(json);
		out.flush();
		out.close();
		JsonNode httpPostResponseNode;
		try {
			httpPostResponseNode = objMapper.readTree(con.getInputStream());
		} finally {
			con.disconnect();
		}

		return httpPostResponseNode;

	}

	/*
	 * -- [S] Supported, [L] Limited support.
	 * >  [L] Deals
	 * -> [S] Add a deal
	 * -> [S] Delete a deal
	 */

	private JsonNode addADeal(String title, String personId, String organizationId, String id) throws MalformedURLException, IOException
	{
		Date ecd = new Date(System.currentTimeMillis() + SIX_WEEKS);
		ObjectNode arguments = (ObjectNode)objMapper.createObjectNode();
		arguments.put("title", title + " deal");
		arguments.put("person_id", personId);
		arguments.put("org_id", organizationId);
		arguments.put("value", 1000);
		arguments.put("expected_close_date", ISO_DATE_FORMAT.format(ecd));
		arguments.put(getSignupKey(), "https://pro.timezynk.com/#companies/" + id + "/users");

		JsonNode result = sendPipeDriveApiCallPost("/deals?api_token=", arguments.toString());
		return result;

	}

	public JsonNode deleteADeal(String id) throws MalformedURLException, IOException
	{

		return sendPipeDriveApiCallDelete("/deals/", id);

	}

	/*
	 * -- [S] Supported, [L] Limited support.
	 * >  [L] Organizations
	 * -> [S] Add an organization
	 * -> [S] Delete an organization
	 * -> [S] Find organization by name
	 */

	private JsonNode addAnOrganization(String name) throws MalformedURLException, IOException
	{

		JsonNode arguments = objMapper.createObjectNode();
		((ObjectNode) arguments).put("name", name);

		JsonNode result = sendPipeDriveApiCallPost("/organizations?api_token=", arguments.toString());
		return result;

	}

	public JsonNode deleteAnOrganization(String id) throws MalformedURLException, IOException
	{

		return sendPipeDriveApiCallDelete("/organizations/", id);

	}

	private JsonNode findOrganizationByName(String name) throws MalformedURLException, IOException
	{

		// URLEncoder replaces spaces with + so we need to replace these with %20 after.
		String nameString = "&term=" +  URLEncoder.encode(name, "UTF-8").replaceAll("\\+", "%20") + "&limit=1";

		JsonNode result = sendPipeDriveApiCallGet("/organizations/find?api_token=", nameString);
		return result;

	}

	/*
	 * -- [S] Supported, [L] Limited support.
	 * >  [L] Persons
	 * -> [S] Add a person
	 * -> [S] Delete a person
	 */

	private JsonNode addAPerson(String name, String email, String phone, String org_id) throws MalformedURLException, IOException
	{

		JsonNode arguments = objMapper.createObjectNode();
		((ObjectNode) arguments).put("name", name);
		((ObjectNode) arguments).put("email", email);
		((ObjectNode) arguments).put("phone", phone);
		((ObjectNode) arguments).put("org_id", org_id);

		JsonNode result = sendPipeDriveApiCallPost("/persons?api_token=", arguments.toString());
		return result; // Defaults return negative/false

	}

	public JsonNode deleteAPerson(String id) throws MalformedURLException, IOException
	{

		return sendPipeDriveApiCallDelete("/persons/", id);

	}

	/*
	 * TimeZynk unique structure, could be extracted into its own class and leave the API "clean"
	 */

	public String parseSignup(String companyStringData) throws JsonProcessingException, IOException, PipedriveApiException
	{

		JsonNode companyNodeData = objMapper.readTree(companyStringData);
		JsonNode resultNodeData = parseSignup(companyNodeData);
		return resultNodeData.toString();

	}

	public Map<String, Object> parseSignup(Map<String, Object> companyMapData) throws JsonProcessingException, IOException, IllegalArgumentException, PipedriveApiException
	{

		JsonNode result = parseSignup((JsonNode)objMapper.valueToTree(companyMapData));
		return parseToMap(result);
		// Skapa map <String, Object> och mata parseSignup med den.

	}

	private String clearString(String value) {

		value = value.replaceAll(SWEDISH_A_LOWER, "a"); // replace å and ä with a
		value = value.replaceAll(SWEDISH_A_UPPER, "A"); // replace Å and Ä with A
		value = value.replaceAll(SWEDISH_O_LOWER, "o"); // replace ö with o
		value = value.replaceAll(SWEDISH_O_UPPER, "O"); // replace Ö with O
		value = value.replaceAll(SUPPORTED_CHARACTERS, ""); // removes everything not supported

		return value;

	}

	// Filtering input data is to limit character usage to what is supported by the API, it will not validate data.
	private JsonNode dataCleanser(JsonNode taintedCompanyMapData) {

		JsonNode companyMapData = objMapper.createObjectNode();
		//52e65263e4b0e67464310cc2
		((ObjectNode) companyMapData).put("id", clearString(taintedCompanyMapData.get("id").textValue()).replaceAll(ID_SUPPORTED_CHARACTERS, ""));

		// Filter input name.
		((ObjectNode) companyMapData).put("name", clearString(taintedCompanyMapData.get("name").textValue()));

		// Filter phone invoice-phone
		((ObjectNode) companyMapData).put("invoice-phone", taintedCompanyMapData.get("invoice-phone").textValue().replaceAll(PHONE_SUPPORTED_CHARACTERS, ""));

		// Filter e-mail, e-mail does NOT support special domains with UTF-8 character coding due to pipedrive limitation.
		((ObjectNode) companyMapData).put("invoice-email", taintedCompanyMapData.get("invoice-email").textValue().replaceAll(SUPPORTED_CHARACTERS, ""));

		// Filter contact name.
		((ObjectNode) companyMapData).put("invoice-contact", clearString(taintedCompanyMapData.get("invoice-contact").textValue()));

		return companyMapData;

	}

	public JsonNode parseSignup(JsonNode taintedCompanyMapData)
			throws MalformedURLException, IOException, PipedriveApiException
	{

		JsonNode dealNode, personNode, organizationNode, organizationByName, idNode = objMapper.createObjectNode();



		// Clean all input data
		JsonNode companyMapData = dataCleanser(taintedCompanyMapData);


		try {

			organizationByName = findOrganizationByName(companyMapData.get("name").textValue());

			if (organizationByName.path("success").asBoolean() && (organizationByName.path("data").toString() != "null"))
			{

				((ObjectNode) idNode).put("organizationDuplicate", true);
				throw new DuplicateOrganizationDetected(organizationByName);

			}

			organizationNode = addAnOrganization(companyMapData.get("name").textValue());

			if (!organizationNode.path("success").asBoolean()) {

				throw new OrganizationFailedException(idNode);

			}

			if (organizationByName.path("data").toString() == "null") {

				((ObjectNode) idNode).put("organizationId", organizationNode.path("data").path("id").toString());
				personNode = addAPerson(companyMapData.path("invoice-contact").textValue(), companyMapData.get("invoice-email").textValue(), companyMapData.get("invoice-phone").textValue(), organizationNode.path("data").path("id").toString());

				if (!organizationNode.path("success").asBoolean()) {

					throw new PersonFailedException(idNode);

				}

				((ObjectNode) idNode).put("personId", personNode.path("data").path("id").toString());
				dealNode = addADeal(companyMapData.get("name").textValue(), idNode.path("personId").textValue(), idNode.path("organizationId").textValue(), companyMapData.path("id").textValue());

				if (!dealNode.path("success").asBoolean()) {

					throw new DealFailedException(idNode);

				}

				((ObjectNode) idNode).put("dealId", dealNode.path("data").path("id").toString());

				log.trace("Added new Organization: " + companyMapData.path("name").textValue()  + ". Addede new Person: " + companyMapData.path("invoice-contact").textValue() + "And created new deal with ID: " + dealNode.path("data").path("id").toString());

			} else {

				log.trace("Duplicate organization detected for: " + companyMapData.path("name").textValue());

			}

			((ObjectNode) idNode).put("success", true);

			return idNode;

		} catch (OrganizationFailedException e) {

			log.error("OrganizationFailed : " + e.getMessage());

			throw e;

		} catch (PersonFailedException e) {

			if (!idNode.path("organizationDuplicate").asBoolean())
			{

				deleteAnOrganization(idNode.path("organizationId").textValue());

			}

			log.error("PersonFailed : " + e.getMessage());

			throw e;

		} catch (DealFailedException e) {

			deleteAnOrganization(idNode.path("organizationId").textValue());
			deleteAPerson(idNode.path("personId").textValue());

			log.error("DealFailed : " + e.getMessage());

			throw e;

		} catch (DuplicateOrganizationDetected e) {

			log.error("DuplicateOrganizationDeteceted : " + e.getMessage());
			throw e;

		}
	}

}
