(ns com.timezynk.domus.mongo
  (:require
   [clojure.set :refer [rename-keys]]
   [com.timezynk.mongo :as m]
   [com.timezynk.mongo.hooks :as mh]
   [com.timezynk.useful.date :as date])
  (:import [java.time LocalDate LocalDateTime LocalTime ZonedDateTime]
           [org.bson BsonType]
           [org.bson.codecs Codec]))

(defn zoneddatetime-codec []
  (reify Codec
    (decode [_this reader _decoder-context]
      (-> (.readDatetime reader)
          (date/to-datetime)))

    (encode [_this writer value _encoder-context]
      (->> (date/to-millis value)
           (.writeDatetime writer)))

    (getEncoderClass [_this]
      ZonedDateTime)))

(defn localdatetime-codec []
  (reify Codec
    (encode [_this writer value _encoder-context]
      (->> (date/to-millis value)
           (.writeDatetime writer)))

    (getEncoderClass [_this]
      LocalDateTime)))

(defn localdate-codec []
  (reify Codec
    (encode [_this writer value _encoder-context]
      (->> (date/to-iso value)
           (.writeString writer)))

    (getEncoderClass [_this]
      LocalDate)))

(defn localtime-codec []
  (reify Codec
    (encode [_this writer value _encoder-context]
      (->> (.toString value)
           (.writeString writer)))

    (getEncoderClass [_this]
      LocalTime)))

(defmacro with-codecs [& body]
  `(m/with-codecs [(zoneddatetime-codec)
                   (localdatetime-codec)
                   (localdate-codec)
                   (localtime-codec)]
                  {BsonType/DATE_TIME ZonedDateTime}
     ~@body))

(defn intersecting-query
  ([from to] (intersecting-query from to :start :end))
  ([from to start-key end-key]
   {start-key {:$lt (date/to-datetime to)}
    end-key   {:$gte (date/to-datetime from)}}))

(defn start-inside-period-query
  ([from to] (start-inside-period-query from to :start))
  ([from to start-key]
   {start-key (merge (when from {:$gte (date/to-datetime from)})
                     (when to {:$lt (date/to-datetime to)}))}))

(defmacro scaffold [& body]
  `(mh/with-hooks {:read  #(rename-keys % {:_id :id})
                   :write #(rename-keys % {:id  :_id})}
     ~@body))
