(ns com.timezynk.domus.fixture
  "Commonly used fixtures for `clojure.test`."
  (:require
   [clojure.tools.logging :as log]
   [com.timezynk.mongo :as m]
   [com.timezynk.mongo.util :as mu]))

(defn with-database
  "Attempts to build a fixture which provides the test a database connection.
   `url-var` is the name of the enviroment variable where the connection string
   is expected to be.
   Throws an exception if no environment variable by that name is found."
  [url-var]
  (if-some [url (System/getenv url-var)]
    (fn [f]
      (mu/set-mongo-uri! url)
      (mu/wrap-mongo
       (f)))
    (->> url-var
         (str "No value set for ")
         (IllegalStateException.)
         (throw))))

(defn- build-database-fixture
  "Attempts to build a fixture which provides a database connection.
   Expects to find the connection string in the `url-var` enviroment variable.
   Throws an exception if no environment variable by that name is found."
  [url-var]
  (if-some [url (System/getenv url-var)]
    (fn [f]
      (mu/set-mongo-uri! url)
      (mu/wrap-mongo
       (f)))
    (->> url-var (str "No value set for ") (IllegalStateException.) (throw))))

(def with-db
  "Provides a connection to the MongoDB database specified in `MONGO_URL`."
  (build-database-fixture "MONGO_URL"))

(defn with-drop
  "Drops all database collections before running the test."
  [f]
  (try
    (run! m/drop-collection! (m/list-collection-names))
    (catch Exception e
      (log/warn e "Failed to drop collections")))
  (f))
