(ns com.timezynk.domus.mongo.queue
  (:require [clojure.tools.logging :as log]
            [com.timezynk.domus.mongo.queue.metrics :as metrics]
            [com.timezynk.mongo :as m]
            [com.timezynk.mongo.util :as mu]
            [com.timezynk.useful.env :as env]
            [com.timezynk.useful.string :as s]
            [com.timezynk.useful.time :as t]
            [com.timezynk.useful.timed-queue :refer [TimedQueue]]))

(def ^:const DEFAULT_NUM_WORKERS 1)

(def ^:const DEFAULT_MIN_INTERVAL 250)

(def ^:const DEFAULT_MIN_SLEEP 100)

(def ^:const DEFAULT_MAX_ATTEMPTS 5)

(def ^:const DEFAULT_RETRY_INTERVAL_MS
  "Milliseconds to wait for before retrying a failed job."
  500)

(def ^:const MAX_RETRY_INTERVAL_MS (* 1000 10 10))

(def ^:private ^:const JOIN_TIMEOUT
  "Maximum number of milliseconds to wait for worker threads to finish."
  20000)

(def DEFAULT_THREAD_PRIORITY
  "Priority to assign to worker threads, if none specified at creation."
  (if env/test? Thread/MAX_PRIORITY Thread/MIN_PRIORITY))

(defprotocol WorkQueue
  (start-workers! [this thread-name] "Starts all worker threads.")
  (stop-workers! [this] "Stops all worker threads.")
  (length [this] "Number of currently enqueued jobs."))

(defn- reserve-job!
  "Makes the head of the queue inaccessible to other workers and returns it."
  [queue]
  (mu/wrap-mongo
   (m/fetch-and-set-one! (.collection queue)
                         {:run-at {:$lte (System/currentTimeMillis)}
                          :locked nil}
                         {:locked true}
                         :sort {:run-at 1})))

(defn failed-queue-collection-name
  "Name of the collection which holds the failed jobs."
  [queue]
  (-> queue (.collection) (name) (str ".failed") (keyword)))

(defn- on-error
  "Handles failure to run the handler of `queue` on `job`."
  [queue job error]
  (let [num-failures (-> job :errors count inc)
        interval (t/exponential-backoff-interval num-failures
                                                 (.retry-interval queue)
                                                 MAX_RETRY_INTERVAL_MS)
        terminal? (>= num-failures (.max-attempts queue))]
    (when terminal?
      (log/error error "Error while handling" (:type job) "job"))
    (mu/wrap-mongo
     (m/insert!
      (if terminal?
        (failed-queue-collection-name queue)
        (.collection queue))
      (-> job
          (dissoc :_id)
          (update :errors conj {:stacktrace (s/from-throwable error)
                                :thrown-at (System/currentTimeMillis)})
          (assoc :run-at (+ (System/currentTimeMillis) interval)))))))

(defn- run-job! [queue job]
  (when job
    (try
      ((.handler queue) job)
      (catch Exception e
        (on-error queue job e)))))

(defn- process-job! [queue job]
  (->> job (run-job! queue) (metrics/report queue job)))

(defn- delete-job! [queue job]
  (when job
    (mu/wrap-mongo
     (m/delete-one! (.collection queue)
                    {:_id (:_id job)}))))

(defn- add-payload-prefix [acc [k v]]
  (assoc acc (str "payload." (name k)) v))

(defmacro ^:private throttle
  "Delegates to `com.timezynk.useful.time/sleep-pad`, unless in testing mode.
   Executes `body` without delay, otherwise."
  [min-duration min-sleep & body]
  `(if-not env/test?
     (t/sleep-pad ~min-duration ~min-sleep ~@body)
     (do ~@body)))

(defn- processing-loop
  "Processes `queue` until the termination condition is met."
  [queue]
  (let [collection (.collection queue)
        go-on? (.go-on? queue)
        min-interval (.min-interval queue)
        min-sleep (.min-sleep queue)]
    (log/info "Starting thread" (-> (Thread/currentThread) (.getName))
              "for queue" collection)
    (while @go-on?
      (try
        (->> (reserve-job! queue)
             (process-job! queue)
             (delete-job! queue)
             (throttle min-interval min-sleep))
        (catch Exception e
          (log/error e "Exception in queue" collection))))
    (log/info "Queue" collection "finished")))

(defn- register-thread [queue thread]
  (-> queue
      (.threads)
      (swap! conj thread)))

(defn- acquire-threads [queue thread-name]
  (let [num-workers (.num-workers queue)
        thread-priority (.thread-priority queue)]
    (dotimes [i num-workers]
      (register-thread queue
                       (doto (Thread. (bound-fn [] (processing-loop queue))
                                      (cond-> thread-name
                                        (not= 1 num-workers) (str "-" i)))
                         (.setDaemon false)
                         (.setPriority thread-priority)
                         (.start))))
    (register-thread queue
                     (metrics/thread queue))))

(defn- release-threads [queue]
  (let [threads (.threads queue)]
    (when (seq @threads)
      (run! #(.join % JOIN_TIMEOUT) @threads)
      (reset! threads []))))

(deftype MongoQueue
         [collection handler num-workers thread-priority min-interval
          min-sleep go-on? threads write-concern
          metrics-id metrics
          max-attempts retry-interval]

  WorkQueue
  (start-workers! [this thread-name]
    (metrics/acquire this)
    (acquire-threads this thread-name)
    (reset! go-on? true))

  (stop-workers! [this]
    (reset! go-on? false)
    (release-threads this)
    (metrics/release this))

  (length [this]
    (metrics/length this))

  TimedQueue
  (push-job! [_this type run-at payload]
    (mu/wrap-mongo
     (m/insert! collection
                {:run-at run-at
                 :type type
                 :payload payload
                 :errors []})))

  (upsert-job! [_this type run-at selector update]
    (mu/wrap-mongo
     (m/update! collection
                (merge
                 {:type type}
                 (reduce add-payload-prefix {} selector))
                (merge
                 {:$set
                  {:run-at run-at
                   :type type
                   :errors []}}
                 update)
                :upsert true))))

(defn create [& {:as params}]
  (let [{:keys [id collection handler num-workers thread-priority
                min-interval min-sleep max-attempts retry-interval]} params]
    (MongoQueue. collection
                 handler
                 (or num-workers DEFAULT_NUM_WORKERS)
                 (or thread-priority DEFAULT_THREAD_PRIORITY)
                 (or min-interval DEFAULT_MIN_INTERVAL)
                 (or min-sleep DEFAULT_MIN_SLEEP)
                 (atom true)
                 (atom [])
                 (if env/test? :acknowledged :unacknowledged)
                 id
                 (atom nil)
                 (or max-attempts DEFAULT_MAX_ATTEMPTS)
                 (or retry-interval DEFAULT_RETRY_INTERVAL_MS))))

(defn drain
  "Puts the calling thread to sleep until the length of `queue` becomes zero."
  [queue]
  (t/wait-till (-> queue length zero?)))
