(ns com.timezynk.domus.mongo
  (:require
   [clojure.set :refer [rename-keys]]
   [com.timezynk.mongo.convert-types :refer [ConvertTypes]]
   [com.timezynk.mongo.hooks :as mh]
   [com.timezynk.useful.date :as date]
   [com.timezynk.useful.mongo :refer [object-id unpack-id unpack-ids]]
   [somnium.congomongo :as mongo]
   [somnium.congomongo.coerce :as mongo-coerce]))

(extend-protocol mongo-coerce/ConvertibleToMongo
  java.time.LocalDate
  (clojure->mongo
    ; Make congomongo convert java.time.LocalDate to string.
    ^String [^java.time.LocalDate d]
    (date/to-iso d))

  java.time.LocalTime
  (clojure->mongo
    ; Make congomongo convert java.time.LocalTime to string.
    ^String [^java.time.LocalTime t]
    (.toString t))

  java.time.LocalDateTime
  (clojure->mongo
    ; Make congomongo convert java.time.LocalDateTime to storable date.
    ^java.util.Date [^java.time.LocalDateTime dt]
    (date/to-utildate dt))

  java.time.ZonedDateTime
  (clojure->mongo
    ; Make congomongo convert java.time.ZonedDateTime to storable date.
    ^java.util.Date [^java.time.ZonedDateTime zdt]
    (date/to-utildate zdt))

  org.joda.time.LocalDateTime
  (clojure->mongo [o] (str o))

  org.joda.time.LocalTime
  (clojure->mongo [o] (str o))

  org.joda.time.LocalDate
  (clojure->mongo [o] (str o)))

(extend-protocol mongo-coerce/ConvertibleFromMongo
  java.util.Date
  (mongo->clojure [o _keywordize]
    (date/to-datetime o)))

;; New MongoDB API can handle date-times
(extend-protocol ConvertTypes
  org.joda.time.DateTime
  (clj->doc ^String [v] (.toString v))

  org.joda.time.LocalDate
  (clj->doc ^String [v] (.toString v))

  org.joda.time.LocalDateTime
  (clj->doc ^String [v] (.toString v))

  org.joda.time.LocalTime
  (clj->doc ^String [v] (.toString v))

  java.time.LocalDate
  (clj->doc ^String [v] (.toString v))

  java.time.LocalTime
  (clj->doc ^String [v] (.toString v))

  java.time.LocalDateTime
  (clj->doc ^java.util.Date [v] (date/to-utildate v))

  java.time.ZonedDateTime
  (clj->doc ^java.util.Date [v] (date/to-utildate v))

  java.util.Date
  (doc->clj ^java.time.ZonedDateTime [v] (date/to-datetime v)))

(defn fetch [& args]
  (unpack-ids (apply mongo/fetch args)))

(defn insert! [collection entry]
  (unpack-id (mongo/insert! collection entry)))

(defn insert-and-wait! [collection entry]
  (unpack-id
   (mongo/insert! collection entry :write-concern :acknowledged)))

(defn find-by-id [collection id]
  (try
    (unpack-id (mongo/fetch-by-id collection (object-id id)))
    (catch IllegalArgumentException _e
      nil)))

(defn intersecting-query
  ([from to] (intersecting-query from to :start :end))
  ([from to start-key end-key]
   {start-key {:$lt (date/to-datetime to)}
    end-key   {:$gte (date/to-datetime from)}}))

(defn start-inside-period-query
  ([from to] (start-inside-period-query from to :start))
  ([from to start-key]
   {start-key
    (merge
     (when from {:$gte (date/to-datetime from)})
     (when to {:$lt (date/to-datetime to)}))}))

(defmacro scaffold [& body]
  (mh/with-hooks {:read  #(rename-keys % {:_id :id})
                  :write #(rename-keys % {:id  :_id})}
    ~@body))
