(ns com.timezynk.domain.mongo.update
  "Abstraction layer over various update strategies."
  (:require [flatland.useful.fn :as ufn]
            [com.timezynk.domain.mongo.channel :as mchan]
            [com.timezynk.domus.mongo.db :refer [db]]
            [com.timezynk.useful.mongo.queries :as uq]
            [com.timezynk.useful.time :as ut]
            [com.timezynk.useful.rest.current-user :as current-session]
            [somnium.congomongo :as mongo]
            [com.timezynk.domain.mongo.id :as id])
  (:import [org.bson.types ObjectId]))

(defprotocol Strategy
  (acquire
    [this]
    "Lock documents so concurrent updates do not collide.")
  (query
    [this]
    "MongoDB query to fetch affected documents.")
  (fetch
    [this]
    "Fetches affected documents.")
  (patcher
    [this]
    "Builds a function which applies the patch without persisting the result.
     The function has a single parameter which is expected to be a document.")
  (update!
    [this oldies newlings]
    "Persists patched documents.")
  (release
    [this]
    "Unlocks documents so other threads can update them."))

(defn- groom
  "Prepares patch for the update."
  [patch now]
  (-> patch
      (id/->mongo)
      (dissoc :_id :_pid :_name)
      (assoc :valid-from now :valid-to nil)))

(defrecord Fast [cname restriction patch now]
  Strategy
  (acquire [_] nil)

  (query [_]
    (assoc restriction :valid-to nil))

  (fetch [this]
    (mongo/fetch cname :where (query this)))

  (patcher [_]
    (fn [doc]
      (merge doc (groom patch now))))

  (update! [this _ _]
    (mongo/update! cname
                   (-> patch
                       (dissoc :changed-by)
                       (uq/not-matching)
                       (merge (query this)))
                   {:$set (groom patch now)}
                   :multiple true
                   :upsert false))

  (release [_] nil))

(defrecord Revisioned [cname restriction patch now lock]
  Strategy
  (acquire [_]
    (mongo/update! cname
                   (assoc restriction :valid-to nil)
                   {:$set lock}
                   :multiple true
                   :upsert false))

  (query [_]
    (merge restriction lock))

  (fetch [this]
    (->> this
         (query)
         (mongo/fetch cname :where)
         (map #(dissoc % :lock-id))))

  (patcher [_]
    (fn [doc]
      (-> doc
          (dissoc :_id :lock-id)
          (merge (groom patch now))
          (assoc :_id (ObjectId.) :_pid (:_id doc)))))

  (update! [_ oldies newlings]
    (when (seq newlings)
      (mongo/insert! cname
                     newlings
                     :many true))
    (when (seq oldies)
      (mongo/insert! :domainlog
                     {:type :update
                      :company-id (-> oldies first :company-id)
                      :collection (name cname)
                      :oldies oldies
                      :tstamp now
                      :user-id (or (current-session/user-id)
                                   (:changed-by patch))}))
    (mongo/destroy! cname
                    {:_id {:$in (mapv :_id oldies)}}))

  (release [_]
    (mongo/update! cname
                   (merge restriction lock)
                   {:$set {:valid-to nil}
                    :$unset {:lock-id ""}}
                   :multiple true
                   :upsert false)))

(defn fast
  "`Fast` constructor."
  [cname restriction patch]
  (->Fast cname
          (id/->->mongo restriction)
          patch
          (ut/current-time-ms)))

(defn revisioned
  "`Revisioned` constructor."
  [cname restriction patch]
  (let [now (ut/current-time-ms)
        lock {:valid-to now
              :lock-id (rand-int 1000000000)}]
    (->Revisioned cname
                  (id/->->mongo restriction)
                  patch
                  now
                  lock)))

(defn- relevant
  "Strips fields which are not part of the problem domain."
  [doc]
  (dissoc doc :_id :_name :_pid :lock-id
    :created :valid-from :valid-to
    :created-by :changed-by))

(defn- dirty? [pair]
  (->> pair (map relevant) (apply not=)))

(defn apply!
  "Persists the update and runs all side-effects."
  [strategy]
  (mongo/with-mongo @db
    (acquire strategy)
    (try
      (let [unpatched (fetch strategy)
            patched (map (patcher strategy) unpatched)
            pairs (map vector unpatched patched)
            [dirty-unpatched dirty-patched] (->> pairs
                                                 (filter dirty?)
                                                 ((juxt
                                                    (partial map first)
                                                    (partial map second))))]
        (when (seq dirty-unpatched)
          (update! strategy dirty-unpatched dirty-patched)
          (mchan/put! :update
                      (:cname strategy)
                      :new (map id/->clj dirty-patched)
                      :old (map id/->clj dirty-unpatched)))
        (map (comp id/->clj
                   (ufn/to-fix dirty? second first))
             pairs))
      (finally
        (release strategy)))))
