(ns com.timezynk.cancancan.condition.log
  "Visual aid for debugging authorization predicates."
  (:require [clojure.pprint :as pp]
            [clojure.string :as string]
            [clojure.tools.logging :as log]))

(def ^:private ^:dynamic *on?* true)

(defmacro with-silence
  "Executes `body` without logging. The effect is thread-local."
  [& body]
  `(binding [*on?* false]
     ~@body))

(def ^:private ^:const PREDICATE_INDENT "--.-> P ")
(def ^:private ^:const ACTION_INDENT    "  |-> A ")
(def ^:private ^:const OBJECT_INDENT    "  |-> O ")
(def ^:private ^:const INSTANCE_INDENT  "  |-> I ")
(def ^:private ^:const USER_INDENT      "  |-> U ")
(def ^:private ^:const RESULT_INDENT    "  `-> R ")
(def ^:private ^:const PLAIN_INDENT     "  |     ")

(defn- clojurize [x]
  (-> x
      (string/replace "_QMARK_" "?")
      (string/replace \_ \-)))

(defn- source-code-name
  "Attempts to recreate the name of function `f` as it is in source code."
  [f]
  (->> f
       (str)
       (clojurize)
       (re-find #"(.*?)\$(.*?)[$@]")
       (drop 1)
       (apply format "%s/%s")))

(defn- pretty-print-indented [x]
  (->> x
       pp/pprint
       with-out-str
       (string/trimr)
       (map #(cond-> %
               (= \newline %) (str PLAIN_INDENT)))
       (apply str)))

(defn wrap
  "Wraps authorization predicate `f` in a function which logs input and output."
  [f user]
  (fn [action object instance]
      (let [result (f action object instance)]
        (when *on?*
          (log/debug (str "authorization predicate\n"
                          PREDICATE_INDENT (source-code-name f) "\n"
                          ACTION_INDENT (or action "nil") "\n"
                          OBJECT_INDENT (or object "nil") "\n"
                          INSTANCE_INDENT (pretty-print-indented instance) "\n"
                          USER_INDENT (pretty-print-indented user) "\n"
                          RESULT_INDENT result)))
        result)))
