(ns com.timezynk.cancancan.restriction
  "Derive database query restrictions from abilities."
  (:refer-clojure :exclude [compile]))

(defn _&
  [x & xs]
  {:all (remove nil? (cons x xs))})

(defn _|
  [x & xs]
  {:any (remove nil? (cons x xs))})

(defn _!
  [x]
  {:neg x})

(declare compile)

(defn- compose
  "First, compiles each of `restrictions` with `user`, `roles1` and `modules`.
   Then, composes the result into a query hash using the boolean `operator`."
  [restrictions operator user roles1 modules]
  (fn [action object instance]
    (->> restrictions
         (map #(-> %
                   (compile user roles1 modules)
                   (apply [action object instance])))
         (reduce conj [])
         (hash-map operator))))

(defn- conjunct [restrictions user roles modules]
  (compose restrictions :$and user roles modules))

(defn- disjunct [restrictions user roles modules]
  (compose restrictions :$or user roles modules))

(defn- negate [restriction user roles1 modules]
  (fn [action object instance]
    {:$not (-> restriction
               (compile user roles1 modules)
               (apply [action object instance]))}))

(defn compile
  "Curries restriction functions with `user`, `roles1` and `modules` to produce
   functions which work on `[action object instance]` arguments."
  [restriction user roles1 modules]
  (let [restriction (or restriction {:all []})]
    (if (map? restriction)
      (let [{:keys [all any neg]} restriction]
        (cond all (conjunct all user roles1 modules)
              any (disjunct any user roles1 modules)
              neg (negate neg user roles1 modules)))
      (restriction user roles1 modules))))

(defmulti make
  "Converts an ability (i.e. action / object / scope triplet) to a restriction."
  ^:private
  vector)

(defmethod make :default
  [action object scope]
  (let [method-map (methods make)
        oox (method-map [action object :_])
        oxo (method-map [action :_ scope])
        oxx (method-map [action :_ :_])
        xoo (method-map [:_ object scope])
        xox (method-map [:_ object :_])
        xxo (method-map [:_ :_ scope])
        f (or oox oxo oxx xoo xox xxo)]
    (when f
      (f action object scope))))

(defn- wrapv
  "Wraps `x` into a vector, unless it is already a vector."
  [x]
  (cond-> x
    (not (vector? x)) vector))

(defmacro defrestriction
  "Expands each of its arguments into an implementation of `make`.
   Uses the first three parameters as dispatch values.
   Uses the fourth as return value."
  [& xs]
  (let [definitions (->> xs
                         (mapcat (fn [[a o s c]]
                                   (for [a (wrapv a)
                                         s (wrapv s)]
                                     [a o s c])))
                         (map (fn [[a o s c]]
                                `(defmethod make [~a ~o ~s]
                                   [_# _# _#]
                                   ~c))))]
    `(do ~@definitions)))

(defn from-ability [action object scope]
  (make action object scope))
