(ns com.timezynk.cancancan.condition
  (:refer-clojure :exclude [compile])
  (:require [com.timezynk.cancancan.condition.log :as log]
            [com.timezynk.cancancan.condition.common :as pred]))

(defn _&
  [x & xs]
  {:all (remove nil? (cons x xs))})

(defn _|
  [x & xs]
  {:any (remove nil? (cons x xs))})

(defn _!
  [x]
  {:neg x})

(declare compile)

(defn- conjunct [conditions user roles modules]
  (fn [action object instance]
    (reduce (fn [acc x]
              (if acc
                (let [pred (compile x user roles modules)]
                  (and pred (pred action object instance)))
                false))
            true
            conditions)))

(defn- disjunct [conditions user roles modules]
  (fn [action object instance]
    (reduce (fn [acc x]
              (if acc
                true
                (let [pred (compile x user roles modules)]
                  (and pred (pred action object instance)))))
            false
            conditions)))

(defn compile [conditions user roles modules]
  (let [conditions (or conditions {:all []})]
    (if (map? conditions)
      (let [{:keys [all any neg]} conditions]
        (cond all (conjunct all user roles modules)
              any (disjunct any user roles modules)
              neg (complement (compile neg user roles modules))))
      (log/wrap (conditions user roles modules) user))))

(defmulti make
  "Converts an ability (i.e. action / object / scope triplet) to a condition."
  ^:private
  vector)

(defmethod make :default
  [action object scope]
  (let [method-map (methods make)
        oox (method-map [action object :_])
        oxo (method-map [action :_ scope])
        oxx (method-map [action :_ :_])
        xoo (method-map [:_ object scope])
        xox (method-map [:_ object :_])
        xxo (method-map [:_ :_ scope])
        f (or oox oxo oxx xoo xox xxo)]
    (when f
      (f action object scope))))

(defn- wrapv
  "Wraps `x` into a vector, unless it is already a vector."
  [x]
  (cond-> x
    (not (vector? x)) vector))

(defmacro defcondition
  "Expands each of its arguments into an implementation of `make`.
   Uses the first three parameters as dispatch values.
   Uses the fourth as return value."
  [& xs]
  (let [definitions (->> xs
                         (mapcat (fn [[a o s c]]
                                   (for [a (wrapv a)
                                         s (wrapv s)]
                                     [a o s c])))
                         (map (fn [[a o s c]]
                                `(defmethod make [~a ~o ~s]
                                   [_# _# _#]
                                   ~c))))]
    `(do ~@definitions)))

(defcondition
  [:_ :_ :none pred/fail])

(defn from-ability [action object scope]
  (_&
    (make action object scope)
    (when (and (not= :all scope)
               (not= :company object)
               (not= :conversation object))
      pred/same-company-id)))
