(ns com.timezynk.bus.core
  (:require [clojure.tools.logging :as log]
            [somnium.congomongo :as mongo]
            [com.timezynk.bus.group :as group]
            [com.timezynk.bus.logging :as buslog]
            [com.timezynk.bus.subscriber :as subscriber]
            [com.timezynk.bus.subscriber.hook :as hook]
            [com.timezynk.bus.task :as task]
            [com.timezynk.useful.env :as env]
            [com.timezynk.useful.mongo.queue :as mq]
            [com.timezynk.useful.timed-queue :as tq])
  (:import [java.util.concurrent LinkedBlockingQueue PriorityBlockingQueue
            BlockingQueue TimeUnit]
           [java.util UUID]))

(def ^:private ^:const DEFAULT_NUM_REQUEST_RESPONSE_WORKERS 2)

(def ^:private ^:const DEFAULT_NUM_BROADCAST_WORKERS 2)

(def ^:private ^:const DEFAULT_NUM_PERSISTED_WORKERS 2)

(def ^:private ^:const BROADCAST_THROTTLE_INTERVAL_MS 50)

(defprotocol Bus
  "Abstraction for media which transfer messages to subscribers.
   Subscription is done by declaring interest in a topic.
   Topics get published on the bus, which routes them to eligible subscribers."
  (initialize [this] [this num-workers] [this num-workers params]
    "Acquires all associated resources.")
  (publish [this context topic collection messages]
    "Puts `messages` on the bus.")
  (subscribe [this topic collection subscriber]
    "Registers `subscriber` for `topic` on `collection`.")
  (eligible-for [bus topic collection]
    "Sequence of subscribers eligible for `topic` on `collection`.")
  (unsubscribe-all [this]
    "Unregisters all subscribers.")
  (destroy [this]
    "Releases all associated resources.")
  (length [this]
    "Number of enqueued messages."))

(defn- publish* ^BlockingQueue
  [bus context topic collection messages]
  (when (seq messages)
    (let [reply-channel (LinkedBlockingQueue.)]
      (doseq [m messages
              s (eligible-for bus topic collection)]
        (try
          (.put (:queue bus)
                (subscriber/publish s
                                    topic
                                    collection
                                    m
                                    (or task/*reply-channel* reply-channel)
                                    context))
          (catch Exception e
            (log/warn e topic collection "failed to publish" m))))
      (.put reply-channel [:queued-message-tasks])
      reply-channel)))

(defn wait-for
  "Awaits responses on `reply-channel`.
   Returns `false` on timeout.
   Throws the caught exception on error.
   Returns `true` otherwise."
  [timeout-ms reply-channel]
  (when reply-channel
    (let [poll #(.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)]
      (loop [[event id payload] (poll)
             tasks #{}]
        (buslog/handle-event event tasks id)
        (case event
          :queued-message-tasks (if (empty? tasks)
                                  true
                                  (recur (poll) tasks))

          :queued               (recur (poll) (conj tasks id))

          :started              (recur (poll) tasks)

          :finished             (let [new-tasks (disj tasks id)]
                                  (if (empty? new-tasks)
                                    true
                                    (recur (poll) new-tasks)))

          :exception            (throw payload)

          (if (seq tasks)
            false
            (recur (poll) tasks)))))))

(defn- wrapv
  "Turns `x` into `[x]`, unless it is sequential."
  [x]
  (cond-> x
    (not (sequential? x)) (vector)))

(defn- subscribe* [bus group topic collection hook]
  (when (and topic hook)
    (let [factory (if (= group/REQUEST_RESPONSE group)
                    subscriber/->RequestResponse
                    subscriber/->Broadcast)]
      (log/debug topic collection "new" (name group) "subscriber")
      (doseq [t (wrapv topic)
              c (wrapv collection)]
        (dosync
          (alter (:subscribers bus)
                 update-in
                 [t c]
                 conj
                 (factory c hook)))))))

(defn- eligible-for* [bus topic collection]
  (let [on-topic (-> bus (:subscribers) (deref) (get topic))]
    (concat (get on-topic nil)
            (when collection
              (get on-topic collection)))))

(defn- unsubscribe-all* [bus]
  (dosync
    (ref-set (:subscribers bus) {})))

(defn- destroy* [bus]
  (let [threads (:workers bus)]
    (when (seq @threads)
      (run! #(.interrupt %) @threads)
      (reset! threads []))))

(defn- length* [bus]
  (-> bus :queue .size))

(defn- route-message [message]
  (when-let [t (:task message)]
    (task/process t)))

(defn- broker-loop
  "Builds an infinitely looping function which:
    * takes messages out of `bus`
    * processes them
    * records state of `bus` in Prometheus
   Will block if `bus` is empty.
   Will wait for `throttle-interval`, if set."
  [bus & [throttle-interval]]
  (fn []
    (log/info "starting message broker")
    (let [go-on? (atom true)]
      (while @go-on?
        (try
          (let [^BlockingQueue queue (:queue bus)
                message (.take queue)
                result (route-message message)]
            (buslog/on-message-processed bus message result))
          (when (and throttle-interval (not env/test?))
            (Thread/sleep throttle-interval))
          (catch InterruptedException _
            (log/info "Bus destroyed, shutting down")
            (reset! go-on? false))
          (catch Exception e
            (log/warn e "Exception in bus broker")
            (Thread/sleep 100)))))))

(defrecord RequestResponse [^BlockingQueue queue queue-id workers subscribers]
  Bus

  (initialize [this]
    (initialize this DEFAULT_NUM_REQUEST_RESPONSE_WORKERS))
  (initialize [this num-workers]
    (initialize this num-workers {}))
  (initialize [this num-workers _]
    (dotimes [i num-workers]
      (swap! workers
             conj
             (doto (Thread. (broker-loop this) (str "mchan-rr-" i))
                     (.setDaemon true)
                     (.start))))
    this)
  
  (publish [this context topic collection messages]
    (publish* this context topic collection messages))

  (subscribe [this topic collection subscriber]
    (subscribe* this group/REQUEST_RESPONSE topic collection subscriber))

  (eligible-for [this topic collection]
    (eligible-for* this topic collection))

  (unsubscribe-all [this]
    (unsubscribe-all* this))

  (destroy [this]
    (destroy* this))

  (length [this]
    (length* this)))

(defrecord Broadcast [^BlockingQueue queue queue-id workers subscribers]
  Bus

  (initialize [this]
    (initialize this DEFAULT_NUM_BROADCAST_WORKERS))
  (initialize [this num-workers]
    (initialize this num-workers {}))
  (initialize [this num-workers _]
    (dotimes [i num-workers]
      (swap! workers
             conj
             (doto (Thread. (broker-loop this BROADCAST_THROTTLE_INTERVAL_MS)
                            (str "mchan-bc-" i))
               (.setDaemon true)
               (.setPriority Thread/MIN_PRIORITY)
               (.start))))
    this)

  (publish [this context topic collection messages]
    (publish* this context topic collection messages))

  (subscribe [this topic collection subscriber]
    (subscribe* this group/BROADCAST topic collection subscriber))

  (eligible-for [this topic collection]
    (eligible-for* this topic collection))

  (unsubscribe-all [this]
    (unsubscribe-all* this))

  (destroy [this]
    (destroy* this))

  (length [this]
    (length* this)))

(defn- handler-for
  "Builds a `com.timezynk.useful.mongo.queue` handler which sends the message
   in `job` to the matching subscriber."
  [bus]
  (fn [job]
    (let [{:keys [topic collection hook-id message]} (:payload job)
          subscriber (->> (eligible-for bus (keyword topic) (keyword collection))
                          (filter (comp #{hook-id} hook/id))
                          (first))]
      (if subscriber
        (hook/call subscriber topic collection nil message)
        (log/warn "No subscriber registered for" topic collection)))))

(defrecord Persisted [queue subscribers]
  Bus

  (initialize [this]
    (initialize this DEFAULT_NUM_PERSISTED_WORKERS))
  (initialize [this num-workers]
    (initialize this num-workers {}))
  (initialize [this num-workers params]
    (let [mongo-queue (mq/create (merge {:id (:queue-id params)
                                         :collection (:queue-collection params)
                                         :handler (handler-for this)
                                         :num-workers num-workers}
                                        (when env/test?
                                          {:min-interval 0
                                           :min-sleep 0})))]
      (mq/start-workers! mongo-queue "mchan-db")
      (assoc this :queue mongo-queue)))

  (publish [this _ topic collection messages]
    (when (seq messages)
      (doseq [m messages
              h (eligible-for this topic collection)]
        (tq/push-job! queue
                      (hook/id h)
                      (System/currentTimeMillis)
                      {:hook-id (hook/id h)
                       :topic topic
                       :collection collection
                       :message m})))
    this)

  (subscribe [this topic collection hook]
    (when (and topic hook)
      (log/debug topic collection "new persisted subscriber")
      (doseq [t (wrapv topic)
              c (wrapv collection)]
        (dosync
          (alter subscribers update-in [t c] conj hook))))
    this)

  (eligible-for [this topic collection]
    (eligible-for* this topic collection))

  (unsubscribe-all [this]
    (unsubscribe-all* this)
    this)

  (destroy [_]
    (mq/stop-workers! queue))

  (length [_]
    (-> queue .collection mongo/fetch-count)))

(defn create [group]
  (condp = group
    group/REQUEST_RESPONSE (->RequestResponse
                             (PriorityBlockingQueue.)
                             (str (UUID/randomUUID))
                             (atom [])
                             (ref {}))
    group/BROADCAST (->Broadcast
                      (PriorityBlockingQueue.)
                      (str (UUID/randomUUID))
                      (atom [])
                      (ref {}))
    group/PERSISTED (->Persisted nil (ref {}))))
