(ns andy.rules
  (:require
    [clojure.string :as str]

    [clojure.math :as math]
    #?(:clj [clojure.instant :as instant])
    #?(:clj  [clojure.test :refer :all]
       :cljs [cljs.test :refer-macros [deftest is are]])
    #?(:cljs [tiltontec.util.base :refer-macros [trx prog1]]
       :clj  [tiltontec.util.base :refer :all])

    [tiltontec.util.core :refer [rmap-setf type-of err]]

    #?(:clj  [tiltontec.cell.base :refer :all :as cty]
       :cljs [tiltontec.cell.base
              :refer-macros [without-c-dependency]
              :refer [c-optimized-away? c-formula? c-value c-optimize cache-value
                      c-unbound? c-input? ia-type? cells-init
                      c-model mdead? c-valid? c-useds c-ref? md-ref?
                      c-state +pulse+ c-pulse-observed
                      *call-stack* *defer-changes* unbound
                      c-rule c-me c-value-state c-callers caller-ensure
                      *causation*
                      c-synaptic? caller-drop
                      c-pulse c-pulse-last-changed c-ephemeral? c-slot
                      *depender*
                      *c-prop-depth* md-slot-owning? c-lazy] :as cty])

    #?(:cljs [tiltontec.cell.integrity
              :refer-macros [with-integrity]]
       :clj  [tiltontec.cell.integrity :refer [with-integrity]])

    [tiltontec.cell.evaluate :refer [c-get]]

    #?(:clj  [tiltontec.cell.observer :refer [defobserver fn-obs]]
       :cljs [tiltontec.cell.observer :refer-macros [defobserver fn-obs]])

    #?(:clj  [tiltontec.cell.synapse :refer :all]
       :cljs [tiltontec.cell.synapse :refer-macros [with-synapse]])

    #?(:cljs [tiltontec.cell.core
              :refer-macros [cF cF+ c_F cF_
                             cf-freeze with-c-accumulating with-c-conj with-c-associating with-c-latest]
              :refer [cI c-reset! make-c-formula dag-dump]]
       :clj  [tiltontec.cell.core :refer :all])

    #?(:clj  [tiltontec.model.core :refer :all :as md]
       :cljs [tiltontec.model.core
              :refer-macros [cFkids the-kids mdv!]
              :refer [md-get md-name fget fm! make md-reset! md-getx]
              :as md])

    #?(:clj [taoensso.nippy :as nippy])

    [#?(:cljs cljs.pprint :clj clojure.pprint) :refer [pprint cl-format]]

    #?(:clj [andy.util :refer :all])

    ))

;; --- domain rules -------------------------------------

(defn resume-play-event? [{:keys [:collection/author :collection/name
                                  :collection/event-type :collection/event-time] :as evt}]
  (or (some #{name} ["end-stoppage" "referee-ball-drop" "substitution" "fifty-fifty"])
    (some #{(first event-type)} ["goal-kick" "free-kick" "corner" "throw-in" "kick-off" "save"])))

(defn possession-established? [me]
  ;; todo combine with player-possession variant
  (let [events (mget me :events)]
    (and (> (count events) 2)
      (> (or (poss-duration me) 0.0) 5.0))))

(defn player-poss-established? [me]
  (or (> (count (mget me :events)) 1)
    (> (player-poss-duration me) 3.0)))

(defn possession-team
  "Returns [evt-millis team-id & player-id] if the event signifies possession,
   else nil if event is possession agnostic."
  [{:keys [:collection/author
           :collection/team-id
           :collection/player-id
           :collection/name
           :collection/type
           :collection/event-type
           :collection/event-time]
    :as   evt}]
  (let [;; evt-ms (event-millis evt)
        team-only [evt team-id]
        team-player [evt team-id player-id]
        team-none [evt :none]
        event-type (first event-type)]
    ;;(prn :pt-sees name event-type (str/join "." (rest player)))
    (case name
      "half-start" team-only

      "ball-recovery"
      ;; observed ball-recovery was a brief interruption of a possession, leading to
      ;; an immediate 'out' by the same team and throw-in by the original possessor, so
      ;; we effectively ignore
      nil

      ("reception" "dribble") team-player

      "out"
      nil
      ;; [when :none]

      ("starting-xi" "tackle") nil

      "goal-keeper" (case event-type
                      ("collected" "keeper-sweeper") team-player
                      (do
                        ;;(prn :ignoring-gk name event-type)
                        nil))
      "clearance"
      ;; none lets us actively end any extant possession
      team-none

      "pass" (case event-type
               ("kick-off" "first-time" "free-kick" "throw-in" "corner")
               team-player

               ("open-play")
               ;; with poss-A we saw tackle-B, pass/open-play-B, ball-recovery-A
               ;; so maybe open-play means "contested", no clear possessing side
               team-player

               ("interception" "recovery")
               ;; pass-recovery may mean "no possession yet"
               nil

               (do
                 ;;(prn :ignoring-pass event-type)
                 nil))
      (do
        ; (prn :poss-team-ignoring-name name event-type player)
        nil))))

;; --- possession and player-possession, models and detectors --------------------------

(defn make-possession [game team player trigger-evt origin]
  (md/make ::possession
    :name :possession
    :game game
    :team team
    :winner player                                          ;; might be nil
    :events (cF (with-c-conj []
                  (let [game (mget me :game)]
                    (if (= me (mget game :possession))
                      (mget game :event)
                      (cf-freeze nil)))))
    :established? (cF+ []
                    (possession-established? me))
    :origin origin))

(defn possession-detector []
  (let [changes (atom 0)]
    (cF+ [:obs (fn [_ _ _ _ c]
                 (dag-dump :poss-detector c))]
      (let [curr-poss (if (= _cache unbound) nil _cache)
            curr-team (when curr-poss (mget curr-poss :team))]
        (if-let [[trigger-evt team player] (mget me :poss-team)]
          (do
            #_(prn (:sid trigger-evt) (team-short-name team) player (:collection/name trigger-evt)
                (first (:collection/event-type trigger-evt)))
            (cond
              (= team :none)
              (do
                ;(prn :pd-no-team-no-poss (swap! changes inc))
                nil)

              (= team curr-team)
              (do
                ;(prn :pd-same-team-same-poss team)
                curr-poss)

              :default
              (do
                ;(prn :pd-change-to team :from curr-team)
                #_(prn :make-poss!! team                    ;; (event-millis trigger-evt) (mget me :half-start)
                    (when-let [hs (mget me :half-start)]
                      (fmt-game-time (- (event-millis trigger-evt) hs))))
                (make-possession me team player trigger-evt (if _cache :takeaway :open-collect)))))
          ;; nil :poss-team signifies no opinion/unrelated
          (do
            ;(prn :no-poss-team-use-cache)
            curr-poss))))))

;;; ---- player possession -----------------------------------------

(defn make-player-possession [game team player trigger-evt]
  (md/make ::player-possession
    :game game
    :team team
    :player player
    :events (cF (with-c-conj []
                  (let [game (mget me :game)]
                    (if (= me (mget game :player-possession))
                      (mget game :event)
                      (cf-freeze nil)))))
    :established? (cF (player-poss-established? me))))

(defn player-possession-detector []
  (cF (let [curr-poss (if (= _cache unbound) nil _cache)
            curr-team (when curr-poss (mget curr-poss :team))
            curr-player (when curr-poss (mget curr-poss :player))]
        (if-let [[trigger-evt team player] (mget me :poss-team)]
          (cond
            (= team :none)
            nil

            (and (= team curr-team)
              (or (= player curr-player)
                (nil? player)))
            curr-poss

            :default
            (when player
              (make-player-possession me team player trigger-evt)))
          curr-poss))))

;; --- aggregators --------------------------------

(defn stoppage-accumulator []
  (let [eprior (atom nil)]
    (cF+ [:obs xstd-watch]
      (with-c-accumulating
        (when-let [{:keys [:collection/author] :as evt} (event me)]
          (let [prior (get @eprior author)]
            (swap! eprior assoc author evt)
            (when (and prior (resume-play-event? evt))
              [+ (event-interval-seconds prior evt)])))))))

(defn game-teams []
  (cF (with-c-associating
        ;; todo freeze after two teams found
        (when-let [{:keys [:collection/name
                           :collection/sport
                           :collection/formation
                           :collection/team-id
                           :collection/positions] :as evt} (event me)]
          (when (= "starting-xi" name)
            (prn :game-teams-rule team-id formation positions)
            [team-id {:formation        formation
                      :position-players positions
                      :player-positions (apply hash-map
                                          (reverse (apply concat positions)))}])))))

(defn player-positions []
  (cF (with-c-associating
        ;; todo freeze after two teams found
        (when-let [{:keys [:collection/name
                           :collection/team-id
                           :collection/positions] :as evt} (event me)]
          (when (= "starting-xi" name)
            (apply concat
              (for [[pos player] positions]
                [player [team-id pos]])))))))

(defn half-start []
  (cF (when-let [{:keys [:collection/name] :as evt} (event me)]
        (if (= "half-start" name)
          (event-millis evt)
          _cache))))