# shadow-graft

shadow-graft facilitates the calling of client-side functions from the server-side generated HTML.

The "graft" name is inspired by a similar function in horticulture.

> Grafting is the act of placing a portion of one plant (bud or scion) into or on a stem, root, or branch of another (stock) in such a way that a union will be formed and the partners will continue to grow. The part of the combination that provides the root is called the stock; the added piece is called the scion.

https://www.britannica.com/topic/graft

The concept is that the server generates the "root/stock" HTML tree and places leaves markers on specific positions. The client-side can then declare "scions" which are meant to enhance/grow the server-side generated HTML.


## How to use

### Client Side

```clojure
(ns demo.app
  (:require
    [shadow.graft :as graft]
    [cljs.reader :as reader]))

(defmethod graft/scion "disappearing-link" [opts link]
  (.addEventListener link "click"
    (fn [e]
      (.preventDefault e)
      (.remove link))))

(defmethod graft/scion "just-log-data" [opts container]
  (js/console.log "just-log-data" opts container))

(defn init []
  (graft/init reader/read-string))
```

The `init` fn should be called by shadow-cljs `:init-fn` in the build config. The first argument is the reader function used to parse data generated by the server. We'll use EDN as an example here. You can provide and function you want (eg. `js/JSON.parse` or transit).

### Server Side

Since there are a variety of ways to manage state on the server I'm going to use the simplest example here. But the server-side components are meant to be integrated into whatever state mechanism you use (eg. mount, component, integrant, etc.).

For demo purposes I'm gonna use a simple var. Since there really is no need to cleanup its state this is fine.

```clojure
(ns demo.server
  (:require
    [hiccup.core :refer (html)]
    [shadow.graft :as graft]))

;; using EDN as the data format via pr-str, could be anything
(def graft (graft/start pr-str))

(defn sample-server-component [req]
  (html
    [:a {:href "http://google.com"} "google.com"]
    (graft "disappearing-link" :prev-sibling)
    ;; or, slightly more verbose
    (graft/add graft "disappearing-link" :prev-sibling)
    ))

...
```

`graft` here takes at least two arguments. The id of the scion, which is also the dispatch value used in the client side `graft/scion` multi-method. The second argument specifies which DOM element this function should be targeting. In this case it targets the previous sibling DOM element.

Valid values here include
- `:none` - no reference node
- `:self` - the node created for the placeholder itself
- `:parent` - the DOM element parent containing the placeholder
- `:prev-sibling` or `:next-sibling`

The third argument is the more important one. Many things will require passing data to the client and thats where this goes.

```clojure
(defn sample-hiccup-with-data [req]
  (html
    [:div
     [:h1 "nonsense example"]
     (graft "just-log-data" :parent {:hello "world"})]))
```

The graft on the server-side just generates `<script type="shadow/graft">` tags. They are not visible and are not further interpreted by browsers until our script use them. For extra security the data is encoded via Base64.

This is intentionally simple so that any server can generate this and still hand off data to the client this way. The default implementation assumes a CLJ server but that is by no means necessary.

## A Closer To Real-World Example

In a typical reagent/re-frame app you'll have something like

```clojure
(ns graft-example.core
  (:require
    [reagent.dom :as rdom]
    [re-frame.core :as re-frame]
    ...
    ))

(defn mount-root []
  (let [root-el (.getElementById js/document "app")]
    (rdom/render [views/main-panel how-to-pass-data-here] root-el)))

(defn init []
  (re-frame/dispatch-sync [::events/initialize-db])
  (mount-root))
```

With a `<div id="app">` generated by the server somehow.

There are two problems with this that shadow-graft solves.

- No implicit way of passing data to the client
- Hardcoded DOM id/class markers

Hardcoding one DOM element id is fine, but quickly becomes a burden if you want multiple. I'd even argue that hardcoding a single id encourages thinking in only "Single Page App"-Architecture terms. Instead, something closer to the [Island Architecture](https://jasonformat.com/islands-architecture/) is desirable. This way "one" is just as easy as "many".

Instead, you now do this

```clojure
(ns graft-example.core
  (:require
    [reagent.dom :as rdom]
    [re-frame.core :as re-frame]
    [shadow.graft :as graft]
    [cljs.reader :as reader]
    ...
    ))

(defmethod graft/scion "menu"
  [{:keys [some-data props] :as opts} container]
  (re-frame/dispatch-sync [::events/initialize-menu some-data])
  (rdom/render [views/menu props] container))

(defmethod graft/scion "signup-form"
  [{:keys [props] :as opts} container]
  (rdom/render [views/signup-form props] container))

(defmethod graft/scion "login-form"
  [{:keys [form-state] :as opts} container]
  (re-frame/dispatch-sync [::events/initialize-login form-state])
  (rdom/render [views/signup-form] container))

(defn init []
  (graft/init reader/read-string))
```

With the server generating the proper graft markers it can seamlessly pass data and that data is guaranteed to be available when the multi-method is called. No need for an extra request to fetch it or embed in some other hacky ways.

Of course a singular "app" scion is just as valid.

```clojure
(defmethod graft/scion "app"
  [_ container]
  (re-frame/dispatch-sync [::events/initialize-db])
  (rdom/render [views/main-panel] container))
```

I omitted the usual hot-reload boilerplate here since the `graft/init` is only supposed to run once on page load. A simple helper method would bring that behavior back easily though. 

Note that the graft points are all traversed in the DOM (depth-first) order. They all execute when the script `init` fn is called. Since the order is guaranteed you can have your server emit a `(graft "configure" :none {:some-data "you-need"})` in your HTML `head` element and ensure that all later scions can rely on `configure` having executed first.

## Notes

If you have been long around enough in web development you might remember something like `$(".some-element").doStuff()` jquery-style plugins. They are similar in nature, but also made suffered the hardcoded id/class problems and made it difficult to pass data as well. It also had the issue of often looking for stuff that wasn't even on the page, just because it was on 1 or 50, and it was easier to just have the script always look them than to modify the script for that one page.

I have used a [similar method](https://code.thheller.com/blog/web/2014/12/20/better-javascript-html-integration.html) exclusivly for many many years. It was time to create a proper library for this, so I can throw away my old hacky functions and finally have a proper name for the technique.

Also took the time to make this work with multiple `:modules` and generating the necessary info via a [shadow-cljs](https://github.com/thheller/shadow-cljs) build hook. Docs on that to follow.