(ns teachscape.etlio.dsl
  "Contains macros and functions that can be used in the task
  definition DSL.
  
  teachscape.etlio.api/read-task evalutes task definitions in
  this namespace.")

;;
;; Implementation
;;


(defn empty-task
  ([task-name ^String description]
   {:name         task-name
    :description  description
    :dependencies #{}
    :standalone-execution true}))


;;
;; API
;;

;; protocol for returns of execution
(defprotocol ExecutionResult
  ;; the number of records that were handled as part of the outputs
  ;; (with the idea that these are the 'accepted' records that resulted
  ;; in actual work being done)
  (nbr-records [this]))

(defrecord SimpleExecutionResult
  [nbr]
  ExecutionResult
  (nbr-records [_] nbr))

(defn context
  [task-m m]
  (assoc task-m :context m))

(defn schedule
  [m sched]
  (assoc m :schedule sched))

(defn setup
  [m & fns]
  (assoc m :setup fns))

(defn finish
  [m & fns]
  (assoc m :finish fns))

(defn inputs
  [m & fns]
  (assoc m :inputs fns))

(defn transforms
  [m & fns]
  (assoc m :transforms fns))

(defn outputs
  [m & fns]
  (assoc m :outputs fns))

(defn dependencies
  [m & xs]
  (assoc m :dependencies (set (flatten xs))))


(defn execution ^ExecutionResult
  [m f]
  (assoc m :execution f))

(defn no-standalone-execution
  "Marks a task as non-standalone. It will never be executed on its
  own, only as a dependency of another task."
  ([m]
   (assoc m :standalone-execution false))
  ([m val]
   (assoc m :standalone-execution val)))

(defmacro deftask
  "Defines an ETL task.
  
  Every task has a name, multiple of inputs,
  multiple transformation steps and multiple outputs"
  [task-name description & body]
  `(let [task# (-> (empty-task ~(name task-name) ~description)
                   ~@body)]
     task#))
