(ns teachscape.etlio.tools.task-runner
  "CLI tool for running individual tasks"
  (:require [clojure.tools.cli :refer [cli]]
            [clojure.java.io            :as io]
            [clojure.edn                :as edn]
            [teachscape.etlio.evaluator :as ev]
            [taoensso.timbre            :as log]
            [me.raynes.fs               :as fs]
            [teachscape.etlio.tools.cli :as ecl])
  (:gen-class))


;;
;; Implementation
;;

(defn process-args
  [{:keys [task-file config-file]} args banner]
  (when (not (ecl/valid-task-file? task-file))
    (ecl/exit! 1 (format "%s does not exist or is not a readable Clojure file (when reading task-file)" task-file)))
  (when (not (ecl/valid-config-file? config-file))
    (ecl/exit! 1 (format "%s does not exist or is not a readable Clojure file (when reading config-file)" config-file)))
  (log/info (format "Will use config file at %s" config-file))
  (log/info (format "Will use task definition at %s" task-file))
  (let [ctx  (edn/read-string (slurp (io/reader config-file)))
        task (ev/read-task task-file)]
    (try
      (if (ev/run-task task ctx)
        (log/info (format "Running %s: success" (:name task)))
        (log/info (format "Running %s: NOT successful" (:name task))))
      (catch Exception e
        (throw (Exception. (format "Error executing task %s" (:name task)) e))))))

(defn entry-point
  [args]
  (let [[opts args banner] (cli args
                                ["-h" "--help"        "Show help" :flag true :default false]
                                ["-T" "--task-file"   "REQUIRED: Path to task definition file"]
                                ["-C" "--config-file" "Path to configuration file"])]
    (when (:help opts)
      (println banner)
      (ecl/exit! 0))
    (process-args opts args banner)))


;;
;; API
;;

(defn -main
  [& args]
  (entry-point args))
