(ns teachscape.etlio.evaluator
  "Task compilation and evaluation implementation"
  (:refer-clojure :exclude [name])
  ;; force compilation
  (:require teachscape.etlio.dsl
            [metrics.counters   :as cm]
            [metrics.histograms :as hm]
            [taoensso.timbre :as log]
            [teachscape.etlio.time :refer [time-body]]))

;;
;; Metrics
;;

(def tasks-executed
  "Counts total number of executed tasks"
  (cm/counter ["etlio" "evaluator" "tasks-executed"]))
(def execution-time
  "Histogram of task execution time, in nanoseconds"
  (hm/histogram ["etlio" "evaluator" "execution-time"]))

;;
;; Implementation
;;

(defn ^:private perform-setup
  [{:keys [setup context] :or {context {} setup []}} ctx]
  (let [merged-ctx (merge ctx context)]
    (doseq [f setup]
      (f merged-ctx))))

(defn ^:private load-inputs
  [{:keys [inputs context name] :or {inputs [] context {}}} ctx]
  (let [merged-ctx (merge ctx context)]
    (when-not (seq inputs)
      (throw (IllegalArgumentException. (format "Task %s has no inputs" name))))
    (mapcat (fn [f]
              (f merged-ctx))
            inputs)))

(defn ^:private transform-inputs
  [{:keys [transforms] :or {transforms []}} rows]
  (reduce (fn [acc f]
            (map f acc))
          rows
          transforms))

(defn ^:private load-outputs
  [{:keys [outputs name context] :or {outputs []}} rows ctx]
  (let [merged-ctx (merge ctx context)]
    (when-not (seq outputs)
      (throw (IllegalArgumentException. (format "Task %s has no outputs" name))))
    (doseq [f outputs]
      (f merged-ctx rows))))

(defn ^:private execute-task
  [m ctx]
  (perform-setup m ctx)
  (let [xs         (load-inputs m ctx)
        ys         (transform-inputs m xs)]
    (load-outputs m ys ctx)))

;;
;; API
;;

(defn read-task
  "Reads a task from a file, evaluates it and returns "
  [^String path]
  (if-let [eval-ctx (find-ns 'teachscape.etlio.dsl)]
    (binding [*ns* eval-ctx]
      (try (load-file path)
           (catch Exception e
             (throw (Exception. (format "Error loading %s" path) e)))))
    (throw (IllegalStateException. "Could not find DSL ns to evaluate in!"))))

(defn run-task
  "Executes a task"
  ([m]
     (run-task m {}))
  ([m ctx]
     ;; TODO: enforce constraints
     (let [nanos (time-body (execute-task m ctx))]
       (hm/update! execution-time nanos)
       (cm/inc! tasks-executed)
       true)))
