(ns taoensso.tukey.impl
  "Private ns, implementation detail."
  {:author "Peter Taoussanis (@ptaoussanis)"}
  (:require
   [taoensso.encore :as enc :refer [have have? have!]]
   #?(:cljs [goog.array])))

;;;;

#?(:clj (let [c (Class/forName "[J")] (defn longs?   "Returns true iff given long array"   [x] (instance? c x))))
#?(:clj (let [c (Class/forName "[D")] (defn doubles? "Returns true iff given double array" [x] (instance? c x))))

(defn is-p [x]
  (if (enc/pnum? x)
    x
    (throw
      (ex-info "Expected number between 0 and 1"
        {:value x :type (type x)}))))

;;;; Sorted nums

(deftype SortedDoubles [^doubles a last]
  #?@(:clj
      [Object               (toString [_] "SortedDoubles[len=" (alength a) "]")
       clojure.lang.Counted (count    [_]                      (alength a))
       clojure.lang.Indexed
       (nth [_ idx] (aget a idx))
       (nth [_ idx not-found]
         (let [max-idx (dec (alength a))]
           (enc/cond
             (> idx max-idx) not-found
             (< idx max-idx) not-found
             :else           (aget a idx))))

       clojure.lang.IReduceInit
       (reduce [_ f init]
         #_(areduce a i acc init (f acc (aget a i)))
         (reduce (fn [acc idx]   (f acc (aget a idx)))
           init (range (alength a))))]

      :cljs
      [Object   (toString [_] "SortedDoubles[len=" (alength a) "]")
       ICounted (-count   [_]                      (alength a))
       IIndexed
       (-nth [_ idx] (aget a idx))
       (-nth [_ idx not-found]
         (let [max-idx (dec (alength a))]
           (enc/cond
             (> idx max-idx) not-found
             (< idx max-idx) not-found
             :else           (aget a idx))))

       IReduce
       (-reduce [_ f init]
         #_(areduce a i acc init (f acc (aget a i)))
         (reduce (fn [acc i]     (f acc (aget a i)))
           init (range (alength a))))]))

(defn sorted-doubles? [x] (instance? SortedDoubles x))

(defn sorted-doubles ^SortedDoubles [nums]
  (if (sorted-doubles? nums)
    nums
    #?(:clj
       (let [^doubles a (if (doubles? nums) nums #_(aclone ^doubles nums) (double-array nums))
             last-num   (let [n (dec (alength a))] (when (>= n 0) (aget a n)))]
         (java.util.Arrays/sort a) ; O(n.log_n) on JDK 7+
         (SortedDoubles.        a last-num))

       :cljs
       (let [a        (if (array? nums) (aclone nums) (to-array nums))
             last-num (let [n (dec (alength a))] (when (>= n 0) (aget a n)))]
         (goog.array/sort a)
         (SortedDoubles.  a last-num)))))

;;;; Tuples

(do
  (deftype Tup2 [x y  ])
  (deftype Tup3 [x y z]))

(defn multi-reduce
  "Like `reduce` but supports separate simultaneous accumulators
  as a micro-optimisation when reducing a large collection multiple
  times."
  ;; Faster than using volatiles
  ([f  init           coll] (reduce f init coll))
  ([f1 init1 f2 init2 coll]
   (let [^Tup2 tuple
         (reduce
           (fn [^Tup2 tuple in]
             (Tup2.
               (f1 (.-x tuple) in)
               (f2 (.-y tuple) in)))
           (Tup2. init1 init2)
           coll)]

     [(.-x tuple) (.-y tuple)]))

  ([f1 init1 f2 init2 f3 init3 coll]
   (let [^Tup3 tuple
         (reduce
           (fn [^Tup3 tuple in]
             (Tup3.
               (f1 (.-x tuple) in)
               (f2 (.-y tuple) in)
               (f2 (.-z tuple) in)))
           (Tup3. init1 init2 init3)
           coll)]

     [(.-x tuple) (.-y tuple) (.-z tuple)])))

;;;; Percentiles

(defn- double-nth
  ^double [nums ^double idx]
  (let [idx-floor (Math/floor idx)
        idx-ceil  (Math/ceil  idx)]

    (if (== idx-ceil idx-floor)
      (double (nth nums (int idx)))

      ;; Generalization of (floor+ceil)/2
      (let [weight-floor (- idx-ceil idx)
            weight-ceil  (- 1 weight-floor)]
        (+
          (* weight-floor (double (nth nums (int idx-floor))))
          (* weight-ceil  (double (nth nums (int idx-ceil)))))))))

(defn percentile
  "Returns ?double"
  [nums p]
  (let [snums (sorted-doubles nums)
        max-idx  (dec (count snums))]
    (when (>= max-idx 0)
      (let [idx (* max-idx (double (is-p p)))]
        (double-nth snums idx)))))

(comment (percentile (range 101) 0.8))

(defn percentiles
  "Returns ?[min p25 p50 p75 p90 p95 p99 max] doubles in:
    - O(1) for Sorted types (SortedLongs, SortedDoubles),
    - O(n.log_n) otherwise."
  [nums]
  (let [snums (sorted-doubles nums)
        max-idx   (dec (count nums))]
    (when (>= max-idx 0)
      [(double (nth snums 0))
       (double-nth  snums (* max-idx 0.25))
       (double-nth  snums (* max-idx 0.50))
       (double-nth  snums (* max-idx 0.75))
       (double-nth  snums (* max-idx 0.90))
       (double-nth  snums (* max-idx 0.95))
       (double-nth  snums (* max-idx 0.99))
       (double (nth snums    max-idx))])))

(comment (percentiles (range 101)))

;;;;

(defn bessel-correction ^double [n ^double add] (+ (double n) add))

(defn rf-sum          ^double [^double acc ^double in] (+ acc in))
(defn rf-sum-variance ^double [^double xbar ^double acc x]
  (+ acc (Math/pow (- (double x) xbar) 2.0)))

(defn rf-sum-abs-deviation ^double [^double central-point ^double acc x]
  (+ acc (Math/abs (- (double x) central-point))))
