(ns taoensso.tukey.rss
  "RollingSummaryStats.
  Private ns, implementation detail."
  {:author "Peter Taoussanis (@ptaoussanis)"}
  (:require
   [taoensso.encore :as enc :refer [have have? have!]]
   [taoensso.tukey.impl :as impl])

  #?(:clj (:import [java.util LinkedList])))

;;;;

(defn- buf-new
  ([    ] #?(:clj (LinkedList.) :cljs (cljs.core/array)))
  ([init]
   #?(:clj  (if init (LinkedList.     init) (LinkedList.))
      :cljs (if init (cljs.core/array init) (cljs.core/array)))))

(defn- buf-add [buf x]
  #?(:clj  (.add ^LinkedList buf x)
     :cljs (.push            buf x)))

(defn- buf-len ^long [buf]
  #?(:clj  (.size ^LinkedList buf)
     :cljs (alength           buf)))

(defprotocol IRollingSummaryStats
  (^:private rss-deref [_])
  (^:private rss-flush [_])
  (^:private rss-add   [_ n]))

;;;;

(deftype RollingSummaryStats [msstats_ buf_ buf-size merge-counter merge-cb]
  Object
  (toString [_] ; "RollingSummaryStats[n=1, pending=8, merged=0]"
    (str
      "RollingSummaryStats[n=" (get @msstats_ :n 0)
      ", pending=" (buf-len @buf_)
      (when-let [mc merge-counter] (str ", merged=" @mc))
      "]"))

  #?@(:clj  [clojure.lang.IDeref ( deref [this] (rss-deref this))]
      :cljs [             IDeref (-deref [this] (rss-deref this))])

  #?@(:clj  [clojure.lang.IFn ( invoke [this n] (rss-add this n))]
      :cljs [             IFn (-invoke [this n] (rss-add this n))])

  IRollingSummaryStats
  (rss-deref [this] (deref (or (rss-flush this) @msstats_)))
  (rss-flush [_]
    (let [[drained] (reset-vals! buf_ (buf-new nil))]
      (if (== (buf-len drained) 0)
        nil
        (let [t0              (enc/now-nano*)
              _               (when-let [mc merge-counter] (mc))
              msstats-drained (impl/summary-stats drained)

              msstats-merged
              ;; Only drainer will update, so should be no contention
              (swap! msstats_ impl/summary-stats-merge
                msstats-drained)]

          (when merge-cb (merge-cb (- (enc/now-nano*) t0))) ; For Tufte, etc.
          msstats-merged))))

  (rss-add [this n]
    (let [buf @buf_]
      (buf-add buf n)

      (when-let [^long nmax buf-size]
        (when (> (buf-len buf) nmax)
          (rss-flush this)))

      nil)))

(defn ^:public rolling-summary-stats
  "EXPERIMENTAL, SUBJECT TO CHANGE!
  Returns a stateful RollingSummaryStats:
    (rss <num>) => Adds given number to internal buffer.
    (deref rss) => Flushes buffer if necessary, and returns a ?map
                   of summary stats for all numbers ever added to
                   rss: {:keys [n min max p25 ... p99 mean var mad]}.

  Useful for summarizing any (possibly infinite) stream of numbers.
  Used by the Tufte profiling library, and the Carmine Redis library.

  Options:
    :buffer-size - The maximum number of numbers that may be buffered
                   before next (rss <num>) call will block to flush
                   buffer and merge with any existing summary stats.

                   Larger buffers mean better performance and more
                   accurate stats, at the cost of more memory use.

    :buffer-init - Initial buffer content, useful for persistent rss.
    :sstats-init - Initial summary stats,  useful for persistent rss."

  {:added "v0.3.0 (2022-12-15)"}
  ([] (rolling-summary-stats nil))
  ([{:keys [buffer-size buffer-init sstats-init merge-cb]
     :or   {buffer-size 1e5}
     :as   opts}]

   (RollingSummaryStats.
     (atom          sstats-init)
     (atom (buf-new buffer-init))
     (long          buffer-size)
     (enc/counter)
     merge-cb ; Undocumented
     )))

(defn rolling-summary-stats-fast
  "Fastest possible `rolling-summary-stats` for Tufte, etc."
  {:added "v0.3.0 (2022-12-15)"}
  [^long buffer-size merge-cb]
  (RollingSummaryStats.
     (atom nil)
     (atom (buf-new))
     buffer-size
     nil
     merge-cb))

(comment
  (let [rss (rolling-summary-stats {:buffer-size 10})] ; 266 qb
    [(enc/qb 1e6 (rss (rand-int 1000))) (str rss) @rss]))

;;;; Print methods

#?(:clj
   (let [ns *ns*]
     (defmethod print-method RollingSummaryStats
       [x ^java.io.Writer w] (.write w (str "#" ns "." x)))))
