(ns taoensso.tufte.stats
  "Basic stats utils. Private, subject to change."
  (:require
   [clojure.string  :as str]
   [taoensso.encore :as enc]
   [taoensso.tukey  :as tukey]))

;;;; Tukey imports

(enc/defalias sstats       tukey/summary-stats)
(enc/defalias sstats-merge tukey/summary-stats-merge)

(comment @(sstats [1 2 3]))

;;;; Formatting

(defn- perc [n d] (str (Math/round (* (/ (double n) (double d)) 100.0)) "%"))
(comment [(perc 1 1) (perc 1 100) (perc 12 44)])

#?(:clj (def ^:private locale (java.util.Locale/getDefault)))
#?(:clj (defn- fmt [pattern & args] (String/format locale pattern (to-array args))))

(defn- fmt-2f    [n] #?(:clj (fmt "%.2f" n) :cljs (str (enc/round2 n))))
(defn- fmt-calls [n] #?(:clj (fmt "%,d"  n) :cljs
                        (str ; Thousands separator
                          (when (neg? n) "-")
                          (->>
                            (str (Math/abs n))
                            (reverse)
                            (partition 3 3 "")
                            (map str/join)
                            (str/join ",")
                            (str/reverse)))))

(defn- fmt-nano [nanosecs]
  (let [ns (double nanosecs)]
    (cond
      (>= ns 6e10) (str (fmt-2f (/ ns 6e10)) "m ")
      (>= ns 1e9)  (str (fmt-2f (/ ns 1e9))  "s ")
      (>= ns 1e6)  (str (fmt-2f (/ ns 1e6))  "ms")
      (>= ns 1e3)  (str (fmt-2f (/ ns 1e3))  "μs")
      :else        (str (fmt-2f    ns)       "ns"))))

(comment
  (fmt "%.2f" 12345.67890)
  (fmt-2f     12345.67890)
  (fmt-calls  12345)
  (fmt-nano   12345.67890))

(def     all-format-columns [:n-calls :min   :p25 :p50   :p75 :p90 :p95 :p99 :max :mean :mad :clock :total])
(def default-format-columns [:n-calls :min #_:p25 :p50 #_:p75 :p90 :p95 :p99 :max :mean :mad :clock :total])

(def default-format-id-fn (fn [id] (str id)))

;; id-sstats* => {<id> <sstats>} or {<id> <sstats-map>}

(defn get-max-id-width
  [id-sstats*
   {:keys [format-id-fn]
    :or   {format-id-fn default-format-id-fn}}]

  (when id-sstats*
    (reduce-kv
      (fn [^long acc id _sstats*]
        (let [c (count (format-id-fn id))]
          (if (> c acc) c acc)))
      9 ; (count "Accounted")
      id-sstats*)))

(defn sstats-format
  "Given {<id> <sstats>} or {<id> <sstats-map>}, returns a formatted table
  string. Assumes nanosecond clock, and stats based on profiling id'd
  nanosecond times."
  [clock-total id-sstats*
   {:keys [columns sort-fn format-id-fn max-id-width] :as opts
    :or   {columns      default-format-columns
           sort-fn      (fn [ss] (get (enc/force-ref ss) :sum))
           format-id-fn default-format-id-fn}}]

  (when id-sstats*
    (enc/have? [:el all-format-columns] :in columns)
    (let [clock-total (long clock-total)
          ^long accounted-total
          (reduce-kv
            (fn [^long acc _id ss]
              (+ acc (long (get (enc/force-ref ss) :sum))))
            0 id-sstats*)

          sorted-ids
          (sort-by
            (fn [id] (sort-fn (get id-sstats* id)))
            enc/rcompare
            (keys id-sstats*))

          ^long max-id-width
          (or
            max-id-width
            (get-max-id-width id-sstats* opts))

          column->pattern
          {:id      {:heading "pId"    :min-width max-id-width :align :left}
           :n-calls {:heading "nCalls"}
           :min     {:heading "Min"}
           :p25     {:heading "25% ≤"}
           :p50     {:heading "50% ≤"}
           :p75     {:heading "75% ≤"}
           :p90     {:heading "90% ≤"}
           :p95     {:heading "95% ≤"}
           :p99     {:heading "99% ≤"}
           :max     {:heading "Max"}
           :mean    {:heading "Mean"}
           :mad     {:heading "MAD"   :min-width 5}
           :total   {:heading "Total" :min-width 6}
           :clock   {:heading "Clock"}}

          sb (enc/str-builder "")

          append-col
          (fn [column s]
            (let [{:keys [min-width align]
                   :or   {min-width 10 align :right}}
                  (get column->pattern column)]

              (enc/sb-append sb
                (enc/format
                  (str "%" (case align :left "-" :right "") min-width "s")
                  s))))]

      ; Write header rows
      (doseq [column (into [:id] columns)]
        (when-not (= :id column)
          (enc/sb-append sb " "))
        (append-col column (get-in column->pattern [column :heading])))

      (enc/sb-append sb "\n\n")

      ; Write id rows
      (doseq [id sorted-ids]
        (let [ssm  (enc/force-ref (get id-sstats* id))
              sum  (get ssm :sum)
              mean (get ssm :mean)]

          (append-col :id (format-id-fn id))
          (doseq [column columns]
            (enc/sb-append sb " ")
            (case column
              :n-calls (append-col column (fmt-calls (get ssm :n)))
              :mean    (append-col column (fmt-nano mean))
              :mad     (append-col column (str "±" (perc (get ssm :mad) mean)))
              :total   (append-col column (perc sum clock-total))
              :clock   (append-col column (fmt-nano sum))
              (do      (append-col column (fmt-nano (get ssm column))))))

          (enc/sb-append sb "\n")))

      ; Write accounted row
      (enc/sb-append sb "\n")
      (append-col :id "Accounted")
      (doseq [column columns]
        (enc/sb-append sb " ")
        (case column
          :total (append-col column (perc accounted-total clock-total))
          :clock (append-col column (fmt-nano accounted-total))
          (do    (append-col column ""))))

      ; Write clock row
      (enc/sb-append sb "\n")
      (append-col :id "Clock")
      (doseq [column columns]
        (enc/sb-append sb " ")
        (case column
          :total (append-col column "100%")
          :clock (append-col column (fmt-nano clock-total))
          (do    (append-col column ""))))

      (enc/sb-append sb "\n")
      (str sb))))

(comment
  (defn rand-vs [n & [max]]
    (take n (repeatedly (partial rand-int (or max Integer/MAX_VALUE)))))

  (println
    (sstats-format (* 1e6 30)
      {:foo (sstats (rand-vs 1e4 20))
       :bar (sstats (rand-vs 1e2 50))
       :baz (sstats (rand-vs 1e5 30))}
      {}) "\n"))
