(ns taoensso.tufte.impl
  "Private implementation details.
  `profiled` -> [<result> <derefable-and-mergeable-pstats>]."
  (:require [clojure.string  :as str]
            [taoensso.encore :as enc :refer-macros []]
            [taoensso.tufte.stats :as stats])

  #?(:clj (:import [java.util.concurrent ArrayBlockingQueue])))

;;;; PData (Profiling Data)
;; Implementation-level state while profiling.

(def ^:dynamic *pdata* "nnil iff profiling active" nil)

(declare ^:private pdata->pstats)

;; id-times: {<id> (<time> ...)_}_ -> {<id> (<time> ...)}
;; id-stats: {<id> (<map>  ...) }_ -> {<id> (<map>  ...)}
#?(:clj
   (deftype PData [^long t0 ^long t1 ^long nmax id-times id-stats]
     clojure.lang.IDeref (deref [this] (pdata->pstats this)))

   :cljs
   (deftype PData [^long t0 ^long t1 ^long nmax id-times id-stats]
     IDeref (-deref [this] (pdata->pstats this))))

(defmacro new-pdata
  ([    ] `(new-pdata 2e6))
  ([nmax] `(PData. (enc/now-nano*) -1 ~nmax (atom {}) (atom nil))))

(comment
  (enc/qb 1e6 (new-pdata)) ; 217.91
  @(new-pdata))

;;;; PStats (Profiling Stats)
;; API-level state we'll return from `profiled`: derefable, mergeable.
;; Basically just an accumulation wrapper around `pdata`s.

#?(:clj
   (deftype PStats [pdatas realized-final-stats_]
     clojure.lang.IDeref   (deref      [_]           @realized-final-stats_)
     clojure.lang.IPending (isRealized [_] (realized? realized-final-stats_)))

   :cljs
   (deftype PStats [pdatas realized-final-stats_]
     IDeref   (-deref     [_]           @realized-final-stats_)
     IPending (-realized? [_] (realized? realized-final-stats_))))

(defn- fast-into [c0 c1] (if (> (count c0) (count c1)) (into c0 c1) (into c1 c0)))
(declare ^:private realize-final-stats)

(defn merge-pstats "Cheap, just merges pdatas"
  [ps0 ps1]
  (if ps0
    (if ps1
      (let [pds (fast-into (.-pdatas ^PStats ps0) (.-pdatas ^PStats ps1))]
        (PStats. pds (delay (realize-final-stats pds))))
      ps0)
    ps1))

(defn- deref-vals [m] (reduce-kv (fn [m k v] (assoc m k @v)) m m))
(defn- pdata->pstats "Wraps `[pdata-with-derefed-stats]`" [^PData pdata]
  (let [id-times (deref-vals @(.-id-times pdata))
        id-stats             @(.-id-stats pdata)
        pds [(PData. (.-t0 pdata) (enc/now-nano*) -1 id-times id-stats)]]
    (PStats. pds (delay (realize-final-stats pds)))))

(defn- merge-list-vals [m0 m1]
  (reduce-kv
    (fn [m k v]
      (assoc m k (fast-into (get m k ()) v)))
    m0 m1))

(comment (merge-list-vals {:a '(:a1 :a2 :a3) :b '(:b3)} {:b '(:b1 :b2)}))

(defn- realize-final-stats
  "Returns final API-level output."
  [pdatas]
  (let [;; t0 (enc/nano-time*)

        [^PData pd1] pdatas
        next-pds (enc/vnext pdatas)

        ^PData pd2 ; Merged clock, times & stats lists
        (reduce
          (fn [^PData pd0 ^PData pd1]
            (let [pd0-t0 (.-t0 pd0)
                  pd0-t1 (.-t1 pd0)
                  pd1-t0 (.-t0 pd1)
                  pd1-t1 (.-t1 pd1)

                  pd2-t0 (if (< pd0-t0 pd1-t0) pd0-t0 pd1-t0)
                  pd2-t1 (if (> pd0-t1 pd1-t1) pd0-t1 pd1-t1)

                  pd2-id-times (merge-list-vals (.-id-times pd0) (.-id-times pd1)) ; {<id> (<time> ...)}
                  pd2-id-stats (merge-list-vals (.-id-stats pd0) (.-id-stats pd1)) ; {<id> (<map>  ...)}
                  ]

              (PData. pd2-t0 pd2-t1 -1 pd2-id-times pd2-id-stats)))
          pd1
          next-pds)

        pd2-id-times (.-id-times pd2)
        pd2-id-stats (.-id-stats pd2)

        ;; Actually merge stats (can be expensive)
        id-final-stats ; {<id> <final-stats-map>}
        (reduce-kv
          (fn [m id times]
            (let [stats<times (stats/stats times)
                  merged (reduce stats/merge-stats stats<times (get pd2-id-stats id))]
              (assoc m id merged)))

          pd2-id-times
          pd2-id-times)

        ;; t1 (enc/nano-time*)
        ]

    {:clock {:t0 (.-t0 pd2) :t1 (.-t1 pd2)}
     :id-stats id-final-stats}))

;;;; Time capture

(defn capture-time! [^PData pdata id ns-elapsed]
  (let [id-times_ (.-id-times pdata) ; {<id> (<time> ...)}
        nmax (.-nmax pdata)

        ;; Get the id'd atom (accumulator)
        times_
        (or (get @id-times_ id) ; Common case
            (get (swap! id-times_ (fn [m] (assoc m id (get m id (atom ()))))) id))

        ?pulled-times
        (loop []
          (let [times0 @times_
                times1 (conj times0 ns-elapsed)]
            (if (< (count times1) nmax) ; Times need compaction
              (if (compare-and-set! times_ times0 times1) ; Does == comparison, fast
                nil ; Common case
                (recur))

              (if (compare-and-set! times_ times0 ()) ; ''
                times1 ; Pull accumulated times, rare
                (recur)))))]

    ;; Do compaction, rare but expensive.
    ;; Though note that expense doesn't generally distort p times unless there's
    ;; p nesting (in which case outside p time will include inside p's capture
    ;; time).
    (when-let [times ?pulled-times]
      (let [t0          (enc/now-nano*)
            stats<times (stats/stats times)
            id-stats_   (.-id-stats pdata)]

        (swap! id-stats_ (fn [m] (assoc m id (conj (get m id ()) stats<times))))
        (recur pdata :tufte/compaction (- (enc/now-nano*) t0))))))

(comment (let [pd (new-pdata)] (enc/qb 1e6 (capture-time! pd :foo 1)))) ; 187.86

;;;; Output handlers

(enc/defonce handlers_ "{<hid> <handler-fn>}" (atom nil))

#?(:clj
   (enc/defonce ^:private ^ArrayBlockingQueue handler-queue
     "While user handlers should ideally be non-blocking, we'll use a queue
     here to be safe + make sure we never tie up the execution thread."
     (ArrayBlockingQueue. 1024)))

(defn- handle-blocking! [m]
  (enc/run-kv!
    (fn [id f]
      (enc/catching (f m) e
        (enc/catching ; Esp. nb for Cljs
          (println (str "WARNING: Uncaught Tufte `" id "` handler error\n" e)))))
    @handlers_))

#?(:clj  (declare ^:private handler-thread_))
#?(:cljs (defn handle! [m] (handle-blocking! m) nil))
#?(:clj  (defn handle! [m] (.offer handler-queue m) @handler-thread_ nil))
#?(:clj
   (defonce ^:private handler-thread_
     (delay
       (let [f (fn []
                 (loop []
                   (let [m (.take handler-queue)]
                     ;; Note: just drop if no registered handlers
                     (handle-blocking! m)
                     (recur))))]
         (doto (Thread. f)
           (.setDaemon true)
           (.start))))))
