(ns taoensso.tufte.impl
  "Private implementation details."
  (:require [clojure.string  :as str]
            [taoensso.encore :as enc :refer-macros ()])
                                       
                                                           
        
  (:require-macros
   [taoensso.tufte.impl :refer
    (nano-time mutable-times mt-add mt-count atom?)]))

;;;; Pdata

(defrecord PData [^long __t0]) ; + {<id> <times> :__m-id-stats {<id> <IdStats>}}
(def ^:dynamic *pdata_* "Non-nil iff dynamic profiling active." nil)

;; Would esp. benefit from ^:static support / direct linking / a Java class
(def ^:static pdata-proxy "Non-nil iff thread-local profiling active."
       
                                                    
       
                              
                                                

         ; Assuming we have Clojure 1.7+ for Cljs
  (let [state_ (volatile! false)] ; Automatically thread-local in js
    (fn
      ([]                @state_)
      ([new-val] (vreset! state_ new-val)))))

(comment (enc/qb 1e6 (pdata-proxy))) ; 48.39

                                                           
                                                            
(comment (macroexpand '(new-pdata-thread)))

;;;; Stats

(defrecord   Clock [^long t0 ^long t1 ^long total])
(defrecord IdStats [^long count ^long time ^long mean ^long mad-sum
                    ^double mad ^long min ^long max])
(defrecord   Stats [clock id-stats-map])

;;;; Time tracking
;; We can use mutable time accumulators when thread-local.
;; Note that LinkedList uses more mem but is faster than
;; java.util.ArrayList.

                                                                        
                                                                                    
                                                                                                                
                                                                                                             

;; Compaction (times->interim-stats) helps to prevent OOMs:
(def      ^:private ^:const nmax-times ">n will trigger compaction" (long 2e6))
(declare  ^:private times->IdStats)
                             
               
                                     
                                       

(defn ^:static capture-time! [pdata-or-pdata_ id t-elapsed]
  (if (atom? pdata-or-pdata_)

    ;; Using immutable thread-safe times, atom for coordination
    (let [pdata_ pdata-or-pdata_
          ?pulled-times
          (loop []
            (let [pdata @pdata_]
              (let [times (get pdata id ())]
                (if (>= (count times) nmax-times)
                  (if (compare-and-set! pdata_ pdata ; Never leave empty
                        (assoc pdata id (conj () t-elapsed)))
                    times ; Pull accumulated times
                    (recur))

                  (if (compare-and-set! pdata_ pdata
                        (assoc pdata id (conj times t-elapsed)))
                    nil
                    (recur))))))]

      (when-let [times ?pulled-times] ; Compact
        (let [id-stats (get-in @pdata_ [:__m-id-stats id])
              id-stats (times->IdStats times id-stats)]
          ;; Can reasonably assume that our id-stats key is
          ;; uncontended atm:
          (swap! pdata_ assoc-in [:__m-id-stats id] id-stats))))

    ;; Using mutable thread-local times (cheap), no coordination
    (let [pdata pdata-or-pdata_]
      (if-let [times (get pdata id)]
        (if (>= (long (mt-count times)) nmax-times) ; Compact
          (let [m-id-stats (get pdata :__m-id-stats)
                id-stats   (get m-id-stats id)
                id-stats   (times->IdStats times id-stats)
                m-id-stats (assoc m-id-stats id id-stats)
                times      (mutable-times)]

            (mt-add times t-elapsed) ; Never leave empty
            (pdata-proxy (assoc pdata id times :__m-id-stats m-id-stats)))

          ;; Common case
          (mt-add times t-elapsed))

        ;; Init case
        (let [times (mutable-times)]
          (mt-add times t-elapsed)
          (pdata-proxy (assoc pdata id times))))))

  nil)

(def ^:private ^:const max-long                             9007199254740991)
(defn- times->IdStats [times ?interim-id-stats]
  (let [times      (vec   times) ; Faster to reduce
        ts-count   (count times)
        _          (assert (not (zero? ts-count)))
        ts-time    (reduce (fn [^long acc ^long in] (+ acc in)) 0 times)
        ts-mean    (/ (double ts-time) (double ts-count))
        ts-mad-sum (reduce (fn [^long acc ^long in] (+ acc (Math/abs (- in ts-mean)))) 0 times)
        ts-min     (reduce (fn [^long acc ^long in] (if (< in acc) in acc)) max-long     times)
        ts-max     (reduce (fn [^long acc ^long in] (if (> in acc) in acc)) 0            times)]

    (if-let [^IdStats id-stats ?interim-id-stats] ; Merge over previous stats
      (let [s-count   (+ (.-count id-stats) ts-count)
            s-time    (+ (.-time  id-stats) ^long ts-time)
            s-mean    (/ (double s-time) (double s-count))
            s-mad-sum (+ (.mad-sum id-stats) ^long ts-mad-sum)
            s0-min    (.-min id-stats)
            s0-max    (.-max id-stats)]

        ;; Batched "online" MAD calculation here is better= the standard
        ;; Knuth/Welford method, Ref. http://goo.gl/QLSfOc,
        ;;                            http://goo.gl/mx5eSK.

        (IdStats. s-count s-time s-mean s-mad-sum
          (/ (double s-mad-sum) (double s-count))
          (if (< s0-min ^long ts-min) s0-min ts-min)
          (if (> s0-max ^long ts-max) s0-max ts-max)))

      (IdStats. ts-count ts-time ts-mean ts-mad-sum
        (/ (double ts-mad-sum) (double ts-count))
        ts-min
        ts-max))))

(comment (times->IdStats (mutable-times) nil)) ; Should throw

;;;;

(defn pdata->Stats
  "Wraps up a pdata run. Nb: recall that we need a *fresh* `(pdata-proxy)`
  here for thread-local profiling."
  [^PData current-pdata]
  (let [t1         (nano-time)
        t0         (.-__t0 current-pdata)
        m-id-stats (get    current-pdata :__m-id-stats)
        m-times    (dissoc current-pdata :__m-id-stats :__t0)]
    (Stats.
      (Clock. t0 t1 (- t1 t0))
      (reduce-kv
        (fn [m id times]
          (assoc m id (times->IdStats times (get m-id-stats id))))
        #_(transient {}) {} ; Usu. <10 entries
        m-times))))

;;;; Misc

;; Code shared with Timbre
(def compile-ns-filter "Returns (fn [?ns]) -> truthy."
  (let [compile1
        (fn [x] ; ns-pattern
          (cond
            (enc/re-pattern? x) (fn [ns-str] (re-find x ns-str))
            (string? x)
            (if (enc/str-contains? x "*")
              (let [re
                    (re-pattern
                      (-> (str "^" x "$")
                          (str/replace "." "\\.")
                          (str/replace "*" "(.*)")))]
                (fn [ns-str] (re-find re ns-str)))
              (fn [ns-str] (= ns-str x)))

            :else (throw (ex-info "Unexpected ns-pattern type"
                           {:given x :type (type x)}))))]

    (fn self
      ([ns-pattern] ; Useful for user-level matching
       (let [x ns-pattern]
         (cond
           (map? x) (self (:whitelist x) (:blacklist x))
           (or (vector? x) (set? x)) (self x nil)
           (= x "*") (fn [?ns] true)
           :else
           (let [match? (compile1 x)]
             (fn [?ns] (if (match? (str ?ns)) true))))))

      ([whitelist blacklist]
       (let [white
             (when (seq whitelist)
               (let [match-fns (mapv compile1 whitelist)
                     [m1 & mn] match-fns]
                 (if mn
                   (fn [ns-str] (enc/rsome #(% ns-str) match-fns))
                   (fn [ns-str] (m1 ns-str)))))

             black
             (when (seq blacklist)
               (let [match-fns (mapv compile1 blacklist)
                     [m1 & mn] match-fns]
                 (if mn
                   (fn [ns-str] (not (enc/rsome #(% ns-str) match-fns)))
                   (fn [ns-str] (not (m1 ns-str))))))]
         (cond
           (and white black)
           (fn [?ns]
             (let [ns-str (str ?ns)]
               (if (white ns-str)
                 (if (black ns-str)
                   true))))

           white (fn [?ns] (if (white (str ?ns)) true))
           black (fn [?ns] (if (black (str ?ns)) true))
           :else (fn [?ns] true) ; Common case
           ))))))

(comment
  (def nsf? (compile-ns-filter #{"foo.*" "bar"}))
  (enc/qb 1e5 (nsf? "foo")) ; 20.44
  )

;;;; Output handlers

(enc/defonce handlers_ "{<hid> <handler-fn>}" (atom nil))

     
                                                        
                                                                        
                                                                    
                             

;; Nb we intentionally, silently swallow any handler errors
(defn- handle-blocking! [m]
  (enc/run-kv! (fn [_ f] (enc/catch-errors* (f m))) @handlers_))

                                          
       (defn handle! [m] (handle-blocking! m) nil)
                                                                       
     
                                  
        
                  
                      
                                              
                                                              
                                      
                             
                       
                         
                    

;;;;;;;;;;;; This file autogenerated from src/taoensso/tufte/impl.cljx
