(ns taoensso.tufte
  "A simple, fast, monitoring profiler for Clojure/Script.

  Usage: wrap+name interesting body exprs with the `p` macro. Then activate
  profiling of these wrapped exprs using the `profiled` or `profile` macros:

    (profiled {} (p :my-fn (my-fn))) ; Returns [<body-result> <?stats-map>]
    (profile  {} (p :my-fn (my-fn))) ; Returns  <body-result>, dispatches
                                     ; ?stats-map to any registered handlers.

  Extensive facilities are provided for compile-time elision and runtime
  filtering.

  See the relevant docstrings for more info:
    `p`, `profiled`, `profile`, `set-handler!` ; Core API

    (p [id & body] [opts & body]) ; e.g. `(p ::my-pid (do-work))`
    (profiled [opts & body])      ; e.g. `(profiled {:level 2} (my-fn))`
    (profile  [opts & body])      ; e.g. `(profiled {:level 2} (my-fn))`

    (set-handler! [handler-id ?handler-fn ?ns-filter])

  How/where to use this library:
    Tufte is highly optimized: even without elision, you can usually leave
    profiling code in production (e.g. for sampled profiling, or to detect
    unusual performance behaviour). Tufte's stats maps are well suited to
    programmatic inspection + analysis."

  {:author "Peter Taoussanis (@ptaoussanis)"}

       
           
                                
                                  

                                                           

        
  (:require
   [taoensso.encore     :as enc]
   [taoensso.tufte.impl :as impl :refer (IdStats Stats Clock)])

         (:require-macros [taoensso.tufte :refer (profiled)]))

;;;; Level filtering

;; We distinguish between run and min levels to ensure that it's
;; always possible to set the min-level > any run level (i.e. to
;; disable profiling)
(defn valid-run-level? [x] (if (#{0 1 2 3 4 5}   x) true false))
(defn valid-min-level? [x] (if (#{0 1 2 3 4 5 6} x) true false))

(def ^:const invalid-run-level-msg         "Invalid Tufte profiling level: should be int e/o #{0 1 2 3 4 5}")
(def ^:const invalid-min-level-msg "Invalid minimum Tufte profiling level: should be int e/o #{0 1 2 3 4 5 6}")

(defn ^:static valid-run-level [x]
  (or (#{0 1 2 3 4 5} x)
      (throw (ex-info invalid-run-level-msg {:given x :type (type x)}))))

(comment (enc/qb 1e5 (valid-run-level 4))) ; 7.82

(defn ^:static valid-min-level [x]
  (or (#{0 1 2 3 4 5 6} x)
      (throw (ex-info invalid-min-level-msg {:given x :type (type x)}))))

(def ^:dynamic  *min-level* "e/o #{0 1 2 3 4 5 6}" 2)
(defn        set-min-level!
  "Sets root binding of minimum profiling level, e/o #{0 1 2 3 4 5 6}.
    0 => Enable  all profiling.
    6 => Disable all profiling."
  [level]
  (valid-min-level level)
         (set!             *min-level*         level)
                                                      )

(comment (enc/qb 1e6 *min-level*)) ; 25.93

                        
                                                                            
                               
                                
                
                      
       
                             
                                                             
                                                              

;;;; Namespace filtering

(def -compile-ns-filter "Caching `impl/comple-ns-filter`."
  (enc/memoize_ impl/compile-ns-filter))

(def ^:dynamic *ns-filter* "(fn [?ns] -> truthy)." (-compile-ns-filter "*"))

(defn set-ns-pattern!
  "Sets root binding of namespace filter.
  See `compile-ns-filter` docstring for details on `ns-pattern` arg."
  [ns-pattern]
  (let [nsf? (-compile-ns-filter ns-pattern)]
           (set!             *ns-filter*        nsf?)
                                                       ))

                         
                                               
                                                                     
                     
                                                          
             

(comment
  (def nsf? (compile-ns-filter #{"foo.*" "bar"}))
  (nsf? "foo.bar")
  (with-ns-pattern "foo.baz"    (profiled {} (p {:id "id"} "body")))
  (with-ns-pattern "taoensso.*" (profiled {} (p {:id "id"} "body"))))

;;;; Combo filtering

     
                                     
                                                        
                                                                    
                             

     
                                     
                                                         
                    
                                                                             
                                              

                                           
             
                                                           
                          
      
        
                      
                                     
                                                                            
                                                           

                          
                                                                     
                                                

(defn may-profile?
  "Returns true iff level and ns are runtime unfiltered."
  ([level   ] (may-profile? level *ns*))
  ([level ns]
   (if (>=  ^long (valid-run-level level)
         ;; ^long (valid-min-level *min-level*)
            ^long                  *min-level* ; Assume valid
         )
     (if (*ns-filter* ns) true false)
     false)))

(comment (enc/qb 1e5 (may-profile? 2))) ; 17.74

;;;; Output handlers
;; Handlers are used for `profile` output, let us nicely decouple stat
;; creation and consumption.

(defrecord HandlerVal [level ns-str ?id stats stats-str_ ?data])
(defn  set-handler!
  "Use to un/register interest in stats output.

  nil  `?handler-fn` => unregister id.
  nnil `?handler-fn` =>   register id:
    `(handler-fn {:level _ :ns-str _ :stats _ :stats-str_ _ :?id _ :?data _})`
    will be called for stats output produced by any `profile` calls.

  Handler ideas:
    Save to a db, log, `put!` to an appropriate `core.async`
    channel, filter, aggregate, use for a realtime analytics dashboard,
    examine for outliers or unexpected output, ...

  NB: handler errors will be silently swallowed. Please `try`/`catch`
  and appropriately deal with (e.g. log) possible errors *within* your
  handler fns."
  ([handler-id ?handler-fn] (set-handler! handler-id ?handler-fn nil))
  ([handler-id ?handler-fn ?ns-filter]
   (if (nil? ?handler-fn)
     (set (keys (swap! impl/handlers_ dissoc handler-id)))
     (let [f ?handler-fn
           f (if (or (nil? ?ns-filter) (= ?ns-filter "*"))
               f
               (let [nsf? (-compile-ns-filter ?ns-filter)]
                 (fn [m]
                   (when (nsf? (get m :?ns-str))
                     (f m)))))]
       (set (keys (swap! impl/handlers_ assoc  handler-id f)))))))

(defn set-basic-println-handler! []
  (set-handler! :basic-println-handler
    (fn [m]
      (let [{:keys [stats-str_ ?id ?data]} m
            stats-str (force stats-str_)]
        (println
          (str
            (when ?id   (str "\nid: "   ?id))
            (when ?data (str "\ndata: " ?data))
            "\n" stats-str))))))

(comment (set-basic-println-handler!))

;;;; Some low-level primitives

(defn profiling? "Returns e/o #{nil :thread :dynamic}."
  [] (if impl/*pdata_* :dynamic (if (impl/pdata-proxy) :thread)))

(comment (enc/qb 1e6 (profiling?))) ; 51.01

(defn start-profiling-thread!
  "Warning: this is a low-level primitive. Prefer higher-level macros
  like `profile` when possible.

  NB: must be accompanied by a call to `stop-profiling-thread!`
  (e.g. using `try`/`finally`)."
  []
  (impl/pdata-proxy (impl/new-pdata-thread))
  nil)

(defn stop-profiling-thread!
  "Warning: this is a low-level primitive."
  []
  (when-let [pdata (impl/pdata-proxy)]
    (let [result (impl/pdata->Stats pdata)]
      (impl/pdata-proxy nil)
      result)))

;;;; Core macros

                  
                                                                     

                                                                          
                        

                                                       
                          

                    
                                                   
                                                                  
                                                                        

               
                          

                         
            
                                                                             
                                                    
                                                      

                                           
                                               
                                               

                                                               

                                     
                      
                           
                                             
                                                        
                                                                      

                      
                               
                                                      
                                                 
                                            
                                                              
                                        
                             

                               
                   
                                                           
                                          
                                                                      
                                    
                                                  
                                   

(declare format-stats)
                 
                                                          

                                                                          
                                                                       
                       

                                                                        
                           

                    
                                                   
                                                                  
                                                                       
                                                                            
                                                                        

               
                          

                         
            
                                                                            
                                                    
                                                     

                                           
                                               
                                              
                                   
                                      

                                                               

                                                      
                     
                        
                                                              
                                                          
                    

(comment
  (profiled {} "body")
  (profiled {:when (chance 0.5)} "body")
  (profile  {:id ::my-id} "body"))

           
                                                                         

                                                                     
               

                    
                                                                   
                                                 

                                          
             
                          
                                                   
                                 
                               

                                                                       
                                                            
                                        

                        
            
                                            
                                                    
                     
                                             

                              
                  
                                             
                                                                       
                                
                                            
                                       
                                             
                                                             
                                 
                        
                           

                                                

(comment
  (p :p1 "body")
  (profiled {} (p :p1))
  (profiled {} (p {:level 5 :id :p1}))
  (profiled {} (p (let [x :foo/id] x) "body"))
  (enc/qb 1e5  (profiled {} 2 (p :p1))) ; 195.56
  (enc/time-ms (profiled {} 2 (enc/qb 1e6 (p :p1)))) ; 2485
  (profiled {:level 2 :when (chance 0.5)} (p :p1 "body")))

;;;; Public user utils

(defn compile-ns-filter
  "Returns (fn [?ns]) -> truthy. Some example patterns:
    \"foo.bar\", \"foo.bar.*\", #{\"foo\" \"bar\"},
    {:whitelist [\"foo.bar.*\"] :blacklist [\"baz.*\"]}"
  [ns-pattern] (impl/compile-ns-filter ns-pattern))

(defn chance "Returns true with 0<`p`<1 probability."
  [p] (< ^double (rand) (double p)))

(defn merge-stats
  "Merges stats maps from multiple runs or threads.
  Automatically identifies and merges concurrent time windows."
  [s1 s2]
  (if s1
    (if s2
      (let [^Stats s1 s1
            ^Stats s2 s2
            ^Clock clock1 (.-clock s1)
            ^Clock clock2 (.-clock s2)
            s1-t0 (.-t0 clock1)
            s1-t1 (.-t1 clock1)
            s2-t0 (.-t0 clock2)
            s2-t1 (.-t1 clock2)
            clock-overlap?
            (and
              (not (zero? s1-t0))
              (not (zero? s2-t0))
              (or
                (and (<= s2-t0 s1-t1)
                     (>= s2-t1 s1-t0))
                (and (<= s1-t0 s2-t1)
                  (>= s1-t1 s2-t0))))

            ^Clock clock3
            (if clock-overlap?
              (let [s3-t0 (if (< s1-t0 s2-t0) s1-t0 s2-t0)
                    s3-t1 (if (< s1-t1 s2-t1) s1-t1 s2-t1)]
                (Clock. s3-t0 s3-t1 (- s3-t1 s3-t0)))
              (Clock. 0 0 (+ (.-total clock1) (.-total clock2))))

            m-id-stats1 (.-id-stats-map s1)
            m-id-stats2 (.-id-stats-map s2)
            all-ids (into (set (keys m-id-stats1)) (keys m-id-stats2))

            m-id-stats3
            (reduce
              (fn [m id]
                (let [sid1 (get m-id-stats1 id)
                      sid2 (get m-id-stats2 id)]
                  (if sid1
                    (if sid2
                      (let [^IdStats sid1 sid1
                            ^IdStats sid2 sid2
                            s1-count   (.-count   sid1)
                            s1-time    (.-time    sid1)
                            s1-mad-sum (.-mad-sum sid1)
                            s1-min     (.-min     sid1)
                            s1-max     (.-max     sid1)

                            s2-count   (.-count   sid2)
                            s2-time    (.-time    sid2)
                            s2-mad-sum (.-mad-sum sid2)
                            s2-min     (.min      sid2)
                            s2-max     (.max      sid2)

                            s3-count   (+ s1-count   s2-count)
                            s3-time    (+ s1-time    s2-time)
                            s3-mad-sum (+ s1-mad-sum s2-mad-sum)]

                        (assoc m id
                          (IdStats.
                            s3-count
                            s3-time
                            (/ (double s3-time) (double s3-count))
                            s3-mad-sum
                            (/ (double s3-mad-sum) (double s3-count))
                            (if (< s1-min s2-min) s1-min s2-min)
                            (if (> s1-max s2-max) s1-max s2-max))))
                      m #_(assoc m id sid1))
                    (assoc m id sid2))))
              #_(transient m-id-stats1) m-id-stats1 ; Usu. <10 entries
              all-ids)]
        (Stats. clock3 m-id-stats3))
      s1)
    s2))

(defn stats-accumulator
  "Experimental, subject to change!
  Small util to help merge stats maps from multiple runs or threads.
  Returns a stateful fn with arities:
    ([stats]) ; Accumulates the given stats (you may call this from any thread)
    ([])      ; Deref: returns the merged value of all accumulated stats"
  []
  (let [acc_ (atom nil)
        reduce-stats_
        (delay
          (let [merge-stats (enc/memoize_ merge-stats)]
            (enc/memoize_ (fn [acc] (reduce merge-stats nil acc)))))]

    (fn stats-accumulator
      ([stats] (when stats (swap! acc_ conj stats)))
      ([] (when-let [acc @acc_] (@reduce-stats_ acc))))))

(defn accumulate-stats "Experimental, subject to change!"
  [stats-accumulator [profiled-result profiled-?stats]]
  (when profiled-?stats (stats-accumulator profiled-?stats))
  profiled-result)

(comment
  (enc/qb 1e5 (stats-accumulator)) ; 5.87
  (let [sacc (stats-accumulator)]
    (accumulate-stats sacc (profiled {} (p :p1)))
    (accumulate-stats sacc (profiled {} (p :p2)))
    (sacc)))

     
                 
                                                                           
                                                                             

(comment (refer-tufte))

;;;; Stats formatting

(defn- perc [n d] (Math/round (* (/ (double n) (double d)) 100.0)))
(comment (perc 14 24))

(defn- ft [nanosecs]
  (let [ns (long nanosecs)] ; Truncate any fractionals
    (cond
      (>= ns 1000000000) (str (enc/round2 (/ ns 1000000000))  "s") ; 1e9
      (>= ns    1000000) (str (enc/round2 (/ ns    1000000)) "ms") ; 1e6
      (>= ns       1000) (str (enc/round2 (/ ns       1000)) "μs") ; 1e3
      :else              (str                ns              "ns"))))

(defn format-stats
  ([stats           ] (format-stats stats :time))
  ([stats sort-field]
   (when stats
     (let [^Stats stats stats
           ^Clock clock (.-clock        stats)
           m-id-stats   (.-id-stats-map stats)
           clock-total  (.-total clock)

           ^long accounted
           (reduce-kv (fn [^long acc k v] (+ acc ^long (:time v))) 0
             m-id-stats)

           sorted-ids
           (sort-by
             (fn [id] (get-in stats [id sort-field]))
             enc/rcompare
             (keys m-id-stats))

           ^long max-id-width
           (reduce-kv
             (fn [^long acc k v]
               (let [c (count (str k))]
                 (if (> c acc) c acc)))
             #=(count "Accounted Time")
             m-id-stats)]

             
       (let [sb
             (reduce
               (fn [acc id]
                 (let [^IdStats id-stats (get m-id-stats id)
                       time (.-time id-stats)]
                   (enc/sb-append acc
                     (str
                       {:id      id
                        :n-calls     (.-count id-stats)
                        :min     (ft (.-min   id-stats))
                        :max     (ft (.-max   id-stats))
                        :mad     (ft (.-mad   id-stats))
                        :mean    (ft (.-mean  id-stats))
                        :time%   (perc time clock-total)
                        :time    (ft   time)}
                       "\n"))))
               (enc/str-builder)
               sorted-ids)]

         (enc/sb-append sb "\n")
         (enc/sb-append sb (str "Clock Time: (100%) " (ft clock-total) "\n"))
         (enc/sb-append sb (str "Accounted Time: (" (perc accounted clock-total) "%) " (ft accounted) "\n"))
         (str           sb))

            
                                                                                  
                                                                                  
               
                    
                           
                                                            
                                              
                                     
                                       
                                             
                                              
                                              
                                              
                                              
                                              
                                      

                                                                                                          
                           

                                                                                                   
                                                                                                                          
                  ))))

;;;; fnp stuff

(defn- fn-sigs [def? fn-name sigs]
  (let [single-arity? (vector? (first sigs))
        sigs    (if single-arity? (list sigs) sigs)
        prepend (if def? "defn_" "fn_")
        get-id  (if single-arity?
                  (fn [fn-name _params] (keyword (str *ns*) (str prepend (name fn-name))))
                  (fn [fn-name  params] (keyword (str *ns*) (str prepend (name fn-name) \_ (count params)))))
        new-sigs
        (map
          (fn [[params & others]]
            (let [has-prepost-map?      (and (map? (first others)) (next others))
                  [?prepost-map & body] (if has-prepost-map? others (cons nil others))]
              (if ?prepost-map
                `(~params ~?prepost-map (p ~(get-id fn-name params) ~@body))
                `(~params               (p ~(get-id fn-name params) ~@body)))))
          sigs)]
    new-sigs))

                                                             
                                                   
                                                       
          
                                                                                         
                                                                            
                
                                
                                   

(comment
  (fn-sigs "foo"       '([x]            (* x x)))
  (macroexpand '(fnp     [x]            (* x x)))
  (macroexpand '(fn      [x]            (* x x)))
  (macroexpand '(fnp bob [x] {:pre [x]} (* x x)))
  (macroexpand '(fn      [x] {:pre [x]} (* x x))))

                                                                 
            
                                                              
                                                                            
          
                                                                     
                                                   
                                 

(comment
  (defnp foo "Docstring"                [x]   (* x x))
  (macroexpand '(defnp foo "Docstring"  [x]   (* x x)))
  (macroexpand '(defn  foo "Docstring"  [x]   (* x x)))
  (macroexpand '(defnp foo "Docstring" ([x]   (* x x))
                                       ([x y] (* x y))))
  (profiled {} (foo 5)))

;;;;

(comment
  (set-basic-println-handler!)
  (defn sleepy-threads []
    (dotimes [n 5]
      (Thread/sleep 100) ; Unaccounted
      (p :future/outer @(future (Thread/sleep 500)))
      @(future (p :future/inner (Thread/sleep 500)))
      (p :1ms  (Thread/sleep 1))
      (p :2s   (Thread/sleep 2000))
      (p :50ms (Thread/sleep 50))
      (p :rand (Thread/sleep (if (> 0.5 (rand)) 10 500)))
      (p :10ms (Thread/sleep 10))
      "Result"))

  (profile {:level 2 :id ::sleepy-threads} (sleepy-threads))
  (profile {:level 2 :id ::sleepy-thread :dynamic? true
            :data "foo"}
    (sleepy-threads))

  (p :hello "Hello, this is a result") ; Falls through (no data context)

  (defnp arithmetic
    []
    (let [nums (vec (range 1000))]
      (+ (p :fast-sleep (Thread/sleep 1) 10)
         (p :slow-sleep (Thread/sleep 2) 32)
         (p :add  (reduce + nums))
         (p :sub  (reduce - nums))
         (p :mult (reduce * nums))
         (p :div  (reduce / nums)))))

  (profile  {} (dotimes [n 100] (arithmetic)))
  (profile  {} (dotimes [n 1e5] (p :p1 nil))) ; 29.37ms
  (profile  {} (dotimes [n 1e6] (p :p1 nil))) ; 181.65ms
  (profiled {} (dotimes [n 1e6] (p :p1 nil)))
  (profiled {:level 2 :when (chance 0.5)} "body"))

;;;;;;;;;;;; This file autogenerated from src/taoensso/tufte.cljx
