<a href="https://www.taoensso.com" title="More stuff by @ptaoussanis at www.taoensso.com">
<img src="https://www.taoensso.com/taoensso-open-source.png" alt="Taoensso open-source" width="400"/></a>

**[CHANGELOG]** | [API] | current [Break Version]:

```clojure
[com.taoensso/tufte "1.0.0-SNAPSHOT"] ; Pre-alpha, subject to change
```

> Please consider helping to [support my continued open-source Clojure/Script work]? 
> 
> Even small contributions can add up + make a big difference to help sustain my time writing, maintaining, and supporting Tufte and other Clojure/Script libraries. **Thank you!**
>
> \- Peter Taoussanis

# Tufte

### Simple profiling and performance monitoring for Clojure/Script

![Hero]

> Charles Joseph Minard's _Carte Figurative_, one of [Edward Tufte][]'s favourite examples of good data visualization.

## 10-second example

```clojure
(taoensso.tufte/refer-tufte) ; Setup Tufte's ns imports (works with clj only)
(taoensso.tufte/set-basic-println-handler!) ; Send `profile` stats to `println`

;;; Let's define a couple dummy fns to simulate doing some expensive work
(defn get-x [] (Thread/sleep 500)             "x val")
(defn get-y [] (Thread/sleep (rand-int 1000)) "y val")

;; How do these fns perform? Let's check:

(profile ; Activate profiling w/in body
  {} ; Profiling options; we'll use the default for now
  (dotimes [_ 5]
    (p :get-x (get-x))
    (p :get-y (get-y))))

;; The following will be printed to *out*:
;;
;;            pId      nCalls       Min        Max       MAD      Mean   Time% Time
;;         :get-x           5   502.3ms   505.07ms    1.17ms   503.4ms      48 2.52s
;;         :get-y           5  171.68ms   940.84ms  264.82ms  541.28ms      52 2.71s
;;     Clock Time                                                          100 5.22s
;; Accounted Time                                                          100 5.22s
```

## Features

 * Small, **fast**, cross-platform Clojure/Script codebase (<1k loc)
 * **Tiny**, flexible API: `p`, `profiled`, `profile`
 * Great **compile-time elision** and **runtime filtering** support
 * Arbitrary Clojure/Script **form-level** profiling
 * Easily capture **raw stats** for further analysis
 * Full support for **thread-local** and **multi-threaded** profiling

## Quickstart

Add the necessary dependency to your project:

```clojure
[com.taoensso/tufte "1.0.0-SNAPSHOT"]
```

And setup your namespace imports:

```clojure
(ns my-clj-ns ; Clojure namespace
  (:require [taoensso.tufte :as tufte :refer (defnp p profile profiled)]))

(ns my-cljs-ns ; ClojureScript namespace
  (:require [taoensso.tufte :as tufte :refer-macros (defnp p profile profiled)]))
```

### Step 1: identify forms you'd like to [sometimes] profile

Just wrap them with `p` and give them an id (namespaced keyword):

```clojure
(defn get-customer-info []
  (let [raw-customer-map (p ::get-raw-customer (fetch-from-db))]
    (p ::enrich-raw-customer
      (do-some-work raw-customer-map))))
```

Tufte will record the execution times of these `p` forms whenever profiling is active. 

Whether or not profiling is active, a `p` form **always return's its normal body result**.

> This last point is important: you never need to worry about Tufte messing with your return values.

### Step 2: activate profiling of `p` forms

Activate profiling with `profile` or `profiled`:

API        | Return value               | Effect                          |
---------- | -------------------------- | --------------------------------|
`profile`  | `<body-result>`            | Sends `<?stats>` to handlers[1] |
`profiled` | `[<body-result> <?stats>]` | None                            |

**[1]** You can register handlers using `tufte/set-handler!`

> Handler ideas: save to a db, log, `put!` to an appropriate `core.async` channel, filter, aggregate, use for a realtime analytics dashboard, examine for outliers or unexpected output, ...

 * Use `profiled` to handle stats yourself directly at the callsite.
 * Use `profile` to handle stats later/elsewhere.

Between the two, you have great flexibility for a wide range of use cases in production and during development/debugging.

## Conditional profiling

Tufte offers extensive facilities to control if and when profiling happens. Both **compile-time elision** and **runtime filtering** are supported.

### Profiling levels

Every `p`, `profile`, and `profiled` form can take an optional **profiling level** ∈ `#{0 1 2 3 4 5}`.

This level must be >= `tufte/*min-level*` for profiling to occur.

For example:

```clojure
(profiled {:level 3} ...) ; Only activates profiling when (>= 3 *min-level)
```

Min level    | Set with                                       |
------------ | ---------------------------------------------- |
Runtime      | `tufte/set-min-level!`, `tufte/with-min-level` |
Compile-time | `TUFTE_MIN_LEVEL` environment variable         |

> Note that runtime filtering stacks with any compile-time elision

### Namespace filtering

Likewise- `p`, `profile`, and `profiled` forms can be elided or filtered by the namespace in which they occur.

Namespace filter | Set with                                         |
---------------- | ------------------------------------------------ |
Runtime          | `tufte/set-ns-pattern!`, `tufte-with-ns-pattern` |
Compile-time     | `TUFTE_NS_PATTERN` environment variable          |

Some example namespace patterns:

```clojure
"foo.bar.baz"
"foo.bar.*"
#{"foo.bar.*" "some.lib.*"}
{:whitelist #{"foo.bar.*"} :blacklist #{"noisy.lib.*"}}
```

### Arbitrary runtime conditions

Finally, `profile` and `profiled` both support an optional arbitrary test expression:

```clojure
(profiled {:when my-cond?} ...) ; Only activates profiling when `my-cond?` is truthy
```

This can be used for a wide range of sophisticated behaviour including smart, **application-aware rate limiting**.

As one simpler example, we can get **sampled profiling** like this:

```clojure
(profiled {:when (tufte/chance 0.5)} ...) ; 50% chance of profiling
```

## FAQ

### Why not just use [YourKit], [JProfiler], or [VisualVM]?

The traditional recommendation for Clojure profiling has usually been to use a standard JVM profiling tool like one of the above.

And they can certainly do the job, but they also tend to be a little hairy: requiring special effort to use, and often producing gobs of information that can be difficult or time-consuming to meaningfully interpret.

In contrast, Tufte offers some interesting benefits:

 * Arbitrary **form-level** profiling; measure [just] what you care about
 * Simple **thread-local or multi-threaded semantics**
 * During dev/debug: check performance **right from within your REPL**
 * During production: **ongoing, application-aware** conditional profiling, logging, and analysis (stats are just **Clojure data**)
 * A **cross-platform API** that works seamlessly between your server (Clj) and client (Cljs) applications


My general preference is to use JVM tools only for very specific forensic profiling when something's producing unexpected performance numbers. In other cases, Tufte's often a better fit IMO.

### How does Tufte compare to the profiling in [Timbre]?

Actually, I developed Tuft one weekend while refactoring Timbre's profiling. It's basically a refinement of the ideas from there.

Decided that I could make some worthwhile improvements with some breaking API changes and a new set of dedicated docs. Tufte's implementation is considerably faster, and its API considerably more flexible.

With the release of Tuft, I'll be **deprecating Timbre's profiling features**.

Note that Tuft's a feature superset of Timbre's profiling, so in most cases porting should be pretty straightforward.

### What's the difference between thread-local and dynamic (multi-threaded) profiling?

If you don't already know the difference, you probably want **thread-local profiling**. It's faster and conceptually simpler: it literally just profiles what happens sequentially on the current thread.

Work being done concurrently in futures and agents will be ignored.

In contrast, **dynamic profiling** works across thread boundaries using Clojure's standard `^:dynamic` binding conveyance. For example:

```clojure
(def ^:dynamic *my-dynamic-var* nil)
(binding [*my-dynamic-var* "foobar!"] ; This val will be available across Clojure threads
  (future (println [:thread1 *my-dynamic-var*]))
  (future (println [:thread2 *my-dynamic-var*]))
  (println [:thread3 *my-dynamic-var*])
  (Thread/sleep 100))
```

### How do I get dynamic (multi-threaded) profiling?

`profile` and `profiled` have a `:dynamic?` option:

```clojure
(profiled {:dynamic? true} ...) ; Activates dynamic (multi-threaded) profiling
```

This works through Clojure's standard `^:dynamic` binding conveyance.

If you really want to get fancy, you can also do _manual_ multi-threaded profiling using `tufte/stats-accumulator`.

## Contacting me / contributions

Please use the project's [GitHub issues page] for all questions, ideas, etc. **Pull requests welcome**. See the project's [GitHub contributors page] for a list of contributors.

Otherwise, you can reach me at [Taoensso.com]. Happy hacking!

\- [Peter Taoussanis]

## License

Distributed under the [EPL v1.0] \(same as Clojure).  
Copyright &copy; 2016 [Peter Taoussanis].

<!--- Standard links -->
[Taoensso.com]: https://www.taoensso.com
[Peter Taoussanis]: https://www.taoensso.com
[@ptaoussanis]: https://www.taoensso.com
[More by @ptaoussanis]: https://www.taoensso.com
[Break Version]: https://github.com/ptaoussanis/encore/blob/master/BREAK-VERSIONING.md
[support my continued open-source Clojure/Script work]: http://taoensso.com/clojure/backers

<!--- Standard links (repo specific) -->
[CHANGELOG]: https://github.com/ptaoussanis/tufte/releases
[API]: http://ptaoussanis.github.io/tufte/
[GitHub issues page]: https://github.com/ptaoussanis/tufte/issues
[GitHub contributors page]: https://github.com/ptaoussanis/tufte/graphs/contributors
[EPL v1.0]: https://raw.githubusercontent.com/ptaoussanis/tufte/master/LICENSE
[Hero]: https://raw.githubusercontent.com/ptaoussanis/tufte/master/hero.png "Title"

<!--- Unique links -->
[Edward Tufte]: https://en.wikipedia.org/wiki/Edward_Tufte
[YourKit]: http://www.yourkit.com/)
[JProfiler]: http://www.ej-technologies.com/products/jprofiler/overview.html
[VisualVM]: http://docs.oracle.com/javase/6/docs/technotes/guides/visualvm/index.html
[Timbre]: https://github.com/ptaoussanis/timbre
