(ns taoensso.truss.impl
  "Private implementation details"
                                         
         (:require [clojure.set :as set])
         (:require-macros
          [taoensso.truss.impl :as impl-macros
           :refer (catch-errors* -invariant1)]))

(comment (require '[taoensso.encore :as enc :refer (qb)]))

;;;; Manual Encore imports
;; A bit of a nuisance but:
;;   * Allows Encore to depend on Truss (esp. nb for back-compatibility wrappers)
;;   * Allows Truss to be entirely dependency free

                                                        
                       
                                                                                  
                                  
            
                                                              
                                                                
                                               
            
                                                                                      
                                                                                         

(defn rsome   [pred coll] (reduce (fn [acc in] (when-let [p (pred in)] (reduced p))) nil coll))
(defn revery? [pred coll] (reduce (fn [acc in] (if (pred in) true (reduced nil))) true coll))

(defn set*     [x] (if (set? x) x (set x)))
(defn ks=      [ks m] (=             (set (keys m)) (set* ks)))
(defn ks<=     [ks m] (set/subset?   (set (keys m)) (set* ks)))
(defn ks>=     [ks m] (set/superset? (set (keys m)) (set* ks)))
(defn ks-nnil? [ks m] (revery? #(not (nil? (get m %)))    ks))

(defn now-dt []                                (js/Date.))

;;;; Truss

(def ^:dynamic *-?data* nil)

(defn non-throwing [pred] (fn [x] (catch-errors* (pred x) _ nil)))

(def -invar-pred
  "Predicate shorthand transformations for convenience"
  (fn self [pred-form]
    (if-not (vector? pred-form)
      pred-form
      (let [[type a1 a2 a3] pred-form]
        (assert a1 "Special predicate [<special-type> <arg>] form w/o <arg>")
        (case type
          :set=             `(fn [~'__x] (=             (set* ~'__x) (set* ~a1)))
          :set<=            `(fn [~'__x] (set/subset?   (set* ~'__x) (set* ~a1)))
          :set>=            `(fn [~'__x] (set/superset? (set* ~'__x) (set* ~a1)))
          :ks=              `(fn [~'__x] (ks=      ~a1 ~'__x))
          :ks<=             `(fn [~'__x] (ks<=     ~a1 ~'__x))
          :ks>=             `(fn [~'__x] (ks>=     ~a1 ~'__x))
          :ks-nnil?         `(fn [~'__x] (ks-nnil? ~a1 ~'__x))
          (    :el     :in) `(fn [~'__x]      (contains? (set* ~a1) ~'__x))
          (:not-el :not-in) `(fn [~'__x] (not (contains? (set* ~a1) ~'__x)))

          :and ; all-of, (apply every-pred preds)
          (cond
            a3 `(fn [~'__x] (and (~(self a1) ~'__x) (~(self a2) ~'__x) (~(self a3) ~'__x)))
            a2 `(fn [~'__x] (and (~(self a1) ~'__x) (~(self a2) ~'__x)))
            a1 (self a1))

          :or  ; any-of, (apply some-fn preds)
          (cond
            a3 `(fn [~'__x] (or ((non-throwing ~(self a1)) ~'__x) ((non-throwing ~(self a2)) ~'__x) (~(self a3) ~'__x)))
            a2 (if (or (= a1 nil?) (= a2 nil?)) ; Common case
                 `(fn [~'__x] (or (~(self a1) ~'__x) (~(self a2) ~'__x)))
                 `(fn [~'__x] (or ((non-throwing ~(self a1)) ~'__x) (~(self a2) ~'__x))))
            a1 (self a1))

          :not ; complement/none-of
          (cond
            a3 `(fn [~'__x] (and (not (~(self a1) ~'__x)) (not (~(self a2) ~'__x)) (not (~(self a3) ~'__x))))
            a2 `(fn [~'__x] (and (not (~(self a1) ~'__x)) (not (~(self a2) ~'__x))))
            a1 `(fn [~'__x] (not ~(self a1)))))))))

(comment
                                         
  (macroexpand '(ep [:and [:or string? integer?] #(not= % "bad")]))

  ((ep [:and [:or string? integer?] #(not= % "bad")]) "bad")
  ((ep [:or nil? string?]) "foo")
  ((ep [:or [:and integer? neg?] string?]) 5)
  ((ep [:or zero? nil?]) nil) ; (zero? nil) throws

  (qb 10000
    (#(or (string? %) (integer? %) (number? %)) 20.3)
    ((ep [:or string? integer? number?]) 20.3)
    (#(or (nil? %) (string? %)) "foo")
    ((ep [:or nil? string?]) "foo")))

(defn -assertion-error [msg]                                    (js/Error. msg))
(def  -invar-undefined-val :invariant/undefined-val)
(defn -invar-violation!
  ;; * http://dev.clojure.org/jira/browse/CLJ-865 would be handy for line numbers
  ;; * Clojure 1.7+'s `pr-str` dumps a ton of error info that we don't want here
  ([] (throw (ex-info "Invariant violation" {:invariant-violation? true})))
  ([assertion? ns-str ?line form val ?err ?data-fn]
   (let [fmt-msg
         (fn [x1 x2 x3 x4]
           ;; Cider unfortunately doesn't seem to print newlines in errors
           (str "Invariant violation in `" x1 ":" x2 "` [pred-form, val]:"
                "\n [" x3 ", " x4 "]"))

         line-str    (or ?line "?")
         form-str    (str form)
         undefn-val? (= val -invar-undefined-val)
         val-str     (if undefn-val? "<undefined>" (str (or val "<nil>")) #_(pr-str val))
         dummy-err?  (:invariant-violation? (ex-data ?err))
         ?err        (when-not dummy-err? ?err)
         ?err-str    (when-let [e ?err] (str ?err) #_(pr-str ?err))
         msg         (let [msg (fmt-msg ns-str line-str form-str val-str)]
                       (cond
                         (not ?err)       msg
                         undefn-val? (str msg       "\n`val` error: " ?err-str)
                         :else       (str msg "\n`pred-form` error: " ?err-str)))
         ?data       (when-let [data-fn ?data-fn]
                       (catch-errors* (data-fn) e {:data-error e}))]

     (throw
       ;; Vestigial, we now prefer to just always throw `ex-info`s:
       ;; (if assertion? (-assertion-error msg) (ex-info msg {}))
       (ex-info msg
         {:dt       (now-dt)
          :ns-str   ns-str
          :?line    ?line
          ;; :?form (when-not (string? form) form)
          :form-str form-str
          :val      (if undefn-val? 'undefined/threw-error val)
          :val-type (if undefn-val? 'undefined/threw-error (type val))
          :?data      ?data  ; Arbitrary user data, handy for debugging
          :*?data*  *-?data* ; ''
          :?err     ?err
          :*assert* *assert*
          :elidable? assertion?})))))

                     
                                                                               
                                           
                                 
                                                                   
                                      
                                           

                                                          
                  
                                    
                                                                      
                                                        

                       
                                                               
                                                                        
                                           

                                                                              
                      
                       
                                                               
                                                                        
                                              

(comment
  (macroexpand '(-invariant1 true false 1    #(string? %) "foo" nil))
  (macroexpand '(-invariant1 true false 1      string?    "foo" nil))
  (macroexpand '(-invariant1 true false 1 [:or string?]   "foo" nil))
  (qb 100000
    (string? "foo")
    (-invariant1 true false 1 string? "foo" nil) ; ~1.2x cheapest possible pred cost
    (-invariant1 true false 1 string? (str "foo" "bar") nil) ; ~3.5x ''
    )

  (-invariant1 false false 1 integer? "foo"   nil) ; Pred failure example
  (-invariant1 false false 1 zero?    "foo"   nil) ; Pred error example
  (-invariant1 false false 1 zero?    (/ 5 0) nil) ; Form error example
  )

                                                     
                                                                             
                                               
                                                  
                                              
                                                                      
                                                                 

                                                                          
                                                        
                                                     
                                                           

                                                                     
                                                                      
                                 
                                       
                                                                             
                             
                                                                            

              
                                  

                 

                     
                               
                                                                        

                                                 
                                                                                           

                     
                                      
                                        
                   
                                                              
                                                                                     

                                                        
                                                        
               
                    
                       
                                                                                              
                      
                     

;;;;;;;;;;;; This file autogenerated from src/taoensso/truss/impl.cljx
