(ns taoensso.timbre.profiling
  "Simple logging profiler for Timbre. Highly optimized; supports
  sampled profiling in production."
  {:author "Peter Taoussanis (@ptaoussanis)"}
  (:require [taoensso.encore :as enc]
            [taoensso.timbre :as timbre])
                                                  )

(comment (require '[taoensso.encore :as enc :refer (qb)]))

;;;; TODO
;; * Support for explicit `config` args?
;; * Support for real level+ns based elision (zero *pdata* check cost, etc.)?
;;   - E.g. perhaps `p` forms could take a logging level?
;;   - Less important if we've got static calls to `-pdata-proxy`.

;;;; Utils

;; Note that we only support *compile-time* ids
(defn- qualified-kw [ns id] (if (enc/qualified-keyword? id) id (keyword (str ns) (name id))))
(comment (qualified-kw *ns* "foo"))

     
                               
                                                                          
                                                      
                                              

;;;;

;; This is substantially faster than a ^:dynamic atom.
(def -pdata-proxy ; Would benefit from ^:static / direct linking / Java class
  "{<id> <times> :__stats <m-stats>} iff profiling active on thread"

       
                                                    
       
                                                           
                                                

        
  (let [state_ (volatile! false)] ; Automatically thread-local in js
    (fn
      ([]                @state_)
      ([new-val] (vreset! state_ new-val)))))

(comment
  (def ^:dynamic *foo* nil)
  (let [^ThreadLocal proxy (proxy [ThreadLocal] [])]
    (qb 1e5
      (if (-pdata-proxy)   true false)
      (if (identity false) true false)
      (if (.get proxy)     true false) ; w/o var indirection cost
      (if false            true false)
      (if *foo*            true false))))

(do
  (defn- add-time    [                           ^ArrayList x t] (.add   x t))
  (defn- count-times [                           ^ArrayList x]   (.size  x))
  (defn- clear-times [                           ^ArrayList x]   (.clear x))
  (defn- new-times []                            (array-list)))

(declare ^:private times->stats)
(defn -capture-time!
  ([      id t-elapsed] (-capture-time! (-pdata-proxy) id t-elapsed)) ; Dev
  ([pdata id t-elapsed] ; Common case
   (if-let [times (get pdata id)]
     (if (>= (long (count-times times)) #_20 2000000) ; Rare in real-world use
       ;; Compact: merge interim stats to help prevent OOMs
       (let [m-stats (get pdata :__stats)
             m-stats (assoc m-stats id (times->stats times (get m-stats id)))]

         (clear-times times)
         (add-time    times t-elapsed) ; Nb: never leave our accumulator empty
         (-pdata-proxy (assoc pdata id times :__stats m-stats)))

       ;; Common case
       (add-time times t-elapsed))

     ;; Init case
     (let [times (new-times)]
       (add-time times t-elapsed)
       (-pdata-proxy (assoc pdata id times))))

   nil))

(comment
                                
                                                                      

  (-with-pdata (qb 1e6 (-capture-time! :foo 1000))) ; 65.84
  (-with-pdata
   (dotimes [_ 20] (-capture-time! :foo 100000))
   (-pdata-proxy)))

(def ^:private ^:const max-long                             9223372036854775807)

(defn- times->stats [times ?base-stats]
  (let [ts-count     (long (count-times times))
        _            (assert (not (zero? ts-count)))
        times        (into [] times) ; Faster to reduce
        ts-time      (reduce (fn [^long acc ^long in] (+ acc in)) 0 times)
        ts-mean      (/ (double ts-time) (double ts-count))
        ts-mad-sum   (reduce (fn [^long acc ^long in] (+ acc (Math/abs (- in ts-mean)))) 0 times)
        ts-min       (reduce (fn [^long acc ^long in] (if (< in acc) in acc)) max-long     times)
        ts-max       (reduce (fn [^long acc ^long in] (if (> in acc) in acc)) 0            times)]

    (if-let [stats ?base-stats] ; Merge over previous stats
      (let [s-count   (+ ^long (get stats :count) ts-count)
            s-time    (+ ^long (get stats :time)  ts-time)
            s-mean    (/ (double s-time) (double s-count))
            s-mad-sum (+ ^long (get stats :mad-sum) ts-mad-sum)
            s-mad     (/ (double s-mad-sum) (double s-count))
            s0-min    (get stats :min)
            s0-max    (get stats :max)]

        ;; Batched "online" MAD calculation here is >= the standard
        ;; Knuth/Welford method, Ref. http://goo.gl/QLSfOc,
        ;;                            http://goo.gl/mx5eSK.

        {:count   s-count
         :time    s-time
         :mean    s-mean
         :mad-sum s-mad-sum
         :mad     s-mad
         :min     (if (< ^long s0-min ^long ts-min) s0-min ts-min)
         :max     (if (> ^long s0-max ^long ts-max) s0-max ts-max)})

      {:count   ts-count
       :time    ts-time
       :mean    ts-mean
       :mad-sum ts-mad-sum
       :mad     (/ (double ts-mad-sum) (double ts-count))
       :min     ts-min
       :max     ts-max})))

(comment (times->stats (new-times) nil))

(defn -compile-final-stats! "Returns {<id> <stats>}"
  [clock-time]
  (let [pdata   (-pdata-proxy) ; Nb must be fresh
        m-stats (get    pdata :__stats)
        m-times (dissoc pdata :__stats)]
    (reduce-kv
      (fn [m id times]
        (assoc m id (times->stats times (get m-stats id))))
      {:clock-time clock-time} m-times)))

(comment
  (qb 1e5
    (-with-pdata
     (-capture-time! :foo 10)
     (-capture-time! :foo 20)
     (-capture-time! :foo 30)
     (-capture-time! :foo 10)
     (-compile-final-stats! 0))) ; 114
  )

;;;;

(defn- perc [n d] (Math/round (/ (double n) (double d) 0.01)))
(comment (perc 14 24))

(defn- ft [nanosecs]
  (let [ns (long nanosecs)] ; Truncate any fractionals
    (cond
      (>= ns 1000000000) (str (enc/round2 (/ ns 1000000000))  "s") ; 1e9
      (>= ns    1000000) (str (enc/round2 (/ ns    1000000)) "ms") ; 1e6
      (>= ns       1000) (str (enc/round2 (/ ns       1000)) "μs") ; 1e3
      :else              (str                ns              "ns"))))

(defn -format-stats
  ([stats           ] (-format-stats stats :time))
  ([stats sort-field]
   (let [clock-time      (get    stats :clock-time)
         stats           (dissoc stats :clock-time)
         ^long accounted (reduce-kv (fn [^long acc k v] (+ acc ^long (:time v))) 0 stats)

         sorted-stat-ids
         (sort-by
           (fn [id] (get-in stats [id sort-field]))
           enc/rcompare
           (keys stats))

         ^long max-id-width
         (reduce-kv
          (fn [^long acc k v]
            (let [c (count (str k))]
              (if (> c acc) c acc)))
          #=(count "Accounted Time")
          stats)]

           
     (let [sb
           (reduce
             (fn [acc id]
               (let [{:keys [count min max mean mad time]} (get stats id)]
                 (enc/sb-append acc
                   (str
                     {:id      id
                      :n-calls count
                      :min     (ft min)
                      :max     (ft max)
                      :mad     (ft mad)
                      :mean    (ft mean)
                      :time%   (perc time clock-time)
                      :time    (ft time)}
                     "\n"))))
             (enc/str-builder)
             sorted-stat-ids)]

       (enc/sb-append sb "\n")
       (enc/sb-append sb (str "Clock Time: (100%) " (ft clock-time) "\n"))
       (enc/sb-append sb (str "Accounted Time: (" (perc accounted clock-time) "%) " (ft accounted) "\n"))
       (str           sb))

          
                                                                                
                                                                                
             
                  
                         
                                                                          
                                   
                                                                      
                                                                   

                                                                                                       
                              

                                                                                                
                                                                                                                       
                )))

;;;;

              
                                                               
                                                                  
             
                                  
                        
                  
                                   
                   
                                      
                                    
                                       
                                                    
                     
                           

                                            ; Alias

(comment (macroexpand '(p :foo (+ 4 2))))

                  
                                   
                                                                     
                                                                          
                                                                  
                                 
                              
                               
                              
        
                       
                                 
                                    
                                 
                                                       
                    
                                     

(comment (profiled (p :foo "foo") [stats result] [stats result]))

                 
                                                                           
                                                                 

                                                                    
          

                       
                                                                 

                       
                                                                    

                   
                                  
                        
                  
                                                           
                                               
                                                   
                                   
                                                  
                           
                                          
                                                 
                      
                        

(comment (profile :info :foo "foo"))

                          
                                                                      
                               
                                                      
                      
                
                                
                                  
                                     

;;;; fnp stuff

(defn -fn-sigs [fn-name sigs]
  (let [single-arity? (vector? (first sigs))
        sigs    (if single-arity? (list sigs) sigs)
        get-id  (if single-arity?
                  (fn [fn-name _params]      (name fn-name))
                  (fn [fn-name  params] (str (name fn-name) \_ (count params))))
        new-sigs
        (map
          (fn [[params & others]]
            (let [has-prepost-map?      (and (map? (first others)) (next others))
                  [?prepost-map & body] (if has-prepost-map? others (cons nil others))]
              (if ?prepost-map
                `(~params ~?prepost-map (pspy ~(get-id fn-name params) ~@body))
                `(~params               (pspy ~(get-id fn-name params) ~@body)))))
          sigs)]
    new-sigs))

                                                             
                                                   
                                                       
          
                                                                                         
                                                                    
                
                                
                                   

(comment
  (-fn-sigs "foo"      '([x]            (* x x)))
  (macroexpand '(fnp     [x]            (* x x)))
  (macroexpand '(fn       [x]            (* x x)))
  (macroexpand '(fnp bob [x] {:pre [x]} (* x x)))
  (macroexpand '(fn       [x] {:pre [x]} (* x x))))

                                                                 
            
                                                              
                                                                            
          
                                                                     
                                               
                                 

(comment
  (defnp foo "Docstring"                [x]   (* x x))
  (macroexpand '(defnp foo "Docstring"  [x]   (* x x)))
  (macroexpand '(defn  foo "Docstring"  [x]   (* x x)))
  (macroexpand '(defnp foo "Docstring" ([x]   (* x x))
                                       ([x y] (* x y))))
  (profile :info :defnp-test (foo 5)))

;;;; Deprecated

                                                                 
                                                       

(comment (profile :info :pspy* (pspy* :foo (fn [] (Thread/sleep 100)))))

;;;;

(comment
  (profile :info :sleepy-threads
    (dotimes [n 5]
      (Thread/sleep 100) ; Unaccounted
      (p :future/outer @(future (Thread/sleep 500)))
      @(future (p :future/inner (Thread/sleep 500)))
      (p :1ms    (Thread/sleep 1))
      (p :2s     (Thread/sleep 2000))
      (p :50ms   (Thread/sleep 50))
      (p :rand   (Thread/sleep (if (> 0.5 (rand)) 10 500)))
      (p :10ms   (Thread/sleep 10))
      "Result"))

  (p :hello "Hello, this is a result") ; Falls through (no thread context)

  (defnp my-fn
    []
    (let [nums (vec (range 1000))]
      (+ (p :fast-sleep (Thread/sleep 1) 10)
         (p :slow-sleep (Thread/sleep 2) 32)
         (p :add  (reduce + nums))
         (p :sub  (reduce - nums))
         (p :mult (reduce * nums))
         (p :div  (reduce / nums)))))

  (profile :info :Arithmetic (dotimes [n 100] (my-fn)))
  (profile :info :high-n     (dotimes [n 1e5] (p :nil nil))) ; ~20ms
  (profile :info :high-n     (dotimes [n 1e6] (p :nil nil))) ; ~116ms
  (profiled (dotimes [n 1e6] (p :nil nil)) [stats result] [stats result])
  (sampling-profile :info 0.5 :sampling-test (p :string "Hello!")))

;;;;;;;;;;;; This file autogenerated from src/taoensso/timbre/profiling.cljx
