(ns taoensso.timbre.profiling
  "Simple logging profiler for Timbre, originally adapted from
  clojure.contrib.profile."
  {:author "Peter Taoussanis (@ptaoussanis)"}
  (:require [taoensso.encore :as enc :refer (qb)]
            [taoensso.timbre :as timbre]))

;;;; TODO
;; * Support for real level+ns based elision (zero *pdata* check cost, etc.)?
;;   - E.g. perhaps `p` forms could take a logging level?
;; * Support for explicit `config` args?
;; * Possible porting to .cljx (any point?)

;;;; Utils

;; TODO Temp, until we have a better elision strategy
(def ^:private elide-profiling? (enc/read-sys-val "TIMBRE_ELIDE_PROFILING"))

(defn- qualified-kw [ns id] (if (enc/qualified-keyword? id) id (keyword (str ns) (name id))))
(comment (qualified-kw *ns* "foo"))

;;;;

(declare -capture-time! -compile-stats! ^:private format-stats)

(def ^:dynamic *pdata_* "{<id> <times>}" nil)
(defn      -new-pdata_ [] (enc/-vol! (transient {})))

(defmacro pspy
  "Profile spy. When in the context of a *pdata_* binding, records execution
  time of named body. Always returns the body's result."
  ;; Note: do NOT implement as `(pspy* ~id (fn [] ~@body))`. The fn wrapping
  ;; can cause unnecessary lazy seq head retention, Ref. http://goo.gl/42Vxph.
  [id & body]
  (let [id (qualified-kw *ns* id)]
    (if elide-profiling?
      `(do ~@body)
      `(if *pdata_*
         (let [t0#     (System/nanoTime)
               result# (do ~@body)
               t1#     (System/nanoTime)]
           (-capture-time! ~id (- t1# t0#))
           result#)
         (do ~@body)))))

(defmacro p [id & body] `(pspy ~id ~@body)) ; Alias

(comment
  (macroexpand '(p :foo (+ 4 2)))
  (qb 1e6 nil *pdata_* (pspy :foo)))

(defmacro with-pdata_ "Returns [result stats]"
  [level & body]
  `(if (timbre/log? ~level ~(str *ns*)) ; Runtime check
     (binding [*pdata_* (-new-pdata_)]
       (let [t0#     (System/nanoTime)
             result# (do ~@body)
             t1#     (System/nanoTime)]
         [result# (-compile-stats! (- t1# t0#))]))
     [(do~@body) nil]))

(defmacro profile
  "When logging is enabled, executes named body with profiling enabled.
  Body forms wrapped by `pspy` will be timed and time stats logged. Always
  returns body's result.

  Note that logging appenders will receive both a formatted profiling string
  AND the raw profiling stats under a special :profiling-stats key (useful
  for queryable db logging)."
  [level id & body]
  (let [id (qualified-kw *ns* id)]
    (if elide-profiling?
      `(do ~@body)
      `(let [[result# stats#] (with-pdata_ ~level ~@body)]
         (when stats#
           (timbre/log! ~level :p
             ["Profiling: " ~id "\n" (format-stats stats#)]
             {:?base-data {:profile-stats stats#}}))
         result#))))

(defmacro sampling-profile
  "Like `profile`, but only enables profiling with given probability."
  [level probability id & body]
  (assert (<= 0 probability 1) "Probability: 0<=p<=1")
  (if elide-profiling?
    `(do ~@body)
    `(if (< (rand) ~probability)
       (profile ~level ~id ~@body)
       (do                 ~@body))))

;;;; Data capturing & aggregation

(def ^:private ^:const nmax-times
  "We limit the number of recorded times to help prevent OOMs. Not the most
  sophisticated approach, but much simpler and *faster* than alternatives
  like batched streaming stats, etc.

  We're trading temporary memory usage for simplicity and speed, a good trade
  for Timbre which tends to encourage keeping sampled profiling on in
  production. If you really need higher rep counts, you'll probably want a
  full-featured JVM profiler anyway.

  When truncation kicks in:
    1. \"nCalls\" will show as \"<n>+\" in output table.
    2. Total (clock) time will still be accurate, but accounted time will be
       capped."

  2000000)

(defn -capture-time! [id t-elapsed]
  (enc/-vol-swap! *pdata_*
    (fn [m]
      (let [times (get m id (transient []))]
        (if (== (count times) nmax-times)
          m ; noop, rare in most real-world cases
          ;; (assoc m id (cons t-elapsed times))
          (assoc! m id (conj! times t-elapsed))))))
  nil)

(comment
  (binding [*pdata_* (-new-pdata_)]
    (dotimes [_ 20] (-capture-time! :foo 100000))
    *pdata_*))

(defn- times->stats [times]
  (let [ts-count   (count times) ; >= 1
        ts-time    (reduce (fn [^long acc ^long in] (+ acc in)) times)
        ts-mean    (/ (double ts-time) (double ts-count))
        ts-mad-sum (reduce (fn [^long acc ^long in] (+ acc (Math/abs (- in ts-mean)))) 0 times)
        ts-mad     (/ (double ts-mad-sum) (double ts-count))
        ts-min     (reduce (fn [^long acc ^long in] (if (< in acc) in acc)) Long/MAX_VALUE times)
        ts-max     (reduce (fn [^long acc ^long in] (if (> in acc) in acc)) 0              times)]

    {:count ts-count
     :min   ts-min
     :max   ts-max
     :mean  ts-mean
     :mad   ts-mad
     :time  ts-time}))

(defn -compile-stats! "Returns {<id> <stats>}"
  [clock-time]
  (when-let [pdata_ *pdata_*]
    (let [-pdata @pdata_
          pdata  (persistent! -pdata)]

      (reduce-kv
       (fn [m id -times]
         (assoc m id (times->stats (persistent! -times))))
       {:clock-time clock-time} pdata))))

(comment
  (binding [*pdata_* (-new-pdata_)]
    (-capture-time! :foo 10)
    (-capture-time! :foo 20)
    (-capture-time! :foo 30)
    (-capture-time! :foo 10)
    (-compile-stats! 0)))

(defn- perc [n d] (Math/round (/ (double n) (double d) 0.01)))
(comment (perc 14 24))

(defn- ft [nanosecs]
  (let [ns (long nanosecs)] ; Truncate any fractionals
    (cond
      (>= ns 1000000000) (str (enc/round2 (/ ns 1000000000))  "s") ; 1e9
      (>= ns    1000000) (str (enc/round2 (/ ns    1000000)) "ms") ; 1e6
      (>= ns       1000) (str (enc/round2 (/ ns       1000)) "μs") ; 1e3
      :else              (str                ns              "ns"))))

(defn format-stats
  ([stats           ] (format-stats stats :time))
  ([stats sort-field]
   (let [;; How long entire profile body took:
         clock-time      (get    stats :clock-time)
         stats           (dissoc stats :clock-time)
         ^long accounted (reduce-kv (fn [^long acc k v] (+ acc ^long (:time v))) 0 stats)

         ^long max-id-width
         (reduce-kv
          (fn [^long acc k v]
            (let [c (count (str k))]
              (if (> c acc) c acc)))
          #=(count "Accounted Time")
          stats)

         pattern   (str "%" max-id-width "s %,11d%1s %9s %10s %9s %9s %7d %1s%n")
         s-pattern (str "%" max-id-width "s %12s %9s %10s %9s %9s %7s %1s%n")

         sorted-stat-ids
         (sort-by
          (fn [id] (get-in stats [id sort-field]))
          enc/rcompare
          (keys stats))]

     (with-out-str
       (printf s-pattern "Id" "nCalls" "Min" "Max" "MAD" "Mean" "Time%" "Time")
       (enc/run!
        (fn [id]
          (let [{:keys [count min max mean mad time]} (get stats id)
                nmax? (= ^long count nmax-times)]
            (printf pattern id count (if nmax? "+" " ") (ft min) (ft max) (ft mad)
                    (ft mean) (perc time clock-time) (ft time))))
        sorted-stat-ids)

       (printf s-pattern "Clock Time" "" "" "" "" "" 100 (ft clock-time))
       (printf s-pattern "Accounted Time" "" "" "" "" ""
               (perc accounted clock-time) (ft accounted))))))

;;;; fnp stuff

(defn -fn-sigs [fn-name sigs]
  (let [single-arity? (vector? (first sigs))
        sigs    (if single-arity? (list sigs) sigs)
        get-id  (if single-arity?
                  (fn [fn-name _params]      (name fn-name))
                  (fn [fn-name  params] (str (name fn-name) \_ (count params))))
        new-sigs
        (map
          (fn [[params & others]]
            (let [has-prepost-map?      (and (map? (first others)) (next others))
                  [?prepost-map & body] (if has-prepost-map? others (cons nil others))]
              (if ?prepost-map
                `(~params ~?prepost-map (pspy ~(get-id fn-name params) ~@body))
                `(~params               (pspy ~(get-id fn-name params) ~@body)))))
          sigs)]
    new-sigs))

(defmacro fnp "Like `fn` but wraps fn bodies with `p` macro."
  {:arglists '([name?  [params*] prepost-map? body]
               [name? ([params*] prepost-map? body)+])}
  [& sigs]
  (let [[?fn-name sigs] (if (symbol? (first sigs)) [(first sigs) (next sigs)] [nil sigs])
        new-sigs        (-fn-sigs (or ?fn-name 'anonymous-fn) sigs)]
    (if ?fn-name
      `(fn ~?fn-name ~@new-sigs)
      `(fn           ~@new-sigs))))

(comment
  (-fn-sigs "foo"      '([x]            (* x x)))
  (macroexpand '(fnp     [x]            (* x x)))
  (macroexpand '(fn       [x]            (* x x)))
  (macroexpand '(fnp bob [x] {:pre [x]} (* x x)))
  (macroexpand '(fn       [x] {:pre [x]} (* x x))))

(defmacro defnp "Like `defn` but wraps fn bodies with `p` macro."
  {:arglists
   '([name doc-string? attr-map?  [params*] prepost-map? body]
     [name doc-string? attr-map? ([params*] prepost-map? body)+ attr-map?])}
  [& sigs]
  (let [[fn-name sigs] (enc/name-with-attrs (first sigs) (next sigs))
        new-sigs       (-fn-sigs fn-name sigs)]
    `(defn ~fn-name ~@new-sigs)))

(comment
  (defnp foo "Docstring"                [x]   (* x x))
  (macroexpand '(defnp foo "Docstring"  [x]   (* x x)))
  (macroexpand '(defn  foo "Docstring"  [x]   (* x x)))
  (macroexpand '(defnp foo "Docstring" ([x]   (* x x))
                                       ([x y] (* x y))))
  (profile :info :defnp-test (foo 5)))

;;;; Deprecated

(def pspy* "Deprecated"
  (if elide-profiling?
    (fn [id f] (f))
    (fn [id f]
      (if *pdata_*
        (let [id (qualified-kw "?" id)
              t0 (System/nanoTime)]
          (try
            (f)
            (finally (-capture-time! id (- (System/nanoTime) t0)))))
        (f)))))

(def p* "Deprecated" pspy*)

;;;;

(comment
  (profile :info :sleepy-threads
    (dotimes [n 5]
      (Thread/sleep 100) ; Unaccounted
      (p :1ms  (Thread/sleep 1))
      (p :2s   (Thread/sleep 2000))
      (p :50ms (Thread/sleep 50))
      (p :rand (Thread/sleep (if (> 0.5 (rand)) 10 500)))
      (p :10ms (Thread/sleep 10))
      "Result"))

  (p :hello "Hello, this is a result") ; Falls through (no *pdata* context)

  (defnp my-fn
    []
    (let [nums (vec (range 1000))]
      (+ (p :fast-sleep (Thread/sleep 1) 10)
         (p :slow-sleep (Thread/sleep 2) 32)
         (p :add  (reduce + nums))
         (p :sub  (reduce - nums))
         (p :mult (reduce * nums))
         (p :div  (reduce / nums)))))

  (profile :info :Arithmetic (dotimes [n 100] (my-fn)))
  (sampling-profile :info 0.5 :sampling-test  (p :string "Hello!"))

  ;; ~90% overhead:
  (profile :info :high-n (dotimes [n 1e6] (p :divs nil))))
