(ns taoensso.timbre
  "Simple, flexible logging for Clojure/Script. No XML."
  {:author "Peter Taoussanis (@ptaoussanis)"}
       
           
                               
                                    
                                                                 
                                                       

        
  (:require
   [clojure.string  :as str]
   [taoensso.encore :as enc :refer () :refer-macros (compile-if have have?)]
   [taoensso.timbre.appenders.core :as core-appenders])

        
  (:require-macros [taoensso.timbre :as timbre-macros :refer ()]))

(if (vector? taoensso.encore/encore-version)
  (enc/assert-min-encore-version [2 58 0])
  (enc/assert-min-encore-version  2.58))

;;;; Config

     
                           
                               
                                           
                                                    
                                                                                 

(declare stacktrace)
(defn default-output-fn
  "Default (fn [data]) -> string output fn.
  You can modify default options with `(partial default-output-fn <opts-map>)`."
  ([     data] (default-output-fn nil data))
  ([opts data]
   (let [{:keys [no-stacktrace? stacktrace-fonts]} opts
         {:keys [level ?err #_vargs msg_ ?ns-str hostname_
                 timestamp_ ?line]}
         data]
     (str
                                         
                                         
       (str/upper-case (name level))  " "
       "[" (or ?ns-str "?") ":" (or ?line "?") "] - "
       (force msg_)
       (when-not no-stacktrace?
         (when-let [err ?err]
           (str "\n" (stacktrace err opts))))))))

;;; Alias core appenders here for user convenience
(declare default-err default-out)
                                                             
                                                          
       (def println-appender core-appenders/println-appender)
       (def console-appender core-appenders/console-appender)

(def example-config
  "Example (+default) Timbre v4 config map.

  APPENDERS
    An appender is a map with keys:
      :min-level       ; Level keyword, or nil (=> no minimum level)
      :enabled?        ;
      :async?          ; Dispatch using agent? Useful for slow appenders (clj only)
      :rate-limit      ; [[ncalls-limit window-ms] <...>], or nil
      :output-fn       ; Optional override for inherited (fn [data]) -> string
      :timestamp-opts  ; Optional override for inherited {:pattern _ :locale _ :timezone _}
      :ns-whitelist    ; Optional, stacks with active config's whitelist
      :ns-blacklist    ; Optional, stacks with active config's blacklist
      :fn              ; (fn [data]) -> side effects, with keys described below

    An appender's fn takes a single data map with keys:
      :config          ; Entire config map (this map, etc.)
      :appender-id     ; Id of appender currently dispatching
      :appender        ; Entire map of appender currently dispatching
      :instant         ; Platform date (java.util.Date or js/Date)
      :level           ; Keyword
      :error-level?    ; Is level e/o #{:error :fatal}?
      :?ns-str         ; String,  or nil
      :?file           ; String,  or nil
      :?line           ; Integer, or nil ; Waiting on CLJ-865
      :?err            ; First-arg platform error, or nil
      :vargs           ; Vector of raw args
      :output_         ; Forceable - final formatted output string created
                       ; by calling (output-fn <this-data-map>)
      :msg_            ; Forceable - args as a string
      :timestamp_      ; Forceable - string
      :hostname_       ; Forceable - string (clj only)
      :output-fn       ; (fn [data]) -> formatted output string
                       ; (see `default-output-fn` for details)
      :context         ; *context* value at log time (see `with-context`)
      :profile-stats   ; From `profile` macro

      **NB** - any keys not specifically documented here should be
      considered private / subject to change without notice.

  MIDDLEWARE
    Middleware are simple (fn [data]) -> ?data fns (applied left->right) that
    transform the data map dispatched to appender fns. If any middleware
    returns nil, NO dispatch will occur (i.e. the event will be filtered).

  The `example-config` source code contains further settings and details.
  See also `set-config!`, `merge-config!`, `set-level!`."

  {:level :debug  ; e/o #{:trace :debug :info :warn :error :fatal :report}

   ;; Control log filtering by namespaces/patterns. Useful for turning off
   ;; logging in noisy libraries, etc.:
   :ns-whitelist  [] #_["my-app.foo-ns"]
   :ns-blacklist  [] #_["taoensso.*"]

   :middleware [] ; (fns [data]) -> ?data, applied left->right

                        
                                ; {:pattern _ :locale _ :timezone _}

   :output-fn default-output-fn ; (fn [data]) -> string

   :appenders
        
                                               
                                                         
     

         
   {;; :println (println-appender {})
    :console (console-appender {})}})

(comment
  (set-config! example-config)
  (infof "Hello %s" "world :-)"))

(enc/defonce* ^:dynamic *config* "See `example-config` for info." example-config)
                                                                                  
                                            
                                                                   

(defn swap-config! [f & args]
         (set!                   *config* (apply f *config* args))
                                                 )

(defn   set-config! [m] (swap-config! (fn [_old] m)))
(defn merge-config! [m] (swap-config! (fn [old] (enc/nested-merge old m))))

(defn     set-level! [level] (swap-config! (fn [m] (assoc m :level level))))
                                   
                                                              

(comment (set-level! :info) *config*)

;;;; Levels

(def ordered-levels [:trace :debug :info :warn :error :fatal :report])
(def ^:private scored-levels (zipmap ordered-levels (next (range))))
(def ^:private valid-levels  (set ordered-levels))
(def ^:private valid-level
  (fn [level]
    (or (valid-levels level)
        (throw (ex-info (str "Invalid logging level: " level) {:level level})))))

(comment (valid-level :info))

(defn level>= [x y] (>= ^long (scored-levels (valid-level x))
                        ^long (scored-levels (valid-level y))))

(comment (qb 10000 (level>= :info :debug)))

     
                                 
                                  
                              
                                                      
                                                           
                                                                  
                                                                   
              

;;;; ns filter

(def ^:private compile-ns-filters
  "(fn [whitelist blacklist]) -> (fn [ns]) -> ?unfiltered-ns"
  (let [->re-pattern
        (fn [x]
          (enc/cond!
            (enc/re-pattern? x) x
            (string? x)
            (let [s (-> (str "^" x "$")
                        (str/replace "." "\\.")
                        (str/replace "*" "(.*)"))]
              (re-pattern s))))]

    (enc/memoize_
      (fn [whitelist blacklist]
        (let [whitelist* (mapv ->re-pattern whitelist)
              blacklist* (mapv ->re-pattern blacklist)

              white-filter
              (cond
                ;; (nil? whitelist)  (fn [ns] false) ; Might be surprising
                (empty?  whitelist*) (fn [ns] true)
                :else (fn [ns] (some #(re-find % ns) whitelist*)))

              black-filter
              (cond
                (empty? blacklist*) (fn [ns] true)
                :else (fn [ns] (not (some #(re-find % ns) blacklist*))))]

          (fn [ns] (when (and (white-filter ns) (black-filter ns)) ns)))))))

(def ^:private ns-filter
  "(fn [whitelist blacklist ns]) -> ?unfiltered-ns"
  (enc/memoize_
    (fn [whitelist blacklist ?ns]
      {:pre [(have? [:or nil? string?] ?ns)]}
      ((compile-ns-filters whitelist blacklist) (or ?ns "")))))

(comment
  (qb 10000 (ns-filter ["foo.*"] ["foo.baz"] "foo.bar"))
  (ns-filter nil nil "")
  (ns-filter nil nil nil))

     
                                     
                                       
                                                                                    
                                                                                     
                                                                                             
                                                                                             
                                                  

;;;; Utils

(declare get-hostname)

(enc/compile-if (do enc/str-join true) ; Encore v2.29.1+ with transducers
  (defn- str-join [xs]
    (enc/str-join " "
      (map
        (fn [x]
          (let [x (enc/nil->str x)] ; Undefined, nil -> "nil"
            (cond
              (record?          x) (pr-str x)
              ;; (enc/lazy-seq? x) (pr-str x) ; Dubious?
              :else x))))
      xs))
  (defn- str-join [xs] (enc/spaced-str-with-nils xs)))

(comment
  (defrecord MyRec [x])
  (str-join ["foo" (MyRec. "foo")]))

     
                                 
                                                                      

(comment (get-agent :my-appender))

(enc/defonce* ^:private get-rate-limiter
  (enc/memoize_ (fn [appender-id specs] (enc/rate-limiter* specs))))

(comment (def rf (get-rate-limiter :my-appender [[10 5000]])))

;;;; Internal logging core

(def ^:dynamic *context*
  "General-purpose dynamic logging context. Context will be included in
  appender data map at logging time." nil)

                                                                               

(defn log?
  "Runtime check: would Timbre currently log at the given logging level?
    * `?ns-str` arg required to support ns filtering
    * `config`  arg required to support non-global config"
  ([level               ] (log? level nil     nil))
  ([level ?ns-str       ] (log? level ?ns-str nil))
  ([level ?ns-str config]
   (let [config       (or config *config*)
         active-level (or (:level config) :report)]
     (and
       (level>= level active-level)
       (ns-filter (:ns-whitelist config) (:ns-blacklist config) ?ns-str)
       true))))

(comment
  (set-level! :debug)
  (log? :trace)
  (with-level :trace (log? :trace))
  (qb 10000
    (log? :trace)
    (log? :trace "foo")
    (tracef "foo")
    (when false "foo"))
  ;; [1.38 1.42 2.08 0.26]

  ;;; Full benchmarks
                                                
                                                                            

  (with-sole-appender {:enabled? true :fn (fn [data] nil)}
    (qb 10000 (info "foo"))) ; ~88ms ; Time to delays ready

  (with-sole-appender {:enabled? true :fn (fn [data] ((:output-fn data) data))}
    (qb 10000 (info "foo"))) ; ~218ms ; Time to output ready
  )

(defn-   next-vargs [v] (if (> (count v) 1) (subvec v 1) []))
(defn- vargs->margs
  "Transforms raw vargs -> {:?err _ :?meta _ ...}, extracting:
  * Special error or ^:meta {} (experimental, undocumented) v0
  * Message format string
  * Message string delay"
  [?err msg-type vargs]
  (let [auto-error? (enc/kw-identical? ?err :auto)
        msg-fmt?    (enc/kw-identical? msg-type :f)
        [v0] vargs]

    (if (and auto-error? (enc/error? v0))
      (let [vargs    (next-vargs vargs)
            ?msg-fmt (if msg-fmt? (let [[v0] vargs] v0) nil)
            vargs    (if msg-fmt? (next-vargs vargs) vargs)
            msg_     (delay
                      (case msg-type
                        nil ""
                        :p  (str-join                            vargs)
                        :f  (enc/format* (have string? ?msg-fmt) vargs)))]

        {:?err v0 :?meta nil :?msg-fmt ?msg-fmt :msg_ msg_ :vargs vargs})

      (let [?meta    (if (and (map? v0) (:meta (meta v0))) v0 nil)
            ?err     (or (:err ?meta) (if auto-error? nil ?err))
            ?meta    (dissoc ?meta :err)
            vargs    (if ?meta (next-vargs vargs) vargs)
            ?msg-fmt (if msg-fmt? (let [[v0] vargs] v0) nil)
            vargs    (if msg-fmt? (next-vargs vargs) vargs)
            msg_     (delay
                      (case msg-type
                        nil ""
                        :p  (str-join                            vargs)
                        :f  (enc/format* (have string? ?msg-fmt) vargs)))]

        {:?err ?err :?meta ?meta :?msg-fmt ?msg-fmt :msg_ msg_ :vargs vargs}))))

(comment
  (let [ex (Exception. "ex")]
    (qb 10000
      (vargs->margs :auto :f ["fmt" :a :b :c])
      (vargs->margs :auto :p [ex    :a :b :c])
      (vargs->margs :auto :p [^:meta {:foo :bar} :a :b :c])
      (vargs->margs :auto :p [       {:foo :bar} :a :b :c])
      (vargs->margs :auto :p [ex])
      (vargs->margs :auto :p [^:meta {:err ex}   :a :b :c])))
  ;; [2.79 2.51 6.13 1.65 1.94 6.2]
  (infof                                 "Hi %s" "steve")
  (infof ^:meta {:hash :bar}             "Hi %s" "steve")
  (infof ^:meta {:err (Exception. "ex")} "Hi %s" "steve"))

(defn -log! "Core low-level log fn. Implementation detail!"
  [config level ?ns-str ?file ?line msg-type ?err vargs_
   ?base-data callsite-id]

  (when (log? level ?ns-str config) ; Runtime check
    (let [instant (enc/now-dt)
          context *context*
          vargs   @vargs_

          ;; {:keys [?err ?meta ?msg-fmt msg_ vargs]}:
          margs (vargs->margs ?err msg-type vargs)
          data
          (merge
           ?base-data
           margs
           {:instant instant
            :level   level
            :context context
            :config  config ; Entire config!
            :?ns-str ?ns-str
            :?file   ?file
            :?line   ?line
                                                         
            :error-level? (#{:error :fatal} level)

            ;; Uniquely identifies a particular logging call for purposes of
            ;; rate limiting, etc.
            :hash_ ; TODO Undocumented (experimental)
            (delay
             (hash
              ;; Nb excl. instant
              [callsite-id ; Only useful for direct macro calls
               (:?msg-fmt margs)
               (get-in margs [:?meta :hash] ; Explicit hash provided
                 (:vargs margs))]))

            ;; :?err     <from-margs>
            ;; :?meta    <from-margs> ; TODO Undocumented (experimental)
            ;; :?msg-fmt <from-margs> ; TODO Undocumented (experimental)
            ;; :msg_     <from-margs>
            ;; :vargs    <from-margs>

            ;;; Deprecated
            :?err_  (delay (:?err  margs))
            :vargs_ (delay (:vargs margs))})

          ?data
          (reduce ; Apply middleware: data->?data
            (fn [acc mf]
              (let [result (mf acc)]
                (if (nil? result)
                  (reduced nil)
                  result)))
            data
            (:middleware config))]

      (when-let [data ?data] ; Not filtered by middleware
        (let [;; Optimization: try maximize output+timestamp sharing
              ;; between appenders
              output-fn1 (enc/memoize_ (get config :output-fn default-output-fn))
                                                                                                    
                                   ; (fn [timestamp-opts]) -> Shared delay
                   
                           
                         
                       
                                                               
                                                                              
                                         ]

          (reduce-kv
           (fn [_ id appender]
             (when (and (:enabled? appender)
                        (level>= level (or (:min-level appender) :trace)))

               ;; Appender ns filter stacks with main config's ns filter:
               (when (ns-filter (:ns-whitelist appender)
                                (:ns-blacklist appender)
                                ?ns-str)

                 (let [rate-limit-specs (:rate-limit appender)
                       rate-limit-okay?
                       (or
                        (empty? rate-limit-specs)
                        (let [rl-fn (get-rate-limiter id rate-limit-specs)]
                          (not (rl-fn (force (:hash_ data))))))]

                   (when rate-limit-okay?
                     (let [{:keys [async?] apfn :fn} appender

                           output-fn
                           (let [f (:output-fn appender)]
                             (if (or (nil? f) (enc/kw-identical? f :inherit))
                               output-fn1
                               f))

                                           
                                
                                                                 
                                                                                   
                                                                     
                                                                             

                           output_
                           (delay
                            (output-fn
                                                                       
                                    data))

                           data ; Final data prep before going to appender
                           (conj data
                             {:appender-id id
                              :appender    appender
                              :output-fn   output-fn
                              :output_     output_
                                                                })]

                             (apfn data) ; Allow errors to throw
                           
                                
                                                                      
                                                           
                         ))))))
          nil
          (:appenders config))))))
  nil)

(comment
  (-log! *config* :info nil nil nil :p :auto
    (delay [(do (println "hi") :x) :y]) nil "callsite-id"))

                       
                         
                                                                  
                                 
                                     
                                                                            
                                                    

                                                                           
                                                   
                     

(comment (-with-elision :info "ns" (println "foo")))

(defn- fline [and-form] (:line (meta and-form)))

                                              
                                                     

                                                     
                                                
                                                                             

                                                            
                                         
                                
                                                        
                                                        
                  
                                                                
                                                                 
                                                       
                                                    
                                                            
                                 
                                             
                                               

                                                            

                                                                            
                                                                             
                                     
                       
                                                                     
                                                

                                                                     
                                                         

(comment
  (log! :info :p ["foo"])
  (macroexpand '(log! :info :p ["foo"]))
  (macroexpand '(log! :info :p ["foo"] {:?line 42})))

;;;; Main public API-level stuff
;; TODO Have a bunch of cruft here trying to work around CLJ-865 to some extent

;;; Log using print-style args
                                                                                                      
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       

;;; Log using format-style args
                                                                                                      
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       

(comment
  (infof "hello %s" "world")
  (infof (Exception.) "hello %s" "world")
  (infof (Exception.)))

                                    
                                                      
                           
                              
                                             
               

                                                
                                                      
                           
                              
                                            
                  
               

                                                                              

                                                                                           
                                                                                           
                                                                                           

     
                                     
                                                    
               
               
                   
                                        
                                                                
                                   

                                              
                                            
                                                                                  

(comment
  (log-errors             (/ 0))
  (log-and-rethrow-errors (/ 0))
  (logged-future          (/ 0))
  (handle-uncaught-jvm-exceptions!))

                                             
                                  
                         
                             
                                                            
                                                                           

                                 
                 

             
                                                                             
                                                                       
                                                                                
                                                                                
                                                                                
                                                                                 

                                    
(comment ((fn foo [x y] (get-env)) 5 10))

     
                  
                 
                                       
                                                                        
                                                                         
                                           
                                                    
                                             
    
                                       
                                                                        
                                                                         
                                           
                                                    
                                             

;;;; Misc public utils

     
                            
                                        
                                                                   
                                                                   
                                                                   
                                    
                                                                 

                                                                
                                                                
                                    
                                                           

     
                 
                                
          
                                                                               
                                                                            
                                                
                                                                             
                                            
                  
                  
                                                                  
                                                                     
                                                 

                                                           

(comment (get-hostname))

     
                                       
                                                          
           

(defn stacktrace
  ([err     ] (stacktrace err nil))
  ([err opts]
          (str err) ; TODO Alternatives?
        
                                                       
                                                  
                                  
                                
                 
                         
                                     

                                     
                                        
                                              
                                               ))

(comment (stacktrace (Exception. "Boo") {:stacktrace-fonts {}}))

                                                     
                      
                                                                 
                                               

;;;; Deprecated

       (def console-?appender core-appenders/console-appender)
(defn logging-enabled? [level compile-time-ns] (log? level (str compile-time-ns)))
(defn str-println      [& xs] (str-join xs))
                                                                            
                                                                            
                                      
                 
                                         
                                                
                                                        
                                                                    

;;;;;;;;;;;; This file autogenerated from src/taoensso/timbre.cljx
