(ns taoensso.timbre
  "Simple, flexible logging for Clojure/Script. No XML."
  {:author "Peter Taoussanis (@ptaoussanis)"}
       
           
                               
                                    
                                                                 
                                                       

        
  (:require
   [clojure.string  :as str]
   [taoensso.encore :as enc :refer () :refer-macros (compile-if have have?)]
   [taoensso.timbre.appenders.core :as core-appenders])

        
  (:require-macros [taoensso.timbre :as timbre-macros :refer ()]))

(if (vector? taoensso.encore/encore-version)
  (enc/assert-min-encore-version [2 50 0]) ; For nested-merge fixes
  (enc/assert-min-encore-version  2.50))

;;;; Config

     
                   
                                          
                                            
                      

     
                           
                               
                                           
                                                    
                                                                                 

(declare stacktrace)
(defn default-output-fn
  "Default (fn [data]) -> string output fn.
  You can modify default options with `(partial default-output-fn <opts-map>)`."
  ([     data] (default-output-fn nil data))
  ([opts data]
   (let [{:keys [no-stacktrace? stacktrace-fonts]} opts
         {:keys [level ?err #_vargs msg_ ?ns-str hostname_
                 timestamp_ ?line]}
         data]
     (str
                                         
                                         
       (str/upper-case (name level))  " "
       "[" (or ?ns-str "?") ":" (or ?line "?") "] - "
       (force msg_)
       (when-not no-stacktrace?
         (when-let [err ?err]
           (str "\n" (stacktrace err opts))))))))

;;; Alias core appenders here for user convenience
(declare default-err default-out)
                                                             
                                                          
       (def println-appender core-appenders/println-appender)
       (def console-appender core-appenders/console-appender)

(def example-config
  "Example (+default) Timbre v4 config map.

  APPENDERS
    An appender is a map with keys:
      :min-level       ; Level keyword, or nil (=> no minimum level)
      :enabled?        ;
      :async?          ; Dispatch using agent? Useful for slow appenders (clj only)
      :rate-limit      ; [[ncalls-limit window-ms] <...>], or nil
      :output-fn       ; Optional override for inherited (fn [data]) -> string
      :fn              ; (fn [data]) -> side effects, with keys described below
      :ns-whitelist    ; Optional, stacks with active config's whitelist
      :ns-blacklist    ; Optional, stacks with active config's blacklist

    An appender's fn takes a single data map with keys:
      :config          ; Entire config map (this map, etc.)
      :appender-id     ; Id of appender currently dispatching
      :appender        ; Entire map of appender currently dispatching

      :instant         ; Platform date (java.util.Date or js/Date)
      :level           ; Keyword
      :error-level?    ; Is level e/o #{:error :fatal}?
      :?ns-str         ; String, or nil
      :?file           ; String, or nil
      :?line           ; Integer, or nil ; Waiting on CLJ-865
      :?err            ; First-arg platform error, or nil
      :vargs           ; Args vector
      :msg_            ; Forceable - args as a string
      :timestamp_      ; Forceable - string
      :hostname_       ; Forceable - string (clj only)
      :output-fn       ; (fn [data]) -> formatted output string
                       ; (see `default-output-fn` for details)
      :context         ; *context* value at log time (see `with-context`)
      :profile-stats   ; From `profile` macro

  MIDDLEWARE
    Middleware are simple (fn [data]) -> ?data fns (applied left->right) that
    transform the data map dispatched to appender fns. If any middleware returns
    nil, NO dispatching will occur (i.e. the event will be filtered).

  The `example-config` source code contains further settings and details.
  See also `set-config!`, `merge-config!`, `set-level!`."

  {:level :debug  ; e/o #{:trace :debug :info :warn :error :fatal :report}

   ;; Control log filtering by namespaces/patterns. Useful for turning off
   ;; logging in noisy libraries, etc.:
   :ns-whitelist  [] #_["my-app.foo-ns"]
   :ns-blacklist  [] #_["taoensso.*"]

   :middleware [] ; (fns [data]) -> ?data, applied left->right

                        
                                ; {:pattern _ :locale _ :timezone _}

   :output-fn default-output-fn ; (fn [data]) -> string

   :appenders
        
                                               
                                                         
     

         
   {;; :println (println-appender {})
    :console (console-appender {})}})

(comment
  (set-config! example-config)
  (infof "Hello %s" "world :-)"))

(enc/defonce* ^:dynamic *config* "See `example-config` for info." example-config)
                                                                                  
                                            
                                                                   

(defn swap-config! [f & args]
         (set!                   *config* (apply f *config* args))
                                                 )

(defn   set-config! [m] (swap-config! (fn [_old] m)))
(defn merge-config! [m] (swap-config! (fn [old] (enc/nested-merge old m))))

(defn     set-level! [level] (swap-config! (fn [m] (merge m {:level level}))))
                                   
                                                                

(comment (set-level! :info) *config*)

;;;; Levels

(def ordered-levels [:trace :debug :info :warn :error :fatal :report])
(def ^:private scored-levels (zipmap ordered-levels (next (range))))
(def ^:private valid-levels  (set ordered-levels))
(def ^:private valid-level
  (fn [level]
    (or (valid-levels level)
        (throw (ex-info (str "Invalid logging level: " level) {:level level})))))

(comment (valid-level :info))

(defn level>= [x y] (>= (long (scored-levels (valid-level x)))
                        (long (scored-levels (valid-level y)))))

(comment (qb 10000 (level>= :info :debug)))

     
                                 
                                  
                              
                                                      
                                                  
                                                         
                                                                   
              

;;;; ns filter

(def ^:private compile-ns-filters
  "(fn [whitelist blacklist]) -> (fn [ns]) -> ?unfiltered-ns"
  (let [->re-pattern
        (fn [x]
          (enc/cond!
            (enc/re-pattern? x) x
            (string? x)
            (let [s (-> (str "^" x "$")
                        (str/replace "." "\\.")
                        (str/replace "*" "(.*)"))]
              (re-pattern s))))]

    (enc/memoize_
      (fn [whitelist blacklist]
        (let [whitelist* (mapv ->re-pattern whitelist)
              blacklist* (mapv ->re-pattern blacklist)

              white-filter
              (cond
                ;; (nil? whitelist)  (fn [ns] false) ; Might be surprising
                (empty?  whitelist*) (fn [ns] true)
                :else (fn [ns] (some #(re-find % ns) whitelist*)))

              black-filter
              (cond
                (empty? blacklist*) (fn [ns] true)
                :else (fn [ns] (not (some #(re-find % ns) blacklist*))))]

          (fn [ns] (when (and (white-filter ns) (black-filter ns)) ns)))))))

(def ^:private ns-filter
  "(fn [whitelist blacklist ns]) -> ?unfiltered-ns"
  (enc/memoize_
    (fn [whitelist blacklist ns]
      {:pre [(have? string? ns)]}
      ((compile-ns-filters whitelist blacklist) ns))))

(comment
  (qb 10000 (ns-filter ["foo.*"] ["foo.baz"] "foo.bar"))
  (ns-filter nil nil ""))

     
                                     
                                       
                                                                           
                                                                            
                                                                                             
                                                                                             
                                                  

;;;; Utils

(declare get-hostname)

(enc/compile-if (do enc/str-join true) ; Encore v2.29.1+ with transducers
  (defn- str-join [xs]
    (enc/str-join " "
      (map
        (fn [x]
          (let [x (enc/nil->str x)] ; Undefined, nil -> "nil"
            (cond
              (record?          x) (pr-str x)
              ;; (enc/lazy-seq? x) (pr-str x) ; Dubious?
              :else x))))
      xs))
  (defn- str-join [xs] (enc/spaced-str-with-nils xs)))

(comment
  (defrecord MyRec [x])
  (str-join ["foo" (MyRec. "foo")]))

(defn default-data-hash-fn
  "Used for rate limiters, some appenders (e.g. Carmine), etc.
  Goal: (hash data-1) = (hash data-2) iff data-1 \"the same\" as data-2 for
  rate-limiting purposes, etc."
  [data]
  (or
   (when-let [h (get-in data [:?meta :hash])] (hash h)) ; Explicit hash given
   (hash [(:?ns-str data) (or (:?line data) (:vargs data))]) ; Autogen hash
   ))

     
                                 
                                                                      

(comment (get-agent :my-appender))

(enc/defonce* ^:private get-rate-limiter
  (enc/memoize_ (fn [appender-id specs] (enc/rate-limiter* specs))))

(comment (def rf (get-rate-limiter :my-appender [[10 5000]])))

(defn- inherit-over [k appender config default]
  (or
    (let [a (get appender k)] (when-not (enc/kw-identical? a :inherit) a))
    (get config k)
    default))

(defn- inherit-into [k appender config default]
  (merge default
    (get config k)
    (let [a (get appender k)] (when-not (enc/kw-identical? a :inherit) a))))

(comment
  (inherit-over :foo {:foo :inherit} {:foo :bar} nil)
  (inherit-into :foo {:foo {:a :A :b :B :c :C}} {:foo {:a 1 :b 2 :c 3 :d 4}} nil))

;;;; Internal logging core

(def ^:dynamic *context*
  "General-purpose dynamic logging context. Context will be included in appender
  data map at logging time." nil)

                                                                               

(defn log?
  "Runtime check: would Timbre currently log at the given logging level?
    * `?ns-str` arg required to support ns filtering
    * `config`  arg required to support non-global config"
  ([level               ] (log? level nil     nil))
  ([level ?ns-str       ] (log? level ?ns-str nil))
  ([level ?ns-str config]
   (let [config       (or config *config*)
         active-level (or (:level config) :report)]
     (and
       (level>= level active-level)
       (ns-filter (:ns-whitelist config) (:ns-blacklist config) (or ?ns-str ""))
       true))))

(comment
  (set-level! :debug)
  (log? :trace)
  (with-level :trace (log? :trace))
  (qb 10000
    (log? :trace)
    (log? :trace "foo")
    (tracef "foo")
    (when false "foo"))
  ;; [1.38 1.42 2.08 0.26]

  ;;; Full benchmarks
                                                
                                                                            

  (with-sole-appender {:enabled? true :fn (fn [data] nil)}
    (qb 10000 (info "foo"))) ; ~88ms ; Time to delays ready

  (with-sole-appender {:enabled? true :fn (fn [data] ((:output-fn data) data))}
    (qb 10000 (info "foo"))) ; ~218ms ; Time to output ready
  )

(defn-   next-vargs [v] (if (> (count v) 1) (subvec v 1) []))
(defn- vargs->margs
  "Extracts special v0 (error or ^:meta {})"
  ;; ^:meta stuff is currently experimental, undocumented
  [?err vargs]
  (let [auto-error? (enc/kw-identical? ?err :auto)
        [v0] vargs]
    (if (and auto-error? (enc/error? v0))
      {:?err v0 :?meta nil :vargs (next-vargs vargs)}
      (let [?meta (if (and (map? v0) (:meta (meta v0))) v0 nil)
            ?err  (or (:err ?meta) (if auto-error? nil ?err))
            ?meta (dissoc ?meta :err)
            vargs (if ?meta (next-vargs vargs) vargs)]
        {:?err ?err :?meta ?meta :vargs vargs}))))

(comment
  (qb 10000
    (vargs->margs :auto [:a :b :c])
    (vargs->margs :auto [(Exception. "ex")  :b :c])
    (vargs->margs :auto [^:meta {:foo :bar} :b :c])
    (vargs->margs :auto [       {:foo :bar} :b :c])
    (vargs->margs :auto [(Exception. "ex")])
    (vargs->margs :auto [^:meta {:err (Exception. "ex")} :b :c]))
  ;; [0.94 35.92 6.95 2.55 35.88]
  (infof                                  "Hi %s" "steve")
  (infof ^:meta {:hash :bar}              "Hi %s" "steve")
  (infof ^:meta {:err  (Exception. "ex")} "Hi %s" "steve"))

(defn -log! "Core low-level log fn. Implementation detail!"
  [config level ?ns-str ?file ?line msg-type ?err vargs_ ?base-data]
  (when (log? level ?ns-str config) ; Runtime check
    (let [data
          (merge
           ?base-data
           (vargs->margs ?err (force vargs_)) ; ?err ?meta vargs
           {:instant (enc/now-dt)
            :context *context*
            :level   level
            :config  config ; Entire config!
            :?ns-str ?ns-str
            :?file   ?file
            :?line   ?line
                                                         
            :error-level? (#{:error :fatal} level)})

          data ; Deprecated
          (assoc data
            :?err_  (delay (:?err data))
            :vargs_ (delay (:vargs data)))

          ?data
          (reduce ; Apply middleware: data->?data
            (fn [acc mf]
              (let [result (mf acc)]
                (if (nil? result)
                  (reduced nil)
                  result)))
            data
            (:middleware config))]

      (when-let [data ?data] ; Not filtered by middleware
        (let [msg-fn
              (fn [vargs]
                (case msg-type
                  nil ""
                  :p  (str-join vargs)
                  :f  (let [[fmt vargs] (enc/vsplit-first vargs)]
                        (have? string? fmt)
                        (enc/format* fmt vargs))))]

          (reduce-kv
           (fn [_ id appender]
             (when (and (:enabled? appender)
                        (level>= level (or (:min-level appender) :trace)))

               ;; Appender ns filter stacks with main config's ns filter:
               (when (ns-filter (:ns-whitelist appender)
                                (:ns-blacklist appender)
                                (or ?ns-str ""))

                 (let [rate-limit-specs (:rate-limit appender)
                       data-hash-fn (inherit-over :data-hash-fn appender config
                                      default-data-hash-fn)
                       rate-limit-okay?
                       (or (empty? rate-limit-specs)
                           (let [rl-fn     (get-rate-limiter id rate-limit-specs)
                                 data-hash (data-hash-fn data)]
                             (not (rl-fn data-hash))))]

                   (when rate-limit-okay?
                     (let [{:keys [async?] apfn :fn} appender
                           msg_      (delay (msg-fn (:vargs data)))
                           output-fn (inherit-over :output-fn appender config
                                       default-output-fn)

                                           
                                
                                 
                                                
                                                               
                                                   
                                                           

                                                                   
                                                 

                                      
                                                              
                                                                     
                                                  

                           data ; Final data prep before going to appender
                           (merge data
                             {:appender-id  id
                              :appender     appender
                              :msg_         msg_
                              :msg-fn       msg-fn
                              :output-fn    output-fn
                              :data-hash-fn data-hash-fn
                                                                })]

                              (apfn data) ; Allow errors to throw
                            
                                 
                                                                       
                                                            
                          ))))))
           nil
           (:appenders config))))))
  nil)

(comment
  (-log! *config* :info nil nil nil :p :auto
    (delay [(do (println "hi") :x) :y]) nil))

                       
                         
                                                                  
                                 
                                     
                                                                            
                                                    

                                                                           
                                                   
                     

(comment (-with-elision :info "ns" (println "foo")))

(defn- fline [and-form] (:line (meta and-form)))

                                              
                                                     

                                                     
                                                
                                                                
                                        

                                                            
                                         
                                
                                                        
                                                        
                  
                                                                
                                                                 
                                                       
                                                    
                                                            
                                 
                                             
                                               

                                                             

                                                                     
                                            

(comment
  (log! :info :p ["foo"])
  (macroexpand '(log! :info :p ["foo"]))
  (macroexpand '(log! :info :p ["foo"] {:?line 42})))

;;;; Main public API-level stuff
;; TODO Have a bunch of cruft here trying to work around CLJ-865 to some extent

;;; Log using print-style args
                                                                                                      
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       

;;; Log using format-style args
                                                                                                      
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       
                                                                                       

(comment
  (infof "hello %s" "world")
  (infof (Exception.) "hello %s" "world")
  (infof (Exception.)))

                                    
                                                      
                           
                              
                                             
               

                                                
                                                      
                           
                              
                                            
                  
               

                                                                              

                                                                                           
                                                                                           
                                                                                           

     
                                     
                                                    
               
               
                   
                                        
                                                                
                                   

                                              
                                            
                                                                                  

(comment
  (log-errors             (/ 0))
  (log-and-rethrow-errors (/ 0))
  (logged-future          (/ 0))
  (handle-uncaught-jvm-exceptions!))

                                             
                                  
                         
                             
                                                            
                                                                           

                                 
                 

             
                                                                             
                                                                       
                                                                                
                                                                                
                                                                                
                                                                                 

                                    
(comment ((fn foo [x y] (get-env)) 5 10))

     
                  
                 
                                       
                                                                        
                                                                         
                                           
                                                    
                                                      
    
                                       
                                                                        
                                                                         
                                           
                                                    
                                                      

;;;; Misc public utils

     
                            
                                        
                                                                   
                                                                   
                                                                   
                                    
                                                                 

                                                                
                                                                
                                    
                                                           

     
                 
                                
          
                                                                               
                                                                            
                                                
                                                                             
                                            
                  
                  
                                                                  
                                                                     
                                                 

                                                           

(comment (get-hostname))

     
                                       
                                                 
           

(defn stacktrace
  ([err     ] (stacktrace err nil))
  ([err opts]
          (str err) ; TODO Alternatives?
        
                                                       
                                                  
                                  
                                
                 
                         
                                     

                                     
                                        
                                              
                                               ))

(comment (stacktrace (Exception. "Boo") {:stacktrace-fonts {}}))

                                                     
                      
                                                                 
                                               

;;;; Deprecated

       (def console-?appender core-appenders/console-appender)
(defn logging-enabled? [level compile-time-ns] (log? level (str compile-time-ns)))
(defn str-println      [& xs] (str-join xs))
                                                                            
                                                                            
                                      
                 
                                         
                                                
                                                        
                                                                    

;;;;;;;;;;;; This file autogenerated from src/taoensso/timbre.cljx
