(ns taoensso.tengen.reagent
  "Simple let-based Reagent component fns for Clojure/Script."
  {:author "Peter Taoussanis (@ptaoussanis)"}

  #?(:clj
     (:require
      [taoensso.encore      :as enc  :refer [have have?]]
      [taoensso.tengen.impl :as impl]))

  #?(:cljs
     (:require
      [taoensso.encore      :as enc  :refer-macros [have have?]]
      [taoensso.tengen.impl :as impl :refer-macros []])))

(defmacro cmptfn
  "Reagent component fn util. Provides a sensible let-flow API for writing
  simple, flexible Clj/s Reagent components.

  (cmptfn
    id              ; Component identifier for debugging, error reporting, etc.
    [x y]           ; Reagent args passed to component
    :let-mount  []  ; Est. with each instance mount,  avail. downstream
    :let-render []  ; Est. with each instance render, avail. downstream, pure!
    :render      <body> ; Or just provide render body as final arg, pure!
    :post-render <body> ; For DOM node setup/mutation
    :unmount     <body> ; For DOM node teardown
  )

  - Magic bindings: `this-cmpt`, `this-mounting?`.
  - `[<cmpt> {:ref (fn [node] _)}]` or (enc/oget ev \"currentTarget\") for nodes.
  - Call Reagent components as:
    * (cmptfn <...>) - to get inlining.
    * [cmptfn <...>] - to get an intermediary Reagent component:
      * Rerender-iff-args-change semantics.
      * Can take ^{:key _} [cmptfn <...>]."
  [id params & args]
  (let [implicit-render-body? (odd? (count args))
        args-map
        (if implicit-render-body?
          (impl/hash-map-with-unique-ks (butlast args))
          (impl/hash-map-with-unique-ks          args))

        _ (have? [:ks<= #{:let-mount :let-render :render
                          :post-render :unmount}]
            args-map)

        _ (when implicit-render-body?
            (assert (not (contains? args-map :render))
              "Ambiguous render body: provided as both a :render value and implicit final arg"))

        have-arg?   (set (keys args-map)) ; For existance check w/o val eval
        render-body (if implicit-render-body? (last args) (:render args-map))
        _           (assert render-body "No (nil) render body provided")

        ;; [x :x y x]
        mount-bindings  (:let-mount  args-map)
        render-bindings (:let-render args-map)

        ;; [[x y] [:x x]] ; We actually just want the lval forms here
        [ mount-lvals  _mount-rvals] (impl/split-let-pairs  mount-bindings)
        [render-lvals _render-rvals] (impl/split-let-pairs render-bindings)

        ;; Define our cfn lifecycle fns
        ;; NB We try minimize code expansion size here (esp. gensyms)

        argv
        (if (seq params)
          (into ['__] params) ; [__ x y z ...]
          '__)

        ?mount-rvals-fn
        (when (seq mount-bindings)
          `(fn [~'this-cmpt ~argv]
             (impl/binding-rvals ~mount-bindings)))

        ?render-rvals-fn
        (when (seq render-bindings)
          `(fn [~'this-cmpt ~argv ~'this-mounting? ~mount-lvals]
             (impl/binding-rvals ~render-bindings)))

        render-fn
        `(fn [~'this-cmpt ~argv ~'this-mounting? ~mount-lvals ~render-lvals]
           ~render-body)

        ?post-render-fn
        (when (have-arg? :post-render)
          `(fn [~'this-cmpt ~argv ~'this-mounting? ~mount-lvals ~render-lvals]
             ~(:post-render args-map)))

        ?unmount-fn
        (when (have-arg? :unmount)
          `(fn [~'this-cmpt ~argv ~mount-lvals ~render-lvals]
             ~(:unmount args-map)))]

    `(impl/new-cmptfn
       ~id
       ~?mount-rvals-fn
       ~?render-rvals-fn
       ~render-fn
       ~?post-render-fn
       ~?unmount-fn)))

(defmacro def-cmptfn
  "Defines a top-level Reagent component fn using `cmptfn`.
  See the `cmptfn` docstring for more details on args, etc."
  [sym & sigs]
  (let [[sym args] (enc/name-with-attrs sym sigs)]
    `(def ~sym
       (cmptfn
         ~(str *ns* "/" sym ":" (:line (meta &form) "?"))
         ~@args))))

(comment
  (macroexpand                  '(cmptfn :id [x] [:div x]))
  (clojure.walk/macroexpand-all '(cmptfn :id [x] [:div x]))
  (clojure.walk/macroexpand-all '(def-cmptfn foo [x] [:div x])))

