(ns taoensso.tengen.impl
  "Private implementation details."
  (:require
   [taoensso.encore :as enc]
   #?(:cljs [reagent.core])))

(defmacro binding-rvals
  "Evaluates and returns vector of rhs values for given bindings while
  preserving support for the usual let-binding facilities like destructuring,
  referring to previous bindings, etc.

    [{:keys [x]} {:x 1}
     <...>
    ] ->
    (let [alias1      {:x 1}
          {:keys [x]} alias1
          <...>
         ]
      [alias1 <...>])"
  [bindings]
  (let [pairs   (partition 2 bindings)
        lvals   (mapv first  pairs)
        rvals   (mapv second pairs)
        aliases (mapv (fn [i] (symbol (str "__lv" i))) (range (count lvals)))

        alias-bindings (interleave aliases rvals)
        lval-bindings  (interleave lvals   aliases)

        alias-bpairs (partition 2 alias-bindings) ; [(<alias>  <rval>) ...]
        lval-bpairs  (partition 2 lval-bindings)  ; [(<lval>  <alias>) ...]

        bindings* (reduce into [] (interleave alias-bpairs lval-bpairs))]

    `(let ~bindings* ~aliases)))

(comment
  (do             (binding-rvals [x 1, {:keys [a b]} {:a x :b x}]))
  (macroexpand-1 '(binding-rvals [x 1, {:keys [a b]} {:a x :b x}]))
  (macroexpand   '(binding-rvals [x 1, {:keys [a b]} {:a x :b x}])))

(defn split-let-pairs [bindings]
  (if (seq bindings)
    (let [parts (partition 2 bindings)]
      [(mapv first parts) (mapv second parts)])
    ['__ #_[] nil]))

(comment
  (split-let-pairs [:a :b :c :d])
  (split-let-pairs nil))

(comment
  (enc/declare-remote
    reagent.core/atom
    reagent.core/argv
    reagent.core/create-class))

#?(:cljs
   (defn new-cmptfn
     "Returns a new Reagent component defined in terms of special cmptfn
     lifecycle fns. These in turn are defined in a special macro context to
     get magic bindings and flow-through behaviour, etc."

     [id               ; For debugging, error reporting, etc.
      ?mount-rvals-fn  ; (fn [cmpt argv])
      ?render-rvals-fn ; (fn [cmpt argv mounting? mount-rvals]), ideally pure
      render-fn        ; (fn [cmpt argv mounting? mount-rvals render-rvals])
      ?post-render-fn  ; (fn [cmpt argv mounting? mount-rvals render-rvals])
      ?unmount-fn      ; (fn [cmpt argv           mount-rvals render-rvals])
      ]

     (reagent.core/create-class
       (enc/assoc-some {}
         ;; Invoked (only once per instance) before 1st rendering
         :component-will-mount
         (when-let [f ?mount-rvals-fn]
           (fn [cmpt]
             (let [argv  (reagent.core/argv cmpt)
                   rvals (f cmpt argv)]
               (set! (.-cfnMountRvals cmpt) rvals))))

         :render
         (fn [cmpt]
           (let [argv        (reagent.core/argv cmpt)
                 mounting?   (not (.-cfnMounted cmpt))
                 mount-rvals (.-cfnMountRvals   cmpt)]

             (try
               (let [render-rvals
                     (when-let [f ?render-rvals-fn]
                       (let [rvals (f cmpt argv mounting? mount-rvals)]
                         (set! (.-cfnRenderRvals cmpt) rvals)
                         rvals))]

                 (render-fn cmpt argv mounting? mount-rvals render-rvals))

               (catch js/Error e
                 (throw
                   (ex-info "Render error"
                     {:id id
                      :path (reagent.core/component-path cmpt)
                      :mounting? mounting?}
                     e))))))

         ;; Invoked (only once per instance) after 1st rendering
         :component-did-mount
         (fn [cmpt]
           (set! (.-cfnMounted cmpt) true)
           (when-let [f ?post-render-fn]
             (let [argv (reagent.core/argv cmpt)
                   mounting? true]
               (f cmpt argv mounting?
                 (.-cfnMountRvals  cmpt)
                 (.-cfnRenderRvals cmpt)))))

         ;; Invoked before every rendering but the first
         ;; :component-will-update nil #_ (fn [cmpt new-argv])

         ;; Invoked after every rendering but the first
         :component-did-update
         (when-let [f ?post-render-fn]
           (fn [cmpt old-argv]
             (let [argv (reagent.core/argv cmpt)
                   mounting? false]
               (f cmpt argv mounting?
                 (.-cfnMountRvals  cmpt)
                 (.-cfnRenderRvals cmpt)))))

         :component-will-unmount
         (when-let [f ?unmount-fn]
           (fn [cmpt]
             (let [argv (reagent.core/argv cmpt)]
               (f cmpt argv
                 (.-cfnMountRvals  cmpt)
                 (.-cfnRenderRvals cmpt)))))))))

(defn hash-map-with-unique-ks [kvs]
  (enc/reduce-kvs
    (fn [acc k v]
      (if (contains? acc k)
        (throw
          (ex-info "Duplicate map key"
            {:k k, :old-v (get acc k), :new-v v}))
        (assoc acc k v)))
    {}
    kvs))
