(ns taoensso.sente
  "Channel sockets for Clojure/Script.

      Protocol  | client>server | client>server ?+ ack/reply | server>user push
    * WebSockets:       ✓              [1]                           ✓
    * Ajax:            [2]              ✓                           [3]

    [1] Emulate with cb-uuid wrapping
    [2] Emulate with dummy-cb wrapping
    [3] Emulate with long-polling

  Abbreviations:
    * chsk      - Channel socket (Sente's own pseudo \"socket\")
    * server-ch - Underlying web server's async channel that implement
                  Sente's server channel interface
    * sch       - server-ch alias
    * uid       - User-id. An application-level user identifier used for async
                  push. May have semantic meaning (e.g. username, email address),
                  may not (e.g. client/random id) - app's discretion.
    * cb        - Callback
    * tout      - Timeout
    * ws        - WebSocket/s
    * pstr      - Packed string. Arbitrary Clojure data serialized as a
                  string (e.g. edn) for client<->server comms

  Special messages:
    * Callback wrapping: [<clj> <?cb-uuid>] for [1], [2]
    * Callback replies: :chsk/closed, :chsk/timeout, :chsk/error

    * Client-side events:
        [:chsk/handshake [<?uid> <?csrf-token> <?handshake-data> <first-handshake?>]]
        [:chsk/state <new-state-map>]
        [:chsk/recv <ev-as-pushed-from-server>] ; Server>user push
        [:chsk/ws-error <websocket-error>] ; Experimental, subject to change

    * Server-side events:
        [:chsk/bad-package <packed-str>]
        [:chsk/bad-event   <chsk-event>]
        [:chsk/uidport-open]
        [:chsk/uidport-close]

  Notable implementation details:
    * core.async is used liberally where brute-force core.async allows for
      significant implementation simplifications. We lean on core.async's
      efficiency here.
    * For WebSocket fallback we use long-polling rather than HTTP 1.1 streaming
      (chunked transfer encoding). Http-kit _does_ support chunked transfer
      encoding but a small minority of browsers &/or proxies do not. Instead of
      implementing all 3 modes (WebSockets, streaming, long-polling) - it seemed
      reasonable to focus on the two extremes (performance + compatibility).
      In any case client support for WebSockets is growing rapidly so fallback
      modes will become increasingly irrelevant while the extra simplicity will
      continue to pay dividends.

  General-use notes:
    * Single HTTP req+session persists over entire chsk session but cannot
      modify sessions! Use standard a/sync HTTP Ring req/resp for logins, etc.
    * Easy to wrap standard HTTP Ring resps for transport over chsks. Prefer
      this approach to modifying handlers (better portability)."

  {:author "Peter Taoussanis (@ptaoussanis)"}

       
  (:require
   [clojure.string     :as str]
   [clojure.core.async :as async  :refer (<! <!! >! >!! put! chan go go-loop)]
   [taoensso.encore    :as enc    :refer (swap-in! reset-in! swapped have have! have?)]
   [taoensso.timbre    :as timbre :refer (tracef debugf infof warnf errorf)]
   [taoensso.sente.interfaces :as interfaces])

        
           
                            
                                                        
                                                                         
                                                                
                                                                                
                                              

        
                  
                                                           )

(if (vector? taoensso.encore/encore-version)
  (enc/assert-min-encore-version [2 52 1])
  (enc/assert-min-encore-version  2.52))

;; (timbre/set-level! :trace) ; Uncomment for debugging

;;;; Events
;; Clients & server both send `event`s and receive (i.e. route) `event-msg`s:
;;   - `event`s have the same form client+server side,
;;   - `event-msg`s have a similar but not identical form

(defn- validate-event [x]
  (cond
    (not (vector? x))        :wrong-type
    (not (#{1 2} (count x))) :wrong-length
    :else (let [[ev-id _] x]
            (cond (not (keyword?  ev-id)) :wrong-id-type
                  (not (namespace ev-id)) :unnamespaced-id
                  :else nil))))

(defn event? "Valid [ev-id ?ev-data] form?" [x] (nil? (validate-event x)))

(defn as-event [x] (if (event? x) x [:chsk/bad-event x]))

(defn assert-event [x]
  (when-let [?err (validate-event x)]
    (let [err-msg
          (str
            (case ?err
              :wrong-type   "Malformed event (wrong type)."
              :wrong-length "Malformed event (wrong length)."
              (:wrong-id-type :unnamespaced-id)
              "Malformed event (`ev-id` should be a namespaced keyword)."
              :else "Malformed event (unknown error).")
            " Event should be of `[ev-id ?ev-data]` form: " x)]
      (throw (ex-info err-msg {:malformed-event x})))))

(defn client-event-msg? [x]
  (and
    (map? x)
    (enc/keys= x #{:ch-recv :send-fn :state :event :id :?data})
    (let [{:keys [ch-recv send-fn state event]} x]
      (and
        (enc/chan? ch-recv)
        (ifn?      send-fn)
        (enc/atom? state)
        (event?    event)))))

(defn server-event-msg? [x]
  (and
    (map? x)
    (enc/keys= x #{:ch-recv :send-fn :connected-uids
                   :ring-req :client-id
                   :event :id :?data :?reply-fn :uid})
    (let [{:keys [ch-recv send-fn connected-uids
                  ring-req client-id event ?reply-fn]} x]
      (and
        (enc/chan?       ch-recv)
        (ifn?            send-fn)
        (enc/atom?       connected-uids)
        (map?            ring-req)
        (enc/nblank-str? client-id)
        (event?          event)
        (or (nil? ?reply-fn)
            (ifn? ?reply-fn))))))

(defn- put-server-event-msg>ch-recv!
  "All server `event-msg`s go through this"
  [ch-recv {:as ev-msg :keys [event ?reply-fn]}]
  (let [[ev-id ev-?data :as valid-event] (as-event event)
        ev-msg* (merge ev-msg {:event     valid-event
                               :?reply-fn ?reply-fn
                               :id        ev-id
                               :?data     ev-?data})]
    (if-not (server-event-msg? ev-msg*)
      (warnf "Bad ev-msg: %s" ev-msg) ; Log 'n drop
      (put! ch-recv ev-msg*))))

;;; Note that cb replys need _not_ be `event` form!
                                                                                                 
                                                                       

;;;; Packing
;; * Client<->server payloads are arbitrary Clojure vals (cb replies or events).
;; * Payloads are packed for client<->server transit.
;; * Packing includes ->str encoding, and may incl. wrapping to carry cb info.

(defn- unpack "prefixed-pstr->[clj ?cb-uuid]"
  [packer prefixed-pstr]
  (have? string? prefixed-pstr)
  (let [wrapped? (enc/str-starts-with? prefixed-pstr "+")
        pstr     (subs prefixed-pstr 1)
        clj
        (try
          (interfaces/unpack packer pstr)
          (catch       Throwable                 t
            (debugf "Bad package: %s (%s)" pstr t)
            [:chsk/bad-package pstr]))

        [clj ?cb-uuid] (if wrapped? clj [clj nil])
        ?cb-uuid (if (= 0 ?cb-uuid) :ajax-cb ?cb-uuid)]

    (tracef "Unpacking: %s -> %s" prefixed-pstr [clj ?cb-uuid])
    [clj ?cb-uuid]))

(defn- with-?meta [x ?m] (if (seq ?m) (with-meta x ?m) x))

(defn- pack "clj->prefixed-pstr"
  ([packer ?packer-meta clj]
   (let [pstr
         (str "-" ; => Unwrapped (no cb metadata)
           (interfaces/pack packer (with-?meta clj ?packer-meta)))]
     (tracef "Packing (unwrapped): %s -> %s" [?packer-meta clj] pstr)
     pstr))

  ([packer ?packer-meta clj ?cb-uuid]
   (let [;;; Keep wrapping as light as possible:
         ?cb-uuid    (if (= ?cb-uuid :ajax-cb) 0 ?cb-uuid)
         wrapped-clj (if ?cb-uuid [clj ?cb-uuid] [clj])
         pstr
         (str "+" ; => Wrapped (cb metadata)
           (interfaces/pack packer (with-?meta wrapped-clj ?packer-meta)))]
     (tracef "Packing (wrapped): %s -> %s" [?packer-meta clj ?cb-uuid] pstr)
     pstr)))

(deftype EdnPacker []
  interfaces/IPacker
  (pack   [_ x] (enc/pr-edn   x))
  (unpack [_ s] (enc/read-edn s)))

(def ^:private default-edn-packer (EdnPacker.))

(defn- coerce-packer [x]
  (if (enc/kw-identical? x :edn)
    default-edn-packer
    (have #(satisfies? interfaces/IPacker %) x)))

(comment
  (do
    (require '[taoensso.sente.packers.transit :as transit])
    (def ^:private default-transit-json-packer (transit/get-transit-packer)))

  (let [pack   interfaces/pack
        unpack interfaces/unpack
        data   {:a :A :b :B :c "hello world"}]

    (enc/qb 10000
      (let [pk default-edn-packer]          (unpack pk (pack pk data)))
      (let [pk default-transit-json-packer] (unpack pk (pack pk data))))))

;;;; Server API

(declare
  ^:private send-buffered-server-evs>ws-clients!
  ^:private send-buffered-server-evs>ajax-clients!)

(defn make-channel-socket-server!
  "Takes a web server adapter[1] and returns a map with keys:
    :ch-recv ; core.async channel to receive `event-msg`s (internal or from clients).
    :send-fn ; (fn [user-id ev] for server>user push.
    :ajax-post-fn                ; (fn [ring-req]) for Ring CSRF-POST + chsk URL.
    :ajax-get-or-ws-handshake-fn ; (fn [ring-req]) for Ring GET + chsk URL.
    :connected-uids ; Watchable, read-only (atom {:ws #{_} :ajax #{_} :any #{_}}).

  Common options:
    :user-id-fn        ; (fn [ring-req]) -> unique user-id for server>user push.
    :csrf-token-fn     ; (fn [ring-req]) -> CSRF token for Ajax POSTs.
    :handshake-data-fn ; (fn [ring-req]) -> arb user data to append to handshake evs.
    :ws-kalive-ms      ; Ping to keep a WebSocket conn alive if no activity
                       ; w/in given number of msecs.
    :lp-timeout-ms     ; Timeout (repoll) long-polling Ajax conns after given msecs.
    :send-buf-ms-ajax  ; [2]
    :send-buf-ms-ws    ; [2]
    :packer            ; :edn (default), or an IPacker implementation (experimental).

  [1] e.g. `taoensso.sente.server-adapters.http-kit/http-kit-adapter` or
           `taoensso.sente.server-adapters.immutant/immutant-adapter`.
      You must have the necessary web-server dependency in your project.clj and
      the necessary entry in your namespace's `ns` form.

  [2] Optimization to allow transparent batching of rapidly-triggered
      server>user pushes. This is esp. important for Ajax clients which use a
      (slow) reconnecting poller. Actual event dispatch may occur <= given ms
      after send call (larger values => larger batch windows)."

  [web-server-adapter ; Actually a server-ch-adapter, but that may be confusing
   & [{:keys [recv-buf-or-n ws-kalive-ms lp-timeout-ms
              send-buf-ms-ajax send-buf-ms-ws
              user-id-fn csrf-token-fn handshake-data-fn packer]
       :or   {recv-buf-or-n    (async/sliding-buffer 1000)
              ws-kalive-ms     (enc/ms :secs 25) ; < Heroku 55s timeout
              lp-timeout-ms    (enc/ms :secs 20) ; < Heroku 30s timeout
              send-buf-ms-ajax 100
              send-buf-ms-ws   30
              user-id-fn    (fn [ring-req] (get-in ring-req [:session :uid]))
              csrf-token-fn (fn [ring-req]
                              (or (get-in ring-req [:session :csrf-token])
                                  (get-in ring-req [:session :ring.middleware.anti-forgery/anti-forgery-token])
                                  (get-in ring-req [:session "__anti-forgery-token"])))
              handshake-data-fn (fn [ring-req] nil)
              packer :edn}}]]

  (have? enc/pos-int? send-buf-ms-ajax send-buf-ms-ws)
  (have? #(satisfies? interfaces/IServerChanAdapter %) web-server-adapter)

  (let [packer  (coerce-packer packer)
        ch-recv (chan recv-buf-or-n)

        conns_          (atom {:ws  {} :ajax  {}}) ; {<uid> {<client-id> <server-ch>}}
        send-buffers_   (atom {:ws  {} :ajax  {}}) ; {<uid> [<buffered-evs> <#{ev-uuids}>]}
        connected-uids_ (atom {:ws #{} :ajax #{} :any #{}})

        ;; Used for ws-kalive, lp-timeout, etc.:
        last-udts_      (atom {}) ; {<client-id> <udt-last-touched>}

        upd-last-udt!
        (fn self
          ([client-id    ] (self client-id (enc/now-udt)))
          ([client-id udt]
           (swap-in! last-udts_ [client-id]
             (fn [?udt]
               (let [init? (nil? ?udt)]
                 (if (or init? (> ^long udt ^long ?udt))
                   (swapped  udt {:init? init? :udt  udt :swapped? true})
                   (swapped ?udt {:init? init? :udt ?udt :swapped? false})))))))

        user-id-fn
        (fn [ring-req client-id]
          ;; Allow uid to depend (in part or whole) on client-id. Be cautious
          ;; of security implications.
          (or (user-id-fn (assoc ring-req :client-id client-id)) ::nil-uid))

        connect-uid!
        (fn [conn-type uid] {:pre [(have? uid)]}
          (let [newly-connected?
                (swap-in! connected-uids_ []
                  (fn [{:keys [ws ajax any] :as old-m}]
                    (let [new-m
                          (case conn-type
                            :ws   {:ws (conj ws uid) :ajax ajax            :any (conj any uid)}
                            :ajax {:ws ws            :ajax (conj ajax uid) :any (conj any uid)})]
                      (swapped new-m
                        (let [old-any (:any old-m)
                              new-any (:any new-m)]
                          (when (and (not (contains? old-any uid))
                                          (contains? new-any uid))
                            :newly-connected))))))]
            newly-connected?))

        upd-connected-uid! ; Useful for atomic disconnects
        (fn [uid] {:pre [(have? uid)]}
          (let [newly-disconnected?
                (swap-in! connected-uids_ []
                  (fn [{:keys [ws ajax any] :as old-m}]
                    (let [conns' @conns_
                          any-ws-clients?   (contains? (:ws   conns') uid)
                          any-ajax-clients? (contains? (:ajax conns') uid)
                          any-clients?      (or any-ws-clients?
                                                any-ajax-clients?)
                          new-m
                          {:ws   (if any-ws-clients?   (conj ws   uid) (disj ws   uid))
                           :ajax (if any-ajax-clients? (conj ajax uid) (disj ajax uid))
                           :any  (if any-clients?      (conj any  uid) (disj any  uid))}]
                      (swapped new-m
                        (let [old-any (:any old-m)
                              new-any (:any new-m)]
                          (when (and      (contains? old-any uid)
                                     (not (contains? new-any uid)))
                            :newly-disconnected))))))]
            newly-disconnected?))

        send-fn ; server>user (by uid) push
        (fn [user-id ev & [{:as opts :keys [flush?]}]]
          (let [uid (if (= user-id :sente/all-users-without-uid) ::nil-uid user-id)
                _   (tracef "Chsk send: (->uid %s) %s" uid ev)
                _   (assert uid
                    (str "Support for sending to `nil` user-ids has been REMOVED. "
                         "Please send to `:sente/all-users-without-uid` instead."))
                _   (assert-event ev)

                ev-uuid (enc/uuid-str)

                flush-buffer!
                (fn [conn-type]
                  (when-let
                      [pulled
                       (swap-in! send-buffers_ [conn-type]
                         (fn [m]
                           ;; Don't actually flush unless the event buffered
                           ;; with _this_ send call is still buffered (awaiting
                           ;; flush). This means that we'll have many (go
                           ;; block) buffer flush calls that'll noop. They're
                           ;; cheap, and this approach is preferable to
                           ;; alternatives like flush workers.
                           (let [[_ ev-uuids] (get m uid)]
                             (if (contains? ev-uuids ev-uuid)
                               (swapped (dissoc m uid)
                                        (get    m uid))
                               (swapped m nil)))))]

                    (let [[buffered-evs ev-uuids] pulled]
                      (have? vector? buffered-evs)
                      (have? set?    ev-uuids)

                      (let [packer-metas         (mapv meta buffered-evs)
                            combined-packer-meta (reduce merge {} packer-metas)
                            buffered-evs-ppstr   (pack packer
                                                   combined-packer-meta
                                                   buffered-evs)]
                        (tracef "buffered-evs-ppstr: %s (with meta %s)"
                          buffered-evs-ppstr combined-packer-meta)

                        (case conn-type
                          :ws
                          (send-buffered-server-evs>ws-clients! conns_
                            uid buffered-evs-ppstr upd-last-udt!)

                          :ajax
                          (send-buffered-server-evs>ajax-clients! conns_
                            uid buffered-evs-ppstr))))))]

            (if (= ev [:chsk/close]) ; Currently undocumented
              (do
                (debugf "Chsk closing (client may reconnect): %s" uid)
                (when flush?
                  (flush-buffer! :ws)
                  (flush-buffer! :ajax))

                (doseq [server-ch (vals (get-in @conns_ [:ws uid]))]
                  (interfaces/sch-close! server-ch))

                (doseq [server-ch (vals (get-in @conns_ [:ajax uid]))]
                  (interfaces/sch-close! server-ch)))

              (do
                ;; Buffer event
                (doseq [conn-type [:ws :ajax]]
                  (swap-in! send-buffers_ [conn-type uid]
                    (fn [?v]
                      (if-not ?v
                        [[ev] #{ev-uuid}]
                        (let [[buffered-evs ev-uuids] ?v]
                          [(conj buffered-evs ev)
                           (conj ev-uuids     ev-uuid)])))))

                ;;; Flush event buffers after relevant timeouts:
                ;; * May actually flush earlier due to another timeout.
                ;; * We send to _all_ of a uid's connections.
                ;; * Broadcasting is possible but I'd suggest doing it rarely, and
                ;;   only to users we know/expect are actually online.
                (go (when-not flush? (<! (async/timeout send-buf-ms-ws)))
                    (flush-buffer! :ws))
                (go (when-not flush? (<! (async/timeout send-buf-ms-ajax)))
                    (flush-buffer! :ajax)))))

          ;; Server-side send is async so nothing useful to return (currently
          ;; undefined):
          nil)

        ev-msg-const
        {:ch-recv        ch-recv
         :send-fn        send-fn
         :connected-uids connected-uids_}]

    {:ch-recv        ch-recv
     :send-fn        send-fn
     :connected-uids connected-uids_

     :ajax-post-fn ; Does not participate in `conns_` (has specific req->resp)
     (fn [ring-req]
       (interfaces/ring-req->server-ch-resp web-server-adapter ring-req
         {:on-open
          (fn [server-ch]
            (let [params        (get ring-req :params)
                  ppstr         (get params   :ppstr)
                  client-id     (get params   :client-id)
                  [clj has-cb?] (unpack packer ppstr)
                  reply-fn
                  (let [replied?_ (atom false)]
                    (fn [resp-clj] ; Any clj form
                      (if (compare-and-set! replied?_ false true)
                        (do
                          (tracef "Chsk send (ajax post reply): %s" resp-clj)
                          (interfaces/-sch-send! server-ch
                            (pack packer (meta resp-clj) resp-clj)
                            :close-after-send))
                        (debugf "Ajax reply noop (already replied): %s" resp-clj))))]

              (put-server-event-msg>ch-recv! ch-recv
                (merge ev-msg-const
                  {;; Note that the client-id is provided here just for the
                   ;; user's convenience. non-lp-POSTs don't actually need a
                   ;; client-id for Sente's own implementation:
                   :client-id client-id #_"unnecessary-for-non-lp-POSTs"
                   :ring-req  ring-req
                   :event     clj
                   :uid       (user-id-fn ring-req client-id)
                   :?reply-fn (when has-cb? reply-fn)}))

              (if has-cb?
                (when-let [ms lp-timeout-ms]
                  (go
                    (<! (async/timeout ms))
                    (reply-fn :chsk/timeout)))
                (reply-fn :chsk/dummy-cb-200))))}))

     :ajax-get-or-ws-handshake-fn ; Ajax handshake/poll, or WebSocket handshake
     (fn [ring-req]
       (let [csrf-token (csrf-token-fn ring-req)
             params     (get ring-req :params)
             client-id  (get params   :client-id)
             uid        (user-id-fn ring-req client-id)
             websocket? (:websocket? ring-req)
             sch-uuid_  (delay (enc/uuid-str 6))

             receive-event-msg! ; Partial
             (fn self
               ([event          ] (self event nil))
               ([event ?reply-fn]
                (put-server-event-msg>ch-recv! ch-recv
                  (merge ev-msg-const
                    {:client-id client-id
                     :ring-req  ring-req
                     :event     event
                     :?reply-fn ?reply-fn
                     :uid       uid}))))

             handshake!
             (fn [server-ch]
               (tracef "Handshake!")
               (let [?handshake-data (handshake-data-fn ring-req)
                     handshake-ev
                     (if-not (nil? ?handshake-data) ; Micro optimization
                       [:chsk/handshake [uid csrf-token ?handshake-data]]
                       [:chsk/handshake [uid csrf-token]])]
                 (interfaces/-sch-send! server-ch
                   (pack packer nil handshake-ev)
                   (not websocket?))))]

         (if (str/blank? client-id)
           (let [err-msg "Client's Ring request doesn't have a client id. Does your server have the necessary keyword Ring middleware (`wrap-params` & `wrap-keyword-params`)?"]
             (errorf (str err-msg ": %s") ring-req) ; Careful re: % in req
             (throw (ex-info err-msg {:ring-req ring-req})))

           (interfaces/ring-req->server-ch-resp web-server-adapter ring-req
             {:on-open
              (fn [server-ch]
                (if websocket?

                  ;; WebSocket handshake
                  (let [udt-open (enc/now-udt)]
                    (tracef "New WebSocket channel: %s (%s)" uid @sch-uuid_)
                    (reset-in! conns_ [:ws uid client-id] server-ch)
                    (upd-last-udt! client-id udt-open)
                    (when (connect-uid! :ws uid)
                      (receive-event-msg! [:chsk/uidport-open]))

                    (handshake! server-ch)

                    ;; Start ws-kalive loop
                    ;; This also works to gc ws conns that were suddenly
                    ;; terminated (e.g. by turning on airplane mode)
                    (when-let [ms ws-kalive-ms]
                      (go-loop [udt-t0 udt-open]
                        (<! (async/timeout ms))
                        (when-let [udt-t1 (get @last-udts_ client-id)]
                          (when (interfaces/sch-open? server-ch)
                            (when (= udt-t1 udt-t0)
                              ;; We've seen no send/recv activity on this
                              ;; conn w/in our kalive window so send a ping
                              ;; ->client (should auto-close conn if it's
                              ;; gone dead)
                              (interfaces/-sch-send! server-ch
                                (pack packer nil :chsk/ws-ping)
                                (not :close-after-send)))
                            (recur udt-t1))))))

                  ;; Ajax handshake/poll
                  (let [udt-open (enc/now-udt)
                        _ (tracef "New Ajax handshake/poll: %s (%s)" uid @sch-uuid_)
                        _ (reset-in! conns_ [:ajax uid client-id] server-ch)

                        handshake?
                        (or (:init? (upd-last-udt! client-id udt-open))
                            (:handshake? params))]

                    (when (connect-uid! :ajax uid)
                      (receive-event-msg! [:chsk/uidport-open]))

                    (if handshake?
                      ;; Client will immediately repoll
                      (handshake! server-ch)

                      (when-let [ms lp-timeout-ms]
                        (go
                          (<! (async/timeout ms))
                          (let [udt-t1 (get @last-udts_ client-id)]
                            (when (= udt-t1 udt-open)
                              ;; Appears to still be the active sch
                              (interfaces/-sch-send! server-ch
                                (pack packer nil :chsk/timeout)
                                :close-after-send)))))))))

              :on-msg ; Only for WebSockets
              (fn [server-ch req-ppstr]
                (upd-last-udt! client-id (enc/now-udt))
                (let [[clj ?cb-uuid] (unpack packer req-ppstr)]
                  (receive-event-msg! clj ; Should be ev
                    (when ?cb-uuid
                      (fn reply-fn [resp-clj] ; Any clj form
                        (tracef "Chsk send (ws reply): %s" resp-clj)
                        ;; true iff apparent success:
                        (interfaces/-sch-send! server-ch
                          (pack packer (meta resp-clj) resp-clj ?cb-uuid)
                          (not :close-after-send)))))))

              :on-close ; We rely on `on-close` to trigger for _every_ conn!
              (fn [server-ch status]
                ;; `status` is currently unused; its form varies depending on
                ;; the underlying web server

                (swap-in! conns_ [(if websocket? :ws :ajax) uid]
                  (fn [?m]
                    (let [new-m (dissoc ?m client-id)]
                      (if (empty? new-m) :swap/dissoc new-m))))

                ;; Allow some time for possible reconnects (repoll,
                ;; sole window refresh, etc.):
                (let [udt-close (enc/now-udt)]
                  (upd-last-udt! client-id udt-close)
                  (go
                    (<! (async/timeout 5000))
                    (let [udt-t1 (get @last-udts_ client-id)]
                      (when (= udt-t1 udt-close)
                        (let [disconnect?
                              (swap-in! last-udts_ [client-id]
                                (fn [udt-t1]
                                  (if (= udt-t1 udt-close)
                                    (swapped :swap/dissoc true)
                                    (swapped udt-t1       false))))]
                          (when (and disconnect? (upd-connected-uid! uid))
                            (receive-event-msg! [:chsk/uidport-close]))))))))}))))}))

(defn- send-buffered-server-evs>ws-clients!
  "Actually pushes buffered events (as packed-str) to all uid's WebSocket conns."
  [conns_ uid buffered-evs-pstr upd-last-udt!]
  (tracef "send-buffered-server-evs>ws-clients!: %s" buffered-evs-pstr)
  (doseq [[client-id server-ch] (get-in @conns_ [:ws uid])]
    (upd-last-udt! client-id (enc/now-udt))
    (interfaces/-sch-send! server-ch buffered-evs-pstr
      (not :close-after-send))))

(defn- send-buffered-server-evs>ajax-clients!
  "Actually pushes buffered events (as packed-str) to all uid's Ajax conns.
  Allows some time for possible Ajax poller reconnects."
  [conns_ uid buffered-evs-pstr & [{:keys [nmax-attempts ms-base ms-rand]
                                    ;; <= 7 attempts at ~135ms ea = 945ms
                                    :or   {nmax-attempts 7
                                           ms-base       90
                                           ms-rand       90}}]]
  (comment (* 7 (+ 90 (/ 90 2.0))))
  (tracef "send-buffered-server-evs>ajax-clients!: %s" buffered-evs-pstr)
  (let [;; All connected/possibly-reconnecting client uuids:
        client-ids-unsatisfied (keys (get-in @conns_ [:ajax uid]))]
    (when-not (empty? client-ids-unsatisfied)
      ;; (tracef "client-ids-unsatisfied: %s" client-ids-unsatisfied)
      (go-loop [n 0 client-ids-satisfied #{}]
        (let [?pulled ; nil or {<client-id> <server-ch>}
              (swap-in! conns_ [:ajax uid]
                (fn [m] ; {<client-id> <server-ch>]}
                  (let [ks-to-pull (remove client-ids-satisfied (keys m))]
                    ;; (tracef "ks-to-pull: %s" ks-to-pull)
                    (if (empty? ks-to-pull)
                      (swapped m nil)
                      (swapped
                        (reduce (fn [m k] (dissoc m k)) m ks-to-pull)
                        (select-keys m ks-to-pull))))))]

          (have? [:or nil? map?] ?pulled)

          (let [?newly-satisfied
                (when ?pulled
                  (reduce-kv
                    (fn [s client-id server-ch]
                      (let [;; server-ch may have closed already (`send!` will noop):
                            sent-now? (interfaces/-sch-send! server-ch
                                        buffered-evs-pstr :close-after-send)]
                        (if sent-now?
                          (conj s client-id)
                          s)))
                    #{} ?pulled))

                now-satisfied (into client-ids-satisfied ?newly-satisfied)]

            ;; (tracef "now-satisfied: %s" now-satisfied)
            (when (and (< n nmax-attempts)
                       (some (complement now-satisfied) client-ids-unsatisfied))
              ;; Allow some time for possible poller reconnects:
              (<! (async/timeout (+ ms-base (rand-int ms-rand))))
              (recur (inc n) now-satisfied))))))))

;;;; Client API

                                                                           
      
                      
                            
                                    
                            
                                     

                                                                        
                                                                              
                                                                              
      
                                                                            
                                 

      
                
                                                                         
                                                     
                                                                         
                                                                    
                 
                                                                         
                               

      
                                  
                                          
                                     
        

      
                                           
                  
                                                 
                                                
                                                                                     
                                                    
                                                                 

      
                                                 
                              
                                    
                                            

      
                                                                    
                             
                           
                         
                                                         

                                                                           
                                                                    
                               
                                      
                                                                
                                            
                                             

                           
                                          
                                 
                                                              
                                                            
                                                              

                                                           

                                    
                                                   
                                                 
                  

      
                    
                                                                         
                                                             
                           
          
                                
       
       
                           
                       
                        
                      
                   
                     
                                                      
                         

      
                                      
                                          
                                        
                            
                       
                                                              
                                                           
                                 

      
                                                  
                                      
                                                                    
                                                         

                                            
                                                    

                    
                                                      
                                            
                                                                        
                     
                                                                        
                                 
                                 
                                 
                                        
                                            
                                              

                        
                            
                                                                  

                                   
                                      
                                                                               

                                          
                                           

                    

      
                      
                                            
                                       

                                  
                                                   
                                              
                                           
                                      
                  

           
                                   
                                                   
               
                                                                                
                                              
                                                           

                          
                                    
                             

                             
                                                               
                                                 
                                        
                                                  
                                   

                                                           
                                                     
                                                           

                                      
                                                            
                                              
                                                 
                                                                               
                                              

              
                                  
                             
                             
                                          
                                          
                                                                          
                                                
                                        
                         

                        
                                                             
                                                                  
                                   
                      
                             
                            
                          
                                                          
                                                                   
                                                                        
                                                                                        
                                                                          

                           
                        
                                 
                                                            
                                                                    
                                                     
                                       
                                                       
                              

                               
                                                         

                                 
                                 
                                     
                                   
                                                              
                                                    
                                                                       
                                                          
                               

                                                                         
                                   
                                                                                   
                                                                                
                                                                                  
                                                  
                                                             
                                                                     
                                                        
                               
                                                          
                                                                        
                                                           
                                                        
                                                                               
                                                          
                                             
                                                                             
                                                       
                                                                                

                                      
                                    
                                                                        
                                                                   
                            

                                                                            
                                        
                                     
                                   
                                                              
                                                                
                                                                   

                                                                       
                                                         
                                      
                                                                                          
                                 
                                                                       
                                                     

                                          
                               
                    
                

      
                             
                   
          
                                                        
                                          
                                 
                                     
                                  
                                    
             

      
                       
                                       
                              

                                         
                                 
                
                       

           
                                   
                                                   
               
                                                                                
                                              
                                         

                          
                                    
                             

                             
                                                               
                                                 
                                        
                                                  
                                   

                                                           
                                               
                        
                            
                                                    
                                                                
                       
                                                                       
                                            

                      
                                                                             
                                                           
                                                                    

                                                                             
                                                                     
                                           

                                                                           
                                                                                
                                          

                                           

                                                  
                       
                                      
                                                     
                                                            
                                                         

                                          
                                         
                                                             
                           
                                    
                                                           
                                                                      
                                                           

                               

                        
                                 
                                            
                                   
                                             
                          
                        
                                                        
                                                          
                                                                      
                                                                                      
                                                                                             

                               
                              
                                  
                                                          
                                                                         
                            
                           

                                                                                 
                                                                               
                                                                     
                                                          

                                                                       
                                              

                                                                          
                                                                      
                                                                           
                                      
                                                                        

                                                         
                              
                           
                                                       
                                                                                 
                             
                                                                   
                                        

                                                                                 
                                            
                                           
                                                          
                           
                                                                    
                                                                 
                                                                        
                                                                      

                                                              
                                           

                                        
                 
             

      
                              
                    
          
                                                          
                                          
                                     
                                    
             

      
                       
                                                    
                                               

                              
                                                   
                                       
    

           
                                   
                           
                                        

                              
                          
                           
                                      
                                

                             
                         
                                
                          
                                          
                                      

                        
                                                              
                                                                               

                                                   
                    
                                                                         
                                                                            
                                                           
                                      
                                                                     
                                              
                                              

                                                                            
                                                                           

                                                 
             

      
                              
                    
          
                                                
                          
             

      
                                             
                           
                                 
                                                                  
                                              

      
                                 
                                              
                                                                           
                                                                                
                                                                          
                                                                                   
                                                               

                 
                                                                                     
                                                                   
                                                                             
                                                       
                                                                                 
                                                                           
                                                                              

         
                                                 
                                                             
               
                               
                                                                               
                              
                                                                           
                                             

                                                                            
                                                                              
                               

                                            

                           

                                       
                                   

                                                                                                                            
                                                                                                            

                                     

                                         
                                         
                                          
                                          

                         
                                                    
                                                       
                                                    
                                                        

                   
                                                   
                                                   
                                                    

                        
                             
                               
                          
                           

                    
                               
                                
                                         

                      
                               
                                  
                                   
                                         

                      
                                       
                                        

             
                       
                    
                                                   
                                                   
                                                      

                        
                                             

                                                                           

                 
                        
                                                        
                                                        
                                                        
                                   
                                                                    
                                
                            

                     
                       
                                                                               
                                                                             
                                 
                                                            
                                 
                                   
                                          
                              
                                 
                                       
                     

                      
                                                        
                         
                                  

                                                  

;;;; Event-msg routers (handler loops)

(defn- -start-chsk-router!
  [server? ch-recv event-msg-handler opts]
  (let [{:keys [trace-evs? error-handler]} opts
        ch-ctrl (chan)]

    (go-loop []
      (let [[v p] (async/alts! [ch-recv ch-ctrl])
            stop? (enc/kw-identical? p  ch-ctrl)]

        (when-not stop?
          (let [{:as event-msg :keys [event]} v
                [_ ?error]
                (enc/catch-errors
                  (when trace-evs? (tracef "Pre-handler event: %s" event))
                  (event-msg-handler
                    (if server?
                      (have! server-event-msg? event-msg)
                      (have! client-event-msg? event-msg))))]

            (when-let [e ?error]
              (let [[_ ?error2]
                    (enc/catch-errors
                      (if-let [eh error-handler]
                        (error-handler e event-msg)
                        (errorf e "Chsk router `event-msg-handler` error: %s" event)))]
                (when-let [e2 ?error2]
                  (errorf e2 "Chsk router `error-handler` error: %s" event))))

            (recur)))))

    (fn stop! [] (async/close! ch-ctrl))))

(defn start-server-chsk-router!
  "Creates a go-loop to call `(event-msg-handler <server-event-msg>)` and
  log any errors. Returns a `(fn stop! [])`.

  For performance, you'll likely want your `event-msg-handler` fn to be
  non-blocking (at least for slow handling operations). Clojure offers
  a rich variety of tools here including futures, agents, core.async,
  etc.

  Advanced users may also prefer to write their own loop against `ch-recv`."
  [ch-recv event-msg-handler & [{:as opts :keys [trace-evs? error-handler]}]]
  (-start-chsk-router! :server ch-recv event-msg-handler opts))

(defn start-client-chsk-router!
  "Creates a go-loop to call `(event-msg-handler <client-event-msg>)` and
  log any errors. Returns a `(fn stop! [])`.

  For performance, you'll likely want your `event-msg-handler` fn to be
  non-blocking (at least for slow handling operations). Clojure offers
  a rich variety of tools here including futures, agents, core.async,
  etc.

  Advanced users may also prefer to write their own loop against `ch-recv`."
  [ch-recv event-msg-handler & [{:as opts :keys [trace-evs? error-handler]}]]
  (-start-chsk-router! (not :server) ch-recv event-msg-handler opts))

;;;; Platform aliases

(def event-msg?       server-event-msg?                         )

(def make-channel-socket!
         make-channel-socket-server!
                                    )

(def start-chsk-router!
         start-server-chsk-router!
                                  )

;;;; Deprecated

     
(defn start-chsk-router-loop!
  "DEPRECATED: Please use `start-chsk-router!` instead"
  [event-msg-handler ch-recv]
  (start-server-chsk-router! ch-recv
    ;; Old handler form: (fn [ev-msg ch-recv])
    (fn [ev-msg] (event-msg-handler ev-msg (:ch-recv ev-msg)))))

      
                             
                                                       
                         
                                    
                                          
                                                                     

(def set-logging-level! "DEPRECATED. Please use `timbre/set-level!` instead" timbre/set-level!)

                                                                                 
      
                                     
                                                                     
                  
                        
                                                   
                      
                                                    

;;;;;;;;;;;; This file autogenerated from src/taoensso/sente.cljx
