(ns taoensso.sente.packers.impl.gzip)

(def ^:private ^:const prefix-raw  0x00)
(def ^:private ^:const prefix-gzip 0x01)
(def ^:private ^:const min-bytes   1024)

(defn- type-name [x]
  (let [ctor (type x)]
    (or
      (.-name ctor) (.-displayName ctor) (goog/typeOf x)
      (try (pr-str ctor) (catch :default _ nil))
      "<unknown>")))

(defn- as-u8s
  "#{Uint8Array ArrayBuffer DataView} -> Uint8Array"
  ^js [input]
  (cond
    (instance? js/Uint8Array  input)                 input
    (instance? js/ArrayBuffer input) (js/Uint8Array. input)
    (instance? js/DataView    input) (js/Uint8Array. (.-buffer     input)
                                                     (.-byteOffset input)
                                                     (.-byteLength input))
    :else
    (throw
      (ex-info "Unexpected input type"
        {:type (type-name input),
         :expected '#{Uint8Array ArrayBuffer DataView}}))))

(defn- prefix
  "Returns given Uint8Array with added flag prefix."
  [gzip? ^js u8s]
  (let [out (js/Uint8Array. (inc (.-length u8s)))]
    (aset out 0 (if gzip? prefix-gzip prefix-raw))
    (.set out u8s 1)
    (do   out)))

(defn- stream->u8s [readable] (-> (js/Response. readable) (.arrayBuffer) (.then #(js/Uint8Array. %))))

(defn gzip
  "Uncompressed Uint8Array -> compressed Uint8Array Promise."
  [^js u8s]
  (if (< (.-length u8s) min-bytes)
    (js/Promise.resolve (prefix false u8s))
    (let [cs     (js/CompressionStream. "gzip")
          writer (.getWriter (.-writable cs))]
      (->
        (.write writer u8s)
        (.then (fn [] (.close writer) (stream->u8s (.-readable cs))))
        (.then (fn [u8s] (prefix true u8s)))))))

(defn gunzip
  "Compressed Uint8Array -> uncompressed Uint8Array Promise."
  [^js u8s]
  (let [flag (aget u8s 0)
        body (.subarray u8s 1)]
    (if-not (== flag prefix-gzip)
      (js/Promise.resolve body)
      (let [ds     (js/DecompressionStream. "gzip")
            writer (.getWriter (.-writable ds))]
        (->
          (.write writer body)
          (.then (fn [] (.close writer) (stream->u8s (.-readable ds)))))))))
