(ns taoensso.sente.server-adapters.undertow
  "Sente server adapter for ring-undertow-adapter."
  {:author "Nik Peric"}
  (:require
    [ring.adapter.undertow.websocket :as websocket]
    [ring.adapter.undertow.response  :as response]
    [taoensso.sente.interfaces :as i])
  (:import
    [io.undertow.websockets.core WebSocketChannel]
    [io.undertow.server HttpServerExchange]
    [io.undertow.websockets
     WebSocketConnectionCallback
     WebSocketProtocolHandshakeHandler]))

;; Websocket
(extend-type WebSocketChannel
  i/IServerChan
  (sch-open?  [this] (.isOpen    this))
  (sch-close! [this] (.sendClose this))
  (sch-send!  [this websocket? msg] (websocket/send msg this)))

(extend-protocol response/RespondBody
  WebSocketConnectionCallback
  (respond [body ^HttpServerExchange exchange]
    (let [handler (WebSocketProtocolHandshakeHandler. body)]
      (.handleRequest handler exchange))))

(defn- ws-ch
  [{:keys [on-open on-close on-msg on-error]} _adapter-opts]
  (websocket/ws-callback
    {:on-open          (when on-open  (fn [{:keys [channel]}]         (on-open  channel true)))
     :on-error         (when on-error (fn [{:keys [channel error]}]   (on-error channel true error)))
     :on-message       (when on-msg   (fn [{:keys [channel data]}]    (on-msg   channel true data)))
     :on-close-message (when on-close (fn [{:keys [channel message]}] (on-close channel true message)))}))

;; AJAX
(defprotocol ISenteUndertowAjaxChannel
  (send!  [this msg])
  (read!  [this])
  (close! [this]))

(deftype SenteUndertowAjaxChannel [resp-promise_ open?_ on-close adapter-opts]
  ISenteUndertowAjaxChannel
  (send!  [this msg] (deliver resp-promise_ msg))
  (read!  [this]
    (let [{:keys [ajax-resp-timeout-ms ajax-resp-timeout-val]} adapter-opts
          resp
          (if ajax-resp-timeout-ms
            (deref resp-promise_ ajax-resp-timeout-ms ajax-resp-timeout-val)
            (deref resp-promise_))]

      (close! this)
      resp))

  (close! [this]
    (when (compare-and-set! open?_ true false)
      (when on-close (on-close this false nil))
      true))

  i/IServerChan
  (sch-send!  [this websocket? msg] (send! this msg))
  (sch-open?  [this] @open?_)
  (sch-close! [this] (close! this)))

(defn- ajax-ch [{:keys [on-open on-close]} adapter-opts]
  (let [open?_  (atom true)
        channel (SenteUndertowAjaxChannel. (promise) open?_ on-close adapter-opts)]
    (when on-open (on-open channel false))
    channel))

(extend-protocol response/RespondBody
  SenteUndertowAjaxChannel
  (respond [body ^HttpServerExchange exchange]
    (response/respond (read! body) exchange)))

;; Adapter
(deftype UndertowServerChanAdapter [adapter-opts]
  i/IServerChanAdapter
  (ring-req->server-ch-resp [sch-adapter ring-req callbacks-map]
    ;; Returns {:body <websocket-implementation-channel> ...}:

    ;; TODO callbacks (sch to impl IServerChan)
    ;; on-open  [server-ch ws?]
    ;; on-close [server-ch ws? status]
    ;; on-msg   [server-ch ws? msg]
    ;; on-error [server-ch ws? error]
    {:body
     (if (:websocket? ring-req)
       (ws-ch   callbacks-map adapter-opts)
       (ajax-ch callbacks-map adapter-opts))}))

(defn get-sch-adapter
  "Returns an Undertow ServerChanAdapter.
  Options:
    :ajax-resp-timeout-ms  ; Max msecs to wait for Ajax responses
    :ajax-resp-timeout-val ; Value returned in case of above timeout"

  ([] (UndertowServerChanAdapter. nil))
  ([{:as adapter-opts
     :keys [ajax-resp-timeout-ms
            ajax-resp-timeout-val]}]
   (UndertowServerChanAdapter. adapter-opts)))

;; TODO :or {ajax-resp-timeout-val :undertow/ajax-resp-timeout}
