(ns taoensso.sente
  "Channel sockets. Otherwise known as The Shiz.

      Protocol  | client>server | client>server + ack/reply | server>clientS[1] push
    * WebSockets:       ✓              [2]                          ✓  ; [3]
    * Ajax:            [4]              ✓                          [5] ; [3]

    [1] All of a user's (uid's) connected clients (browser tabs, devices, etc.).
    [2] Emulate with cb-uuid wrapping.
    [3] By uid only (=> logged-in users only).
    [4] Emulate with dummy-cb wrapping.
    [5] Emulate with long-polling against uid (=> logged-in users only).

  Special messages (implementation detail):
    * cb replies: :chsk/closed, :chsk/timeout, :chsk/error.
    * client-side events:
        [:chsk/handshake <#{:ws :ajax}>],
        [:chsk/ping      <#{:ws :ajax}>], ; Though no :ajax ping
        [:chsk/state [<#{:open :first-open :closed}> <#{:ws :ajax}]],
        [:chsk/recv  <`server>client`-event>]. ; Async event

    * server-side events:
       [:chsk/bad-edn <edn>],
       [:chsk/bad-event <chsk-event>],
       [:chsk/uidport-open  <#{:ws :ajax}>],
       [:chsk/uidport-close <#{:ws :ajax}>].

    * event wrappers: {:chsk/clj <clj> :chsk/dummy-cb? true} (for [2]),
                      {:chsk/clj <clj> :chsk/cb-uuid <uuid>} (for [4]).

  Implementation notes:
    * A server>client w/cb mechanism would be possible BUT:
      * No fundamental use cases. We can always simulate as server>client w/o cb,
        client>server w or w/o cb.
      * Would yield a significantly more complex code base.
      * Cb semantic is fundamentally incongruous with server>client since
        multiple clients may be connected simultaneously for a single uid.

  General-use notes:
    * Single HTTP req+session persists over entire chsk session but cannot
      modify sessions! Use standard a/sync HTTP Ring req/resp for logins, etc.
    * Easy to wrap standard HTTP Ring resps for transport over chsks. Prefer
      this approach to modifying handlers (better portability).

  Multiple clients (browser tabs, devices, etc.):
    * client>server + ack/reply: sends always to _single_ client. Note that an
      optional _multi_ client reply API wouldn't make sense (we're using a cb).
    * server>clientS push: sends always to _all_ clients.
    * Applications will need to be careful about which method is preferable, and
      when."
  {:author "Peter Taoussanis"}

       
  (:require [clojure.string :as str]
            [clojure.core.async :as async :refer (<! <!! >! >!! put! chan
                                                     go go-loop)]
            [clojure.tools.reader.edn :as edn]
            [org.httpkit.server       :as http-kit]
            [taoensso.encore          :as encore]
            [taoensso.timbre          :as timbre])

        
                                    
                                                                
                                     
                                         

        
                                                                           )

;;;; TODO
;; * Docs, examples, README, testing (esp. new Ajax stuff).
;; * No need/desire for a send buffer, right? Would seem to violate user
;;   expectations (either works now or doesn't, not maybe works later).
;; * Consider later using clojure.browser.net (Ref. http://goo.gl/sdS5wX)
;;   and/or Google Closure for some or all basic WebSockets support,
;;   reconnects, etc.

;;;; Shared (client+server)

(defn- chan? [x]
         (instance? clojure.core.async.impl.channels.ManyToManyChannel x)
                                                                         )

(defn event? "Valid [ev-id ?ev-data] form?" [x]
  (and (vector? x) (#{1 2} (count x))
       (let [[ev-id _] x]
         (and (keyword? ev-id) (namespace ev-id)))))

(defn cb-success? [cb-reply] ;; Cb reply need _not_ be `event` form!
  (not (#{:chsk/closed :chsk/timeout :chsk/error} cb-reply)))

      ; For #+cljs we'd rather just throw client-side on bad edn from server
(defn- try-read-edn [edn]
  (try (edn/read-string edn)
       (catch Throwable t [:chsk/bad-edn edn])))

(defn- unwrap-edn-msg-with-?cb->clj "edn -> [clj ?cb-uuid]"
  [edn]
  (let [msg             (try-read-edn    edn)
                                             
        ?cb-uuid (and (map? msg) (:chsk/cb-uuid msg))
        clj      (if-not ?cb-uuid msg (:chsk/clj msg))]
    [clj ?cb-uuid]))

;;;; Server

     
(defn event-msg?
  "Valid {:client-uuid _ :ring-req _ :event _ :?reply-fn _} form?"
  [x]
  (and (map? x) (= (count x) 4)
       (every? #{:client-uuid :ring-req :event :?reply-fn} (keys x))
       (let [{:keys [client-uuid hk-ch ring-req event ?reply-fn]} x]
         (and (string? client-uuid) ; Set by client (Ajax) or server (WebSockets)
              (map? ring-req)
              (event? event)
              (or (nil? ?reply-fn) (ifn? ?reply-fn))))))

     
(defn- receive-event-msg!
  [ch-recv {:as ev-msg :keys [client-uuid ring-req event ?reply-fn]}]
  (let [ev-msg*
        {:client-uuid client-uuid ; Browser-tab / device identifier
         :ring-req    (select-keys ring-req [:t :locale :session #_:flash :params])
         :event       (if (event? event) event [:chsk/bad-event event])
         :?reply-fn
         (if (ifn? ?reply-fn) ?reply-fn
           (-> (fn [resp-clj] ; Dummy warn fn
                 (timbre/warnf "Trying to reply to non-cb req: %s" event))
               ;; Useful to distinguish between a real cb reply fn and dummy:
               (with-meta {:dummy-reply-fn? true})))}]

    (if (event-msg? ev-msg*) ; Be conservative about what we put to chan!
      (put! ch-recv ev-msg*)
      (timbre/warnf "Bad ev-msg!: %s (%s)" ev-msg* ev-msg))))

     
(defn- ch-pull-ajax-hk-chs!
  "Starts a go loop to pull relevant client hk-chs. Several attempts are made in
  order to provide some reliability against possibly-reconnecting Ajax pollers.
  Pulls at most one hk-ch per client-uuid so works fine with multiple clients.

  Returns a channel to which we'll send the hk-chs set, or close.

  More elaborate implementations (involving tombstones) could cut down on
  unnecessary waiting - but this solution is small, simple, and plenty fast in
  practice."
  [clients_ uid] ; `hk-chs` = http-kit channels
  (let [ch (chan)]
    (go-loop [pulled {} ; {<client-uuid> <hk-ch>}
              n      0]
      (if (= n 3) ; Try three times, always

        ;; >! set of unique-client hk-chs, or nil:
        (if (empty? pulled)
          (async/close! ch)
          (>! ch (set (vals pulled))))

        (let [?pulled-now ; nil or {<client-uuid> <hk-ch>}
              (first
               (swap! clients_
                 (fn [[_ m]]
                   (let [m-in       (get m uid)
                         ks-to-pull (filter #(not (contains? pulled %))
                                            (keys m-in))]
                     (if (empty? ks-to-pull) [nil m]
                       [(select-keys m ks-to-pull)
                        (assoc m uid (apply dissoc m-in ks-to-pull))])))))]

          ;; Allow some time for possible poller reconnects:
          (<! (async/timeout (+ 80 (rand-int 50)))) ; ~105ms
          (recur (merge pulled ?pulled-now) (inc n)))))
    ch))

(comment (time (dotimes [_ 50000] (ch-pull-ajax-hk-chs! (atom [nil {}]) 10))))

     
(defn make-channel-socket!
  "Returns `{:keys [ch-recv send-fn ajax-post-fn ajax-get-or-ws-handshake-fn]}`.

  ch-recv - core.async channel ; For server-side chsk request router, will
                               ; receive `event-msg`s from clients.
  send-fn - (fn [user-id ev])   ; For server>clientS push
  ajax-post-fn                - (fn [ring-req]) ; For Ring POST, chsk URL
  ajax-get-or-ws-handshake-fn - (fn [ring-req]) ; For Ring GET, chsk URL (+CSRF)"
  [& [{:keys [recv-buf-or-n]
       :or   {recv-buf-or-n (async/sliding-buffer 1000)}}]]
  (let [ch-recv      (chan recv-buf-or-n)
        clients-ajax (atom [nil {}]) ; [<#{pulled-hk-chs}> {<uid> {<client-uuid> <hk-ch>}}]
        clients-ws   (atom {})       ; {<uid> <#{hk-chs}>}
        ]

    {:ch-recv ch-recv
     :send-fn ; Async server>clientS (by uid) push sender
     (fn [uid ev]
       (timbre/tracef "Chsk send: (->uid %s) %s" uid ev)
       (assert (event? ev))
       (when uid
         (let [send-to-hk-chs! ; Async because of http-kit
               (fn [hk-chs]
                 ;; Remember that http-kit's send to a closed ch is just a no-op
                 (when hk-chs ; No cb, so no need for (cb-fn :chsk/closed)
                   (assert (and (set? hk-chs) (not (empty? hk-chs))))
                   (if (= ev [:chsk/close])
                     (do (timbre/debugf "Chsk CLOSING: %s" uid)
                       (doseq [hk-ch hk-chs] (http-kit/close hk-ch)))

                     (let [ev-edn (pr-str ev)]
                       (->>
                        (for [hk-ch hk-chs] ; Broadcast to all uid's clients/devices
                          (http-kit/send! hk-ch ev-edn))
                        ;; true iff apparent success for >=1 client:
                        (some identity))))))]

           (send-to-hk-chs! (@clients-ws uid)) ; WebSocket clients
           (go ; Need speed here for broadcasting purposes:
            ;; Prefer broadcasting only to users we know/expect to be online:
            (send-to-hk-chs! (<! (ch-pull-ajax-hk-chs! clients-ajax uid))))

           nil ; Always return nil
           )))

     :ajax-post-fn ; Does not participate in `clients-ajax` (has specific req->resp)
     (fn [ring-req]
       (http-kit/with-channel ring-req hk-ch
         (let [msg       (-> ring-req :params :edn try-read-edn)
               dummy-cb? (and (map? msg) (:chsk/dummy-cb? msg))
               clj       (if-not dummy-cb? msg (:chsk/clj msg))]

           (receive-event-msg! ch-recv
             {;; Don't actually use the Ajax POST client-uuid, but we'll set
              ;; one anyway for `event-msg?`:
              :client-uuid (encore/uuid-str)
              :ring-req ring-req
              :event clj
              :?reply-fn
              (when-not dummy-cb?
                (fn reply-fn [resp-clj] ; Any clj form
                  (timbre/tracef "Chsk send (ajax reply): %s" resp-clj)
                  (let [resp-edn (pr-str resp-clj)]
                    ;; true iff apparent success:
                    (http-kit/send! hk-ch resp-edn))))})

           (when dummy-cb?
             (timbre/tracef "Chsk send (ajax reply): dummy-200")
             (http-kit/send! hk-ch (pr-str :chsk/dummy-200))))))

     :ajax-get-or-ws-handshake-fn ; ajax-poll or ws-handshake
     (fn [{:as ring-req {:keys [uid] :as session} :session}]
       (http-kit/with-channel ring-req hk-ch
         (let [client-uuid ; Browser-tab / device identifier
               (str uid "-" ; Security measure (can't be controlled by client)
                 (or (:ajax-client-uuid ring-req)
                     (encore/uuid-str)))

               receive-event-msg!* ; Partial
               (fn [event & [?reply-fn]]
                 (receive-event-msg! ch-recv
                   {:client-uuid  client-uuid ; Fixed (constant) with handshake
                    :ring-req     ring-req    ; ''
                    :event        event
                    :?reply-fn    ?reply-fn}))]

           (if-not (:websocket? ring-req)
             (when uid ; Server shouldn't attempt a non-uid long-pollling GET anyway
               (swap! clients-ajax (fn [[_ m]]
                                     [nil (assoc-in m [uid client-uuid] hk-ch)]))

               ;; Currently relying on `on-close` to _always_ trigger for every
               ;; connection. If that's not the case, will need some kind of gc.
               (http-kit/on-close hk-ch
                 (fn [status]
                   (swap! clients-ajax
                     (fn [[_ m]]
                       (let [new (dissoc (get m uid) client-uuid)]
                         [nil (if (empty? new)
                                (dissoc m uid)
                                (assoc  m uid new))])))
                   (receive-event-msg!* [:chsk/uidport-close :ajax])))
               (receive-event-msg!* [:chsk/uidport-open :ajax]))

             (do
               (timbre/tracef "New WebSocket channel: %s %s"
                 (or uid "(no uid)") (str hk-ch)) ; _Must_ call `str` on ch
               (when uid
                 (swap! clients-ws (fn [m] (assoc m uid (conj (m uid #{}) hk-ch))))
                 (receive-event-msg!* [:chsk/uidport-open :ws]))

               (http-kit/on-receive hk-ch
                 (fn [req-edn]
                   (let [[clj ?cb-uuid] (unwrap-edn-msg-with-?cb->clj req-edn)]
                     (receive-event-msg!* clj
                       (when ?cb-uuid
                         (fn reply-fn [resp-clj] ; Any clj form
                           (timbre/tracef "Chsk send (ws reply): %s" resp-clj)
                           (let [resp-edn (pr-str {:chsk/clj     resp-clj
                                                   :chsk/cb-uuid ?cb-uuid})]
                             ;; true iff apparent success:
                             (http-kit/send! hk-ch resp-edn))))))))

               ;; Currently relying on `on-close` to _always_ trigger for every
               ;; connection. If that's not the case, will need some kind of gc.
               (http-kit/on-close hk-ch
                 (fn [status]
                   (when uid
                     (swap! clients-ws
                       (fn [m]
                         (let [new (disj (m uid #{}) hk-ch)]
                           (if (empty? new) (dissoc m uid)
                                            (assoc  m uid new)))))
                     (receive-event-msg!* [:chsk/uidport-close :ws]))))

               (http-kit/send! hk-ch (pr-str [:chsk/handshake :ws])))))))}))

;;;; Client

      
                      
                    
                                                                            

      
                                           
                  
                                                 
                                                   
                        
                                                                           
                        
                                                
                                                                     
                                      

      
                                               
               
                             
                                                
                                                
                                       

      
                                                         
                                      
                                                
                                                                             
                                                                               
                             
                  
                        
                                                          
                       
                                            
                                                                        
                                        
                    

      
                      
                                                 
                                                                                    
                                                 
                                                                                  
                                      
                                                                               
                                                

      
                                                                 
                               
                            
                                                 
                                                          
                                   
                  

                                     
                                  
                                    
                            
                         
                          
                        
                     
                                                                     
                                     

                                                     
                                                                          
                                                                     
                         
           
                      
                         
                                                     
                                       
                                                                    
                                         
                                             
                                        
                                                           
                                                
                                                        
                                                                 
              
                                    
                                      
                             
                             
                                              
                            
                                                                          
                                         
                                        
                         

                                        
                                                       
                                                         
                             
                                              
                                               
                                                                          
                                        
               
                        
                                         
                                                                             
                               
                           
                                          
                                                                           
                                                    
                                                                          
                                                
                                                     
                                                   
                                  
                                                                                 
                                     
                                                                            
                                                  
                                          
                                                          
                            
                            
                                      
                                           
                            
                                                                     
                                                             
                                                  
                                 
                                                                          
                    
                             
                            
                                                                           
                                                                        
                                                    
                                                                         
                                                            
                                       
                                                                                              
                                     
                                                                               
                                                                 
                                 

                                                                            
                                                            
         
             

      
                                                       
                                             
           
                        
                            
                                                     
                                       
                                                                    
                                         
                                             
                                                        
                            
                                                           
                                                
           
                               
                                              
                   
                                        
                                                                    
                                                                        
                                         
                                                        
                                                 

                                                
                      
                                                         
                                                     
                                                   
                                                         

                                     
                                                         
                                             
                                                        
                                                                             
                                                  

                               

                                      
                                                                              
                                                                               
                                                                               
                
                    
                                                                             
                                                 
                                                       
                     
                                      
                                                
                                                                     
                                                                
                                                       
                             
                                                                
                                                
                                                          
                                                                

                                       
                                                      
                                                                                   
                                         
                                             
                                                     
                                                      

                                            
                                                                                  
                          

                                                 
                                                                    
                  
          

      
                                     
                                                                      
                                                                             
                                        

      
                          
                                                                                      
                                
                                                                         
                                                                    
                                                                      
                                                

                                                                            
                                           
                                    
                                                         
                                 
                                                     
                                 
                                     

                               
                                                   

                                                                             
                                                      
                                           
                                            

            
           
                               
                         
                                               
                                                                  
                                
                                          

                             
                                                                       
                                                       
                           
                                                               
                                                        
                                          

                                                    
                                 
                                                                          
                                                                                

              
                 
               
                   
                                                                             
                                                                                      
                                                                       
                                             

;;;; Routers

     
(defn start-chsk-router-loop! [event-msg-handler ch]
  (go-loop []
    (try
      (let [event-msg (<! ch)]
        (try
          (timbre/tracef "Event-msg: %s" event-msg)
          (event-msg-handler event-msg ch)
          (catch Throwable t
            (timbre/errorf t "Chsk-router-loop handling error: %s" event-msg))))
      (catch Throwable t
        (timbre/errorf t "Chsk-router-loop channel error!")))
    (recur)))

      
                                                
             
                                      
                                                                       
                                                      
                

;;;;;;;;;;;; This file autogenerated from src/taoensso/sente.cljx
