(ns taoensso.sente
  "Channel sockets. Otherwise known as The Shiz.

      Protocol  | client>server | client>server ?+ ack/reply | server>user[1] push
    * WebSockets:       ✓              [2]                           ✓
    * Ajax:            [3]              ✓                           [4]

    [1] By user-id => ALL of a user's connected clients (browser tabs, devices,
        etc.). Note that user > session > client > connection for consistency
        over time + multiple devices.
    [2] Emulate with cb-uuid wrapping.
    [3] Emulate with dummy-cb wrapping.
    [4] Emulate with long-polling.

  Abbreviations:
    * chsk  - Channel socket.
    * hk-ch - Http-kit Channel.
    * uid   - User-id. An application-specified identifier unique to each user
              and sessionized under `:uid` key to enable server>user push.
              May have semantic meaning (e.g. username, email address), or not
              (e.g. random uuid) - app's discresion.
    * cb    - Callback.
    * tout  - Timeout.
    * ws    - WebSocket/s.
    * pstr  - Packed string. Arbitrary Clojure data encoded as a string (e.g.
              edn) for client<->server comms.

  Special messages (implementation detail):
    * Callback replies: :chsk/closed, :chsk/timeout, :chsk/error.
    * Client-side events:
        [:chsk/handshake [<?uid> <?csrf-token>]],
        [:chsk/ws-ping],
        [:chsk/state <new-state>],
        [:chsk/recv <[buffered-evs]>] ; server>user push

    * Server-side events:
        [:chsk/bad-package <packed-str>], ; was :chsk/bad-edn
        [:chsk/bad-event <chsk-event>],
        [:chsk/uidport-open],
        [:chsk/uidport-close].

    * Callback wrapping: [<clj>], or [<clj> <cb-uuid>] with a uuid for [2],
                         or `ajax-dummy-cb-uuid` for [3].

  Notable implementation details:
    * core.async is used liberally where brute-force core.async allows for
      significant implementation simplifications. We lean on core.async's strong
      efficiency here.
    * For WebSocket fallback we use long-polling rather than HTTP 1.1 streaming
      (chunked transfer encoding). Http-kit _does_ support chunked transfer
      encoding but a small minority of browsers &/or proxies do not. Instead of
      implementing all 3 modes (WebSockets, streaming, long-polling) - it seemed
      reasonable to focus on the two extremes (performance + compatibility). In
      any case client support for WebSockets is growing rapidly so fallback
      modes will become increasingly irrelevant while the extra simplicity will
      continue to pay dividends.

  General-use notes:
    * Single HTTP req+session persists over entire chsk session but cannot
      modify sessions! Use standard a/sync HTTP Ring req/resp for logins, etc.
    * Easy to wrap standard HTTP Ring resps for transport over chsks. Prefer
      this approach to modifying handlers (better portability)."
  {:author "Peter Taoussanis"}

       
                                    
                                                                         
                                                                 
                                                 
                                                    
                                                  
                                                  
                                                       

        
  (:require [clojure.string  :as str]
            [cljs.core.async :as async :refer (<! >! put! chan)]
            ;; [cljs.reader     :as edn]
            [taoensso.encore :as encore :refer (format)]
            [taoensso.sente.interfaces :as interfaces])

        
  (:require-macros [cljs.core.async.macros :as asyncm :refer (go go-loop)]))

;;;; Events
;; * Clients send `event`s to server; server receives `event-msg`s.
;; * Server sends `event`s to clients; clients receive `event`s.

(defn- validate-event-form [x]
  (cond
    (not (vector? x))        :wrong-type
    (not (#{1 2} (count x))) :wrong-length
    :else (let [[ev-id _] x]
            (cond (not (keyword? ev-id))  :wrong-id-type
                  (not (namespace ev-id)) :unnamespaced-id
                  :else nil))))

(defn event? "Valid [ev-id ?ev-data] form?" [x] (nil? (validate-event-form x)))

(defn assert-event [x]
  (when-let [?err (validate-event-form x)]
    (let [err-fmt
          (str
           (case ?err
             :wrong-type   "Malformed event (wrong type)."
             :wrong-length "Malformed event (wrong length)."
             (:wrong-id-type :unnamespaced-id)
             "Malformed event (`ev-id` should be a namespaced keyword)."
             :else "Malformed event (unknown error).")
           " Event should be of `[ev-id ?ev-data]` form: %s")]
      (throw (ex-info (format err-fmt (str x)) {:malformed-event x})))))

(defn- chan? [x]
                                                                         
         (instance?    cljs.core.async.impl.channels.ManyToManyChannel x))

     
                    
      
            
                                                                      
                                                                       
                    
                                                                            
                                     
                            
                            
                                                       
             
                                   
                                                          
                                           
                             
                          
                                                     

     
                             
                        
                                                                             
                                   
               
                                
                                
                                       
           
                                                                      
                                 
                                                                          
                   
                                       
                                             
                                                                          
                                                                             
                                                      

                                                    
                                                          
                               

      
(defn cb-success? "Note that cb reply need _not_ be `event` form!"
  [cb-reply-clj] (not (#{:chsk/closed :chsk/timeout :chsk/error} cb-reply-clj)))

;;;; Payloads
;; * Client<->server payloads are arbitrary Clojure values (for callback
;;   replies), otherwise `event`s.
;; * Client>server payloads are un/wrapped for a consistent callback API over
;;   both WebSockets & Ajax.
;; * All payloads are un/packed to an encoded string during client<->server
;;   transit.

(def ^:private pack "clj->pstr" interfaces/pack) ; Alias
(defn- unpack "pstr->clj" [packer pstr]
         (interfaces/unpack packer pstr) ; Let client throw on bad pstr from server
                                     
                                             
                                                           )

(def ^:private ajax-dummy-cb-uuid
  "Sentinel cb-uuid value used by wrapped Ajax payloads that don't have any
  client-side callback fn (a dummy server>client reply will be required)."
  0)

(defn- cb-wrap [chsk-type clj ?cb-uuid-or-have-cb?]
  (with-meta
    (case chsk-type
      :ws   (if-let [cb-uuid  ?cb-uuid-or-have-cb?] [clj cb-uuid] [clj])
      :ajax (if-let [have-cb? ?cb-uuid-or-have-cb?] [clj] [clj ajax-dummy-cb-uuid]))
    (meta clj)))

(defn- cb-unwrap [wrapped-clj] {:pre [(vector? wrapped-clj)]}
  (let [[unwrapped-clj ?cb-uuid] wrapped-clj
        ajax-dummy-cb-uuid? (= ?cb-uuid ajax-dummy-cb-uuid)
        ?cb-uuid            (when-not ajax-dummy-cb-uuid? ?cb-uuid)]
    (if ajax-dummy-cb-uuid?
      [unwrapped-clj :ajax-without-cb]
      [unwrapped-clj ?cb-uuid])))

(comment
  (def edn-packer interfaces/edn-packer)
  (pack edn-packer (cb-wrap :ajax "hello"           :ajax-with-cb))
  (pack edn-packer (cb-wrap :ws   [:foo/bar "data"] "ws-cb-uuid"))
  (cb-unwrap       (cb-wrap :ajax "hello"           :ajax-with-cb))
  (cb-unwrap       (cb-wrap :ajax "hello"           (not :ajax-with-cb))))

;;;; Server API

                                                      
                                                         

     
                          
                           
                                                                                     
                                                     
                                                                                 
                                                                           
                                                                                  

                 
                                                                               
                                                                     
                           
                           
                                                                                    

                                                                     
                                                                             
                                                                             
                                                               
                                                           
                                              
                                                       
                                  
                                 
                                                                             
                                          
                                                                          
                                                                                                               
                                                                                       
                             

                                           
                                           

                                                 
                                    
                                                     
                                                                                         
                         
                                                           
                                                                                           

                    
                      
                                
                                                   
                                                       
                               
                                    
                                                                                               
                                                                                                 
                                           
                                                  
                                                   
                                                                  
                                                                  
                                                   
                              

                                                          
                 
                                   
                                                   
                                                       
                                        
                                                                          
                                                                          
                                                               
                                                                  
                               
                                                                                       
                                                                                       
                                                                                         
                                           
                                                  
                                                   
                                                                 
                                                                   
                                                      
                                 

                                           
                       
                                                  
                                                       
                                
                                             
                                                                        
                                   
                                         

                             
                          
                           
                             
                                                            
                                
                                                                            
                                                                               
                                                                         
                                                                             
                                                                        
                                                              
                                                          
                                                             
                                                             
                                                              
                                                          
                                                         
                                                     
                                                 

                                                                        
                                                                               
                                                             
                                                                          
                                                                            
                                  
                                                                       
                                                        
                                                                       
                                                              

                                                             
                 
                                                                                  
                                        
                                           
                                          

                                                                                        
                                                               
                                         
                                              
                                                                           

                 
                               
                                         
                                                           
                               
                                                     
                                                                  
                                                       
                                                                  

                                                                
                                                                       
                                                             
                                                                                  
                                                                      
                                                                                     
                                      
                                                                                       
                                           

                                                                             
                        
              

                    
                                
                                
                                          

                            
                            
                                    

                                                                              
                   
                                            
                                                                
                                                   
                                                       

                                          
                                
                                                                                     
                                                                          
                                     
                                
                          
                                   
                                                        
                                                                         
                                                         
                                                           
                                                   
                                                            

                          
                                                                   
                                                                        

                                                                               
                   
                                            
                                                  
                                              
                                                  
                                                            
                                                                              
                                                                  
                                        

                                           
                                        
                                                
                                      
                                                                                
                                                   
                                        
                                                

                         
                                                
                                                                               

                                     
                                      
                                                            
                                                                
                                                                                  
                                           
                                                           

                                         
                               
                                                                
                                                                
                                                           
                                     
                                                               
                                                                              
                                                                            
                                                                       
                                                          
                                                                    

                                                                    
                                       
                             
                                                
                                                 
                                                              
                                         
                                              
                                                 

                                                    
                                                                   

                      
                                                                            
                                       
                                              

                                                                         
                                                                                 
                                                                           
                                                                               
                       
                                                   
                                                                     

                                  

                                               
                                                                   
                                                                  
                            
                                      
                                                 
                                     

                                             
                                                           

                                                                    
                                       
                             
                                                                  
                                                                                

                                                           
                        
                                                                         
                                                
                                          
                                                            
                                                                            
                                                              
                                                                   
                                                    
                                                                              
                                                                
                                                                    
                                                        
                                                                           
                                                                                
                                                      
                                                         
                                                              
                                                              
                                                             
                                            
                                                         
                                                                              

                               
                                                                    
                          

     
                                    
                                                                                 
                                
                                           
                                              

     
                                      
                                                                           
                                                        
                                                                         
                                                                         
                                                          
                                                           
                                                               
                                   
                                                            
                                                                     
                                               
                                                                        
                                               
                                                                              
                                                 
                                                                         
                                                                            
                                                          
                                           
                                            
                                     
                               
                                   
                                                                        
                                                                    
                                       
                                                        
                                                     
                                
                             
                            
                                                 
                                          
                                                                                  
                                                                             
                        
                                                           
                                                                             
                                                       
                                          
                                                                                  
                                                                
                                                                 
                                                  

;;;; Client API

      
(defprotocol IChSocket
  (chsk-init!      [chsk] "Implementation detail.")
  (chsk-destroy!   [chsk] "Kills socket, stops auto-reconnects.")
  (chsk-reconnect! [chsk] "Drops connection, allows auto-reconnect. Useful for reauthenticating after login/logout.")
  (chsk-send!      [chsk ev] [chsk ev ?timeout-ms ?cb]
    "Sends `[ev-id ?ev-data :as event]`, returns true on apparent success."))

      
(defn- assert-send-args [x ?timeout-ms ?cb]
  (assert-event x)
  (assert (or (and (nil? ?timeout-ms) (nil? ?cb))
              (and (encore/nneg-int? ?timeout-ms)))
          (format "cb requires a timeout; timeout-ms should be a +ive integer: %s"
           ?timeout-ms))
  (assert (or (nil? ?cb) (ifn? ?cb) (chan? ?cb))
          (format "cb should be nil, an ifn, or a channel: %s" (type ?cb))))

      
(defn- pull-unused-cb-fn! [cbs-waiting_ ?cb-uuid]
  (when ?cb-uuid
    (first (swap! cbs-waiting_
             (fn [[_ m]] (if-let [f (m ?cb-uuid)]
                          [f (dissoc m ?cb-uuid)]
                          [nil m]))))))

      
(defn- merge>chsk-state! [{:keys [chs state_] :as chsk} merge-state]
  (let [[old-state new-state]
        (encore/swap-in! state_ []
          (fn [old-state]
            (let [new-state (merge old-state merge-state)]
              (encore/swapped new-state [old-state new-state]))))]
    (when (not= old-state new-state)
      ;; (encore/debugf "Chsk state change: %s" new-state)
      (put! (:state chs) new-state)
      new-state)))

      
(defn- cb-chan-as-fn
  "Experimental, undocumented. Allows a core.async channel to be provided
  instead of a cb-fn. The channel will receive values of form
  [<event-id>.cb <reply>]."
  [?cb ev]
  (if (or (nil? ?cb) (ifn? ?cb)) ?cb
    (do (assert (chan? ?cb))
        (assert-event ev)
        (let [[ev-id _] ev
              cb-ch ?cb]
          (fn [reply]
            (put! cb-ch [(keyword (str (encore/fq-name ev-id) ".cb"))
                         reply]))))))

      
(defn- receive-buffered-evs! [ch-recv clj] {:pre [(vector? clj)]}
  (let [buffered-evs clj]
    (doseq [ev buffered-evs]
      (assert-event ev)
      (put! ch-recv ev))))

      
(defn- handle-when-handshake! [chsk clj]
  (when (and (vector? clj) ; Nb clj may be callback reply
             (= (first clj) :chsk/handshake))
    (let [[_ [uid csrf-token]] clj]
      (when (str/blank? csrf-token)
        (encore/warnf "Sente warning: NO CSRF TOKEN AVAILABLE"))
      (merge>chsk-state! chsk
        {:open?      true
         :uid        uid
         :csrf-token csrf-token})
      :handled)))

       ;; Handles reconnects, keep-alives, callbacks:
(defrecord ChWebSocket
    [url chs socket_ kalive-ms kalive-timer_ kalive-due?_ nattempt_
     cbs-waiting_ ; [dissoc'd-fn {<uuid> <fn> ...}]
     state_       ; {:type _ :open? _ :uid _ :csrf-token _ :destroyed? _}
     packer       ; IPacker
     ]

  IChSocket
  (chsk-send! [chsk ev] (chsk-send! chsk ev nil nil))
  (chsk-send! [chsk ev ?timeout-ms ?cb]
    ;; (encore/debugf "Chsk send: (%s) %s" (if ?cb "cb" "no cb") ev)
    (assert-send-args ev ?timeout-ms ?cb)
    (let [?cb-fn (cb-chan-as-fn ?cb ev)]
      (if-not (:open? @state_) ; Definitely closed
        (do (encore/warnf "Chsk send against closed chsk.")
            (when ?cb-fn (?cb-fn :chsk/closed)))
        (let [?cb-uuid    (when ?cb-fn
                            ;; Around briefly + per-client, so conserve
                            ;; bandwidth by using a mini-uuid:
                            (encore/substr (encore/uuid-str) 0 6))
              wrapped-clj (cb-wrap :ws ev ?cb-uuid)
              pstr        (pack packer wrapped-clj)]

          (when ?cb-uuid
            (swap! cbs-waiting_
              (fn [[_ m]] [nil (assoc m ?cb-uuid ?cb-fn)]))
            (when ?timeout-ms
              (go (<! (async/timeout ?timeout-ms))
                (when-let [cb-fn* (pull-unused-cb-fn! cbs-waiting_ ?cb-uuid)]
                  (cb-fn* :chsk/timeout)))))

          (try
            (.send @socket_ pstr)
            (reset! kalive-due?_ false)
            :apparent-success
            (catch js/Error e
              (encore/errorf "Chsk send %s" e)
              (when ?cb-uuid
                (let [cb-fn* (or (pull-unused-cb-fn! cbs-waiting_ ?cb-uuid)
                                 ?cb-fn)]
                  (cb-fn* :chsk/error)))
              false))))))

  (chsk-reconnect!  [chsk] (when-let [s @socket_] (.close s)))
  (chsk-destroy!    [chsk]
    (merge>chsk-state! chsk {:destroyed? true :open? false})
    (chsk-reconnect!   chsk))

  (chsk-init! [chsk]
    (when-let [WebSocket (or (aget js/window "WebSocket")
                             (aget js/window "MozWebSocket"))]
      ((fn connect! []
         (when-not (:destroyed? @state_)
           (let [retry!
                 (fn []
                   (let [nattempt* (swap! nattempt_ inc)]
                     (.clearInterval js/window @kalive-timer_)
                     (encore/warnf "Chsk is closed: will try reconnect (%s)."
                       nattempt*)
                     (encore/set-exp-backoff-timeout! connect! nattempt*)))]

             (if-let [socket (try (WebSocket. url)
                                  (catch js/Error e
                                    (encore/errorf "WebSocket js/Error: %s" e)
                                    false))]
               (reset! socket_
                 (doto socket
                   (aset "onerror"
                     (fn [ws-ev] (encore/errorf "WebSocket error: %s" ws-ev)))

                   (aset "onmessage" ; Nb receives both push & cb evs!
                     (fn [ws-ev]
                       (let [pstr        (aget ws-ev "data")
                             wrapped-clj (unpack packer pstr)
                             ;; Nb may or may NOT satisfy `event?` since we also
                             ;; receive cb replies here!:
                             [clj ?cb-uuid] (cb-unwrap wrapped-clj)]
                         ;; (assert-event clj) ;; NO!
                         (or
                           (and (handle-when-handshake! chsk clj)
                                (reset! nattempt_ 0))
                           (if ?cb-uuid
                             (if-let [cb-fn (pull-unused-cb-fn! cbs-waiting_
                                              ?cb-uuid)]
                               (cb-fn clj)
                               (encore/warnf "Cb reply w/o local cb-fn: %s" clj))
                             (let [buffered-evs clj]
                               (receive-buffered-evs! (:<server chs)
                                 buffered-evs)))))))

                   (aset "onopen"
                     (fn [_ws-ev]
                       (reset! kalive-timer_
                         (.setInterval js/window
                           (fn []
                             (when @kalive-due?_ ; Don't ping unnecessarily
                               (chsk-send! chsk [:chsk/ws-ping]))
                             (reset! kalive-due?_ true))
                           kalive-ms))
                       ;; (merge>chsk-state! chsk
                       ;;   {:open? true}) ; NO, handshake better!
                       ))

                   (aset "onclose" ; Fires repeatedly when server is down
                     (fn [_ws-ev] (merge>chsk-state! chsk {:open? false})
                       (retry!)))))

               ;; Couldn't even get a socket:
               (retry!))))))
      chsk)))

      
(defrecord ChAjaxSocket [url chs timeout ajax-client-uuid curr-xhr_ state_ packer]
  IChSocket
  (chsk-send! [chsk ev] (chsk-send! chsk ev nil nil))
  (chsk-send! [chsk ev ?timeout-ms ?cb]
    ;; (encore/debugf "Chsk send: (%s) %s" (if ?cb "cb" "no cb") ev)
    (assert-send-args ev ?timeout-ms ?cb)
    (let [?cb-fn (cb-chan-as-fn ?cb ev)]
      (if-not (:open? @state_) ; Definitely closed
        (do (encore/warnf "Chsk send against closed chsk.")
            (when ?cb-fn (?cb-fn :chsk/closed)))
        (do
          (encore/ajax-lite url
           {:method :post :timeout ?timeout-ms
            :resp-type :text ; We'll do our own pstr decoding
            :params
            (let [wrapped-clj (cb-wrap :ajax ev ?cb-fn)
                  pstr        (pack packer wrapped-clj)]
              {:_ (encore/now-udt) ; Force uncached resp
               :pstr pstr :csrf-token (:csrf-token @state_)})}

           (fn ajax-cb [{:keys [content error]}]
             (if error
               (if (= error :timeout)
                 (when ?cb-fn (?cb-fn :chsk/timeout))
                 (do (merge>chsk-state! chsk {:open? false})
                     (when ?cb-fn (?cb-fn :chsk/error))))

               (let [resp-pstr content
                     resp-clj  (unpack packer resp-pstr)]
                 (if ?cb-fn (?cb-fn resp-clj)
                   (when (not= resp-clj :chsk/cb-dummy-200)
                     (encore/warnf "Cb reply w/o local cb-fn: %s" resp-clj)))
                 (merge>chsk-state! chsk {:open? true})))))

          :apparent-success))))

  (chsk-reconnect!  [chsk] (when-let [x @curr-xhr_] (.abort x)))
  (chsk-destroy!    [chsk]
    (merge>chsk-state! chsk {:destroyed? true :open? false})
    (chsk-reconnect!   chsk))

  (chsk-init! [chsk]
    ((fn async-poll-for-update! [nattempt]
       (when-not (:destroyed? @state_)
         (let [retry!
               (fn []
                 (let [nattempt* (inc nattempt)]
                   (encore/warnf
                     "Chsk is closed: will try reconnect (%s)."
                     nattempt*)
                   (encore/set-exp-backoff-timeout!
                     (partial async-poll-for-update! nattempt*)
                     nattempt*)))

               ajax-req! ; Just for Pace wrapping below
               (fn []
                 (reset! curr-xhr_
                   (encore/ajax-lite url
                     {:method :get :timeout timeout
                      :resp-type :text ; Prefer to do our own pstr reading
                      :params {:_ (encore/now-udt) ; Force uncached resp
                               :ajax-client-uuid ajax-client-uuid}}
                     (fn ajax-cb [{:keys [content error]}]
                       (if error
                         (if (or (= error :timeout)
                                 (= error :abort) ; Abort => intentional, not err
                                 ;; It's particularly important that reconnect
                                 ;; aborts don't mark a chsk as closed since
                                 ;; we've no guarantee that a new handshake will
                                 ;; take place to remark as open (e.g. if uid
                                 ;; hasn't changed since last handshake).
                                 )
                           (async-poll-for-update! 0)
                           (do (merge>chsk-state! chsk {:open? false})
                               (retry!)))

                         ;; The Ajax long-poller is used only for events, never cbs:
                         (let [pstr content
                               clj  (unpack packer pstr)]
                           (or
                             (handle-when-handshake! chsk clj)
                             (let [buffered-evs clj]
                               (receive-buffered-evs! (:<server chs) buffered-evs)
                               (merge>chsk-state! chsk {:open? true})))
                           (async-poll-for-update! 0)))))))]

           ;; TODO Make this pluggable
           (if-let [pace (aget js/window "Pace")]
             ;; Assumes relevant extern is defined for :advanced mode compilation:
             (.ignore pace ajax-req!) ; Pace.js shouldn't trigger for long-polling
             (ajax-req!)))))
     0)
    chsk))

      
(def default-chsk-url-fn
  "(ƒ [path window-location websocket?]) -> server-side chsk route URL string.

    * path       - As provided to client-side `make-channel-socket!` fn
                   (usu. \"/chsk\").
    * websocket? - True for WebSocket connections, false for Ajax (long-polling)
                   connections.
    * window-location - Map with keys:
      :href     ; \"http://www.example.org:80/foo/bar?q=baz#bang\"
      :protocol ; \"http:\" ; Note the :
      :hostname ; \"example.org\"
      :host     ; \"example.org:80\"
      :pathname ; \"/foo/bar\"
      :search   ; \"?q=baz\"
      :hash     ; \"#bang\"

  Note that the *same* URL is used for: WebSockets, POSTs, GETs. Server-side
  routes should be configured accordingly."
  (fn [path {:as window-location :keys [protocol host pathname]} websocket?]
    (str (if-not websocket? protocol (if (= protocol "https:") "wss:" "ws:"))
         "//" host (or path pathname))))

      
(defn make-channel-socket!
  "Returns a map with keys:
    :ch-recv ; core.async channel to receive `event`s (internal or from server).
    :send-fn ; (fn [event & [?timeout-ms ?cb-fn]]) for client>server send.
    :state   ; Watchable, read-only (atom {:type _ :open? _ :uid _ :csrf-token _}).
    :chsk    ; IChSocket implementer. You can usu. ignore this.

  Common options:
    :type         ; e/o #{:auto :ws :ajax}. You'll usually want the default (:auto).
    :ws-kalive-ms ; Ping to keep a WebSocket conn alive if no activity w/in given
                  ; number of milliseconds.
    :lp-kalive-ms ; Ping to keep a long-polling (Ajax) conn alive ''.
    :chsk-url-fn  ; Please see `default-chsk-url-fn` for details.
    :packer       ; :edn (default), or an IPacker implementation (experimental)."
  [path &
   & [{:keys [type recv-buf-or-n ws-kalive-ms lp-timeout chsk-url-fn packer]
       :or   {type          :auto
              recv-buf-or-n (async/sliding-buffer 2048) ; Mostly for buffered-evs
              ws-kalive-ms  25000 ; < Heroku 30s conn timeout
              lp-timeout    25000 ; ''
              chsk-url-fn   default-chsk-url-fn
              packer        :edn}}
      _deprecated-more-opts]]

  {:pre [(#{:ajax :ws :auto} type)]}
  (when (not (nil? _deprecated-more-opts))
    (encore/warnf
     "`make-channel-socket!` fn signature CHANGED with Sente v0.10.0."))

  (let [packer (interfaces/coerce-packer packer)
        window-location (encore/get-window-location)
        private-chs {:state    (chan (async/sliding-buffer 1))
                     :internal (chan (async/sliding-buffer 10))
                     :<server  (chan recv-buf-or-n)}

        ever-opened?_ (atom false)
        state*        (fn [state]
                        (if (or (not (:open? state)) @ever-opened?_)
                          state
                          (do (reset! ever-opened?_ true)
                              (assoc state :first-open? true))))

        public-ch-recv
        (async/merge
          ;; TODO Async map< is deprecated in favour of transformers:
          [(async/map< (fn [ev]    (assert (event? ev)) ev)   (:internal private-chs))
           (async/map< (fn [state] [:chsk/state (state* state)]) (:state private-chs))
           (async/map< (fn [ev]    [:chsk/recv  ev])             (:recv  private-chs))]
          recv-buf-or-n)

        chsk
        (or
         (and (not= type :ajax)
              (chsk-init!
                (map->ChWebSocket
                  {:url           (chsk-url-fn path window-location :ws)
                   :chs           private-chs
                   :packer        packer
                   :socket_       (atom nil)
                   :kalive-ms     ws-kalive-ms
                   :kalive-timer_ (atom nil)
                   :kalive-due?_  (atom true)
                   :nattempt_     (atom 0)
                   :cbs-waiting_  (atom [nil {}])
                   :state_        (atom {:type :ws :open? false
                                         :destroyed? false})})))

         (and (not= type :ws)
              (let [;; Unchanging over multiple long-poll (re)connects:
                    ajax-client-uuid (encore/uuid-str)]
                (chsk-init!
                  (map->ChAjaxSocket
                    {:url              (chsk-url-fn path window-location (not :ws))
                     :chs              private-chs
                     :packer           packer
                     :timeout          lp-timeout
                     :ajax-client-uuid ajax-client-uuid
                     :curr-xhr_        (atom nil)
                     :state_           (atom {:type :ajax :open? false
                                              :destroyed? false})})))))]

    (when chsk
      {:chsk    chsk
       :ch-recv public-ch-recv
       :send-fn (partial chsk-send! chsk)
       :state   (:state_ chsk)})))

;;;; Example routers

     
                        
                                                                            
                                                          
                             
                       
               
               
                                    
                                                      
                                             
                                
                    
                                                           
                                               
                     
                                    
                                                                     
                                     
                            
                                                            
                 
                                          

      
(defn start-chsk-router!
  "Creates a go-loop to call `(event-handler <event>)` and
  returns a `(fn stop! [])`. Allows errors to throw & break loop."
  [ch-recv event-handler]
  (let [ch-ctrl (chan)]
    (go-loop []
      (let [[v p] (async/alts! [ch-recv ch-ctrl])]
        (if (identical? p ch-ctrl) ::stop
          (let [[id data :as event] v]
            (event-handler event)
            (recur)))))
    (fn stop! [] (async/close! ch-ctrl))))

;;;; Deprecated

     
                             
                                                        
                             
                             
                                                                

      
(defn start-chsk-router-loop!
  "DEPRECATED: Please use `start-chsk-router!` instead."
  [event-handler ch-recv] (start-chsk-router! ch-recv event-handler))

;;;;;;;;;;;; This file autogenerated from src/taoensso/sente.cljx
