(ns taoensso.sente
  "Channel sockets. Otherwise known as The Shiz.

      Protocol  | client>server | client>server ?+ ack/reply | server>user[1] push
    * WebSockets:       ✓              [2]                           ✓
    * Ajax:            [3]              ✓                           [4]

    [1] By user-id => ALL of a user's connected clients (browser tabs, devices,
        etc.). Note that user > session > client > connection for consistency
        over time + multiple devices.
    [2] Emulate with cb-uuid wrapping.
    [3] Emulate with dummy-cb wrapping.
    [4] Emulate with long-polling.

  Abbreviations:
    * chsk  - Channel socket.
    * hk-ch - Http-kit Channel.
    * uid   - User-id. An application-specified identifier unique to each user
              and sessionized under `:uid` key to enable server>user push.
              May have semantic meaning (e.g. username, email address), or not
              (e.g. random uuid) - app's discresion.
    * cb    - Callback.
    * tout  - Timeout.
    * ws    - WebSocket/s.
    * pstr  - Packed string. Arbitrary Clojure data encoded as a string (e.g.
              edn) for client<->server comms.

  Special messages (implementation detail):
    * Callback replies: :chsk/closed, :chsk/timeout, :chsk/error.
    * Client-side events:
        [:chsk/handshake [<?uid> <?csrf-token>]],
        [:chsk/ws-ping],
        [:chsk/state <new-state>],
        [:chsk/recv <[buffered-evs]>] ; server>user push

    * Server-side events:
        [:chsk/bad-package <packed-str>], ; was :chsk/bad-edn
        [:chsk/bad-event <chsk-event>],
        [:chsk/uidport-open],
        [:chsk/uidport-close].

    * Callback wrapping: [<clj>], or [<clj> <cb-uuid>] with a uuid for [2],
                         or `ajax-dummy-cb-uuid` for [3].

  Notable implementation details:
    * core.async is used liberally where brute-force core.async allows for
      significant implementation simplifications. We lean on core.async's strong
      efficiency here.
    * For WebSocket fallback we use long-polling rather than HTTP 1.1 streaming
      (chunked transfer encoding). Http-kit _does_ support chunked transfer
      encoding but a small minority of browsers &/or proxies do not. Instead of
      implementing all 3 modes (WebSockets, streaming, long-polling) - it seemed
      reasonable to focus on the two extremes (performance + compatibility). In
      any case client support for WebSockets is growing rapidly so fallback
      modes will become increasingly irrelevant while the extra simplicity will
      continue to pay dividends.

  General-use notes:
    * Single HTTP req+session persists over entire chsk session but cannot
      modify sessions! Use standard a/sync HTTP Ring req/resp for logins, etc.
    * Easy to wrap standard HTTP Ring resps for transport over chsks. Prefer
      this approach to modifying handlers (better portability)."
  {:author "Peter Taoussanis"}

       
  (:require [clojure.string :as str]
            [clojure.core.async :as async :refer (<! <!! >! >!! put! chan
                                                     go go-loop)]
            ;; [clojure.tools.reader.edn :as edn]
            [org.httpkit.server        :as http-kit]
            [taoensso.encore           :as encore]
            [taoensso.timbre           :as timbre]
            [taoensso.sente.interfaces :as interfaces])

        
                                     
                                                                
                                        
                                                        
                                                       

        
                                                                           )

;;;; Events
;; * Clients send `event`s to server; server receives `event-msg`s.
;; * Server sends `event`s to clients; clients receive `event`s.

(defn- validate-event-form [x]
  (cond
    (not (vector? x))        :wrong-type
    (not (#{1 2} (count x))) :wrong-length
    :else (let [[ev-id _] x]
            (cond (not (keyword? ev-id))  :wrong-id-type
                  (not (namespace ev-id)) :unnamespaced-id
                  :else nil))))

(defn event? "Valid [ev-id ?ev-data] form?" [x] (nil? (validate-event-form x)))

(defn assert-event [x]
  (when-let [?err (validate-event-form x)]
    (let [err-fmt
          (str
           (case ?err
             :wrong-type   "Malformed event (wrong type)."
             :wrong-length "Malformed event (wrong length)."
             (:wrong-id-type :unnamespaced-id)
             "Malformed event (`ev-id` should be a namespaced keyword)."
             :else "Malformed event (unknown error).")
           " Event should be of `[ev-id ?ev-data]` form: %s")]
      (throw (ex-info (format err-fmt (str x)) {:malformed-event x})))))

(defn- chan? [x]
         (instance? clojure.core.async.impl.channels.ManyToManyChannel x)
                                                                         )

     
(defn event-msg? "Valid {:client-uuid _ :ring-req _ :event _ :?reply-fn _} form?"
  [x]
  (and (map? x) (= (count x) 4)
       (every? #{:client-uuid :ring-req :event :?reply-fn} (keys x))
       (let [{:keys [client-uuid hk-ch ring-req event ?reply-fn]} x]
         (and (string? client-uuid) ; Set by client (Ajax) or server (WebSockets)
              (map? ring-req)
              (event? event)
              (or (nil? ?reply-fn) (ifn? ?reply-fn))))))

     
(defn- put-event-msg>ch-recv!
  [ch-recv {:as ev-msg :keys [client-uuid ring-req event ?reply-fn]}]
  (let [ev-msg*
        {:client-uuid client-uuid ; Browser-tab / device identifier
         :ring-req    ring-req
         :event       (if (event? event) event [:chsk/bad-event event])
         :?reply-fn
         (if (ifn? ?reply-fn) ?reply-fn
           (-> (fn [resp-clj] ; Dummy warn fn
                 (timbre/warnf "Trying to reply to non-cb req: %s" event))
               ;; Useful to distinguish between a real cb reply fn and dummy:
               (with-meta {:dummy-reply-fn? true})))}]

    (if-not (event-msg? ev-msg*) ; NB conservative!!
      (timbre/warnf "Bad ev-msg!: %s (%s)" ev-msg* ev-msg)
      (put! ch-recv ev-msg*))))

      
                                                                  
                                                                                

;;;; Payloads
;; * Client<->server payloads are arbitrary Clojure values (for callback
;;   replies), otherwise `event`s.
;; * Client>server payloads are un/wrapped for a consistent callback API over
;;   both WebSockets & Ajax.
;; * All payloads are un/packed to an encoded string during client<->server
;;   transit.

(def ^:private pack "clj->pstr" interfaces/pack) ; Alias
(defn- unpack "pstr->clj" [packer pstr]
                                         ; Let client throw on bad pstr from server
         (try (assert (string? pstr))
              (interfaces/unpack packer pstr)
              (catch Throwable t [:chsk/bad-package pstr])))

(def ^:private ajax-dummy-cb-uuid
  "Sentinel cb-uuid value used by wrapped Ajax payloads that don't have any
  client-side callback fn (a dummy server>client reply will be required)."
  0)

(defn- cb-wrap [chsk-type clj ?cb-uuid-or-have-cb?]
  (with-meta
    (case chsk-type
      :ws   (if-let [cb-uuid  ?cb-uuid-or-have-cb?] [clj cb-uuid] [clj])
      :ajax (if-let [have-cb? ?cb-uuid-or-have-cb?] [clj] [clj ajax-dummy-cb-uuid]))
    (meta clj)))

(defn- cb-unwrap [wrapped-clj] {:pre [(vector? wrapped-clj)]}
  (let [[unwrapped-clj ?cb-uuid] wrapped-clj
        ajax-dummy-cb-uuid? (= ?cb-uuid ajax-dummy-cb-uuid)
        ?cb-uuid            (when-not ajax-dummy-cb-uuid? ?cb-uuid)]
    (if ajax-dummy-cb-uuid?
      [unwrapped-clj :ajax-without-cb]
      [unwrapped-clj ?cb-uuid])))

(comment
  (def edn-packer interfaces/edn-packer)
  (pack edn-packer (cb-wrap :ajax "hello"           :ajax-with-cb))
  (pack edn-packer (cb-wrap :ws   [:foo/bar "data"] "ws-cb-uuid"))
  (cb-unwrap       (cb-wrap :ajax "hello"           :ajax-with-cb))
  (cb-unwrap       (cb-wrap :ajax "hello"           (not :ajax-with-cb))))

;;;; Server API

      (declare ^:private send-buffered-evs>ws-clients!
               ^:private send-buffered-evs>ajax-clients!)

     
(defn make-channel-socket!
  "Returns a map with keys:
    :ch-recv ; core.async channel to receive `event-msg`s (internal or from clients).
    :send-fn ; (fn [user-id ev] for server>user push.
    :ajax-post-fn                ; (fn [ring-req]  for Ring CSRF-POST + chsk URL.
    :ajax-get-or-ws-handshake-fn ; (fn [ring-req]) for Ring GET + chsk URL.
    :connected-uids ; Watchable, read-only (atom {:ws #{_} :ajax #{_} :any #{_}}).

  Common options:
    :user-id-fn       ; (fn [ring-req]) -> unique user-id for server>user push.
    :csrf-token-fn    ; (fn [ring-req]) -> CSRF token for Ajax POSTs.
    :send-buf-ms-ajax ; [1]
    :send-buf-ms-ws   ; [1]
    :packer           ; :edn (default), or an IPacker implementation (experimental).

  [1] Optimization to allow transparent batching of rapidly-triggered
      server>user pushes. This is esp. important for Ajax clients which use a
      (slow) reconnecting poller. Actual event dispatch may occur <= given ms
      after send call (larger values => larger batch windows)."
  [& [{:keys [recv-buf-or-n send-buf-ms-ajax send-buf-ms-ws
              user-id-fn csrf-token-fn packer]
       :or   {recv-buf-or-n (async/sliding-buffer 1000)
              send-buf-ms-ajax 100
              send-buf-ms-ws   30
              user-id-fn    (fn [ring-req] (get-in ring-req [:session :uid]))
              csrf-token-fn (fn [ring-req]
                              (or (get-in ring-req [:session :csrf-token])
                                  (get-in ring-req [:session :ring.middleware.anti-forgery/anti-forgery-token])
                                  (get-in ring-req [:session "__anti-forgery-token"])))
              packer :edn}}]]

  {:pre [(encore/pos-int? send-buf-ms-ajax)
         (encore/pos-int? send-buf-ms-ws)]}

  (let [packer  (interfaces/coerce-packer packer)
        ch-recv (chan recv-buf-or-n)
        conns_  (atom {:ws   {} ; {<uid> <#{hk-chs}>}
                       :ajax {} ; {<uid> {<client-uuid> [<?hk-ch> <udt-last-connected>]}}
                       })
        connected-uids_ (atom {:ws #{} :ajax #{} :any #{}})
        send-buffers_   (atom {:ws  {} :ajax  {}}) ; {<uid> [<buffered-evs> <#{ev-uuids}>]}

        connect-uid!
        (fn [type uid]
          (let [newly-connected?
                (encore/swap-in! connected-uids_ []
                  (fn [{:keys [ws ajax any] :as old-m}]
                    (let [new-m
                          (case type
                            :ws   {:ws (conj ws uid) :ajax ajax            :any (conj any uid)}
                            :ajax {:ws ws            :ajax (conj ajax uid) :any (conj any uid)})]
                      (encore/swapped new-m
                        (let [old-any (:any old-m)
                              new-any (:any new-m)]
                          (when (and (not (contains? old-any uid))
                                          (contains? new-any uid))
                            :newly-connected))))))]
            newly-connected?))

        upd-connected-uid! ; Useful for atomic disconnects
        (fn [uid]
          (let [newly-disconnected?
                (encore/swap-in! connected-uids_ []
                  (fn [{:keys [ws ajax any] :as old-m}]
                    (let [conns' @conns_
                          any-ws-clients?   (contains? (:ws   conns') uid)
                          any-ajax-clients? (contains? (:ajax conns') uid)
                          any-clients?      (or any-ws-clients?
                                                any-ajax-clients?)
                          new-m
                          {:ws   (if any-ws-clients?   (conj ws   uid) (disj ws   uid))
                           :ajax (if any-ajax-clients? (conj ajax uid) (disj ajax uid))
                           :any  (if any-clients?      (conj any  uid) (disj any  uid))}]
                      (encore/swapped new-m
                        (let [old-any (:any old-m)
                              new-any (:any new-m)]
                          (when (and      (contains? old-any uid)
                                     (not (contains? new-any uid)))
                            :newly-disconnected))))))]
            newly-disconnected?))]

    {:ch-recv ch-recv
     :connected-uids connected-uids_
     :send-fn ; server>user (by uid) push
     (fn [user-id ev
         ;; Extra arity currently undocumented:
         & [{:as _opts :keys [flush-send-buffer?]}]]
       (let [uid      user-id
             uid-name (str (or uid "nil"))
             _ (timbre/tracef "Chsk send: (->uid %s) %s" uid-name ev)
             _ (assert-event ev)
             ev-uuid (encore/uuid-str)

             flush-buffer!
             (fn [type]
               (when-let [pulled
                          (encore/swap-in! send-buffers_ [type]
                            (fn [m]
                              ;; Don't actually flush unless the event buffered
                              ;; with _this_ send call is still buffered (awaiting
                              ;; flush). This means that we'll have many (go
                              ;; block) buffer flush calls that'll noop. They're
                              ;;  cheap, and this approach is preferable to
                              ;; alternatives like flush workers.
                              (let [[_ ev-uuids] (get m uid)]
                                (if (contains? ev-uuids ev-uuid)
                                  (encore/swapped (dissoc m uid)
                                                  (get    m uid))
                                  (encore/swapped m nil)))))]
                 (let [[buffered-evs ev-uuids] pulled]
                   (assert (vector? buffered-evs))
                   (assert (set?    ev-uuids))

                   (let [packer-metas         (map meta buffered-evs)
                         combined-packer-meta (reduce merge {} packer-metas)
                         buffered-evs-pstr    (pack packer
                                                (with-meta buffered-evs
                                                  combined-packer-meta))]
                     (case type
                       :ws   (send-buffered-evs>ws-clients!   conns_
                               uid buffered-evs-pstr)
                       :ajax (send-buffered-evs>ajax-clients! conns_
                               uid buffered-evs-pstr))))))]

         (if (= ev [:chsk/close]) ; Currently undocumented
           (do
             (timbre/debugf "Chsk closing (client may reconnect): %s" uid-name)
             (when flush-send-buffer?
               (doseq [type [:ws :ajax]]
                 (flush-buffer! type)))

             (doseq [hk-ch      (get-in @conns_ [:ws   uid])] (http-kit/close hk-ch))
             (doseq [hk-ch (->> (get-in @conns_ [:ajax uid])
                                (vals)
                                (map first)
                                (remove nil?))] (http-kit/close hk-ch)))

           (do
             ;; Buffer event
             (doseq [type [:ws :ajax]]
               (encore/swap-in! send-buffers_ [type uid]
                 (fn [old-v]
                   (if-not old-v [[ev] #{ev-uuid}]
                     (let [[buffered-evs ev-uuids] old-v]
                       [(conj buffered-evs ev)
                        (conj ev-uuids     ev-uuid)])))))

             ;;; Flush event buffers after relevant timeouts:
             ;; * May actually flush earlier due to another timeout.
             ;; * We send to _all_ of a uid's connections.
             ;; * Broadcasting is possible but I'd suggest doing it rarely, and
             ;;   only to users we know/expect are actually online.
             (go (when-not flush-send-buffer? (<! (async/timeout send-buf-ms-ws)))
                 (flush-buffer! :ws))
             (go (when-not flush-send-buffer? (<! (async/timeout send-buf-ms-ajax)))
                 (flush-buffer! :ajax)))))

       ;; Server-side send is async so nothing useful to return (currently
       ;; undefined):
       nil)

     :ajax-post-fn ; Does not participate in `conns_` (has specific req->resp)
     (fn [ring-req]
       (http-kit/with-channel ring-req hk-ch
         (let [pstr            (get-in ring-req [:params :pstr])
               wrapped-clj     (unpack packer pstr)
               [clj dummy-cb?] (cb-unwrap wrapped-clj)]

           (put-event-msg>ch-recv! ch-recv
             {;; Currently unused for non-lp POSTs, but necessary for `event-msg?`:
              :client-uuid "dummy-ajax-post-fn-uuid" ; (encore/uuid-str)
              :ring-req    ring-req
              :event       clj
              :?reply-fn
              (when-not dummy-cb?
                (fn reply-fn [resp-clj] ; Any clj form
                  (timbre/tracef "Chsk send (ajax reply): %s" resp-clj)
                  (let [;; Retains correct packer-meta:
                        resp-pstr (pack packer resp-clj)]
                    ;; true iff apparent success:
                    (http-kit/send! hk-ch resp-pstr))))})

           (when dummy-cb?
             (timbre/tracef "Chsk send (ajax reply): cb-dummy-200")
             (http-kit/send! hk-ch (pack packer :chsk/cb-dummy-200))))))

     :ajax-get-or-ws-handshake-fn ; Ajax handshake/poll, or WebSocket handshake
     (fn [ring-req]
       (http-kit/with-channel ring-req hk-ch
         (let [uid        (user-id-fn    ring-req)
               uid-name   (str (or uid "nil"))
               csrf-token (csrf-token-fn ring-req)
               client-uuid ; Browser-tab / device identifier
               (str uid "-" ; Security measure (can't be controlled by client)
                 (or (get-in ring-req [:params :ajax-client-uuid])
                     (encore/uuid-str)))

               receive-event-msg! ; Partial
               (fn [event & [?reply-fn]]
                 (put-event-msg>ch-recv! ch-recv
                   {:client-uuid  client-uuid ; Fixed (constant) with handshake
                    :ring-req     ring-req    ; ''
                    :event        event
                    :?reply-fn    ?reply-fn}))

               handshake!
               (fn [hk-ch] (http-kit/send! hk-ch
                            (pack packer [:chsk/handshake [uid csrf-token]])))]

           (if (:websocket? ring-req)
             (do ; WebSocket handshake
               (timbre/tracef "New WebSocket channel: %s %s"
                 uid-name (str hk-ch)) ; _Must_ call `str` on ch
               (encore/swap-in! conns_ [:ws uid] (fn [s] (conj (or s #{}) hk-ch)))
               (when (connect-uid! :ws uid)
                 (receive-event-msg! [:chsk/uidport-open]))

               (http-kit/on-receive hk-ch
                 (fn [req-pstr]
                   (let [wrapped-clj    (unpack packer req-pstr)
                         [clj ?cb-uuid] (cb-unwrap wrapped-clj)]
                     (receive-event-msg! clj ; Should be ev
                       (when ?cb-uuid
                         (fn reply-fn [resp-clj] ; Any clj form
                           (timbre/tracef "Chsk send (ws reply): %s" resp-clj)
                           (let [wrapped-clj (cb-wrap :ws resp-clj ?cb-uuid)
                                 resp-pstr   (pack packer wrapped-clj)]
                             ;; true iff apparent success:
                             (http-kit/send! hk-ch resp-pstr))))))))

               ;; We rely on `on-close` to trigger for _every_ conn:
               (http-kit/on-close hk-ch
                 (fn [status]
                   (encore/swap-in! conns_ [:ws]
                     (fn [m] ; {<uid> <#{hk-chs}>
                       (let [new (disj (get m uid #{}) hk-ch)]
                         (if (empty? new)
                           (dissoc m uid) ; gc
                           (assoc  m uid new)))))

                   ;; (when (upd-connected-uid! uid)
                   ;;   (receive-event-msg! [:chsk/uidport-close]))

                   (go
                     ;; Allow some time for possible reconnects (sole window
                     ;; refresh, etc.):
                     (<! (async/timeout 5000))

                     ;; Note different (simpler) semantics here than Ajax
                     ;; case since we don't have/want a `udt-disconnected` value.
                     ;; Ajax semantics: 'no reconnect since disconnect+5s'.
                     ;; WS semantics: 'still disconnected after disconnect+5s'.
                     ;;
                     (when (upd-connected-uid! uid)
                       (receive-event-msg! [:chsk/uidport-close])))))

               (handshake! hk-ch))

             ;; Ajax handshake/poll connection:
             (let [handshake? ; Initial connection for this client?
                   (encore/swap-in! conns_ [:ajax uid client-uuid]
                     (fn [v]
                       (encore/swapped
                         [hk-ch (encore/now-udt)]
                         (nil? v))))]

               (when (connect-uid! :ajax uid)
                 (receive-event-msg! [:chsk/uidport-open]))

               ;; We rely on `on-close` to trigger for _every_ conn:
               (http-kit/on-close hk-ch
                 (fn [status]
                   (encore/swap-in! conns_ [uid :ajax client-uuid]
                     (fn [[hk-ch udt-last-connected]] [nil udt-last-connected]))

                   (let [udt-disconnected (encore/now-udt)]
                     (go
                       ;; Allow some time for possible poller reconnects:
                       (<! (async/timeout 5000))
                       (let [disconnected?
                             (encore/swap-in! conns_ [:ajax]
                               (fn [m] ; {<uid> {<client-uuid> [<?hk-ch> _]}
                                 (let [[_ ?udt-last-connected]
                                       (get-in m [uid client-uuid])
                                       disconnected?
                                       (and ?udt-last-connected ; Not yet gc'd
                                            (>= udt-disconnected
                                              ?udt-last-connected))]
                                   (if-not disconnected?
                                     (encore/swapped m (not :disconnected))
                                     (let [new (dissoc (get m uid) client-uuid)]
                                       (encore/swapped
                                         (if (empty? new)
                                           (dissoc m uid) ; Gc
                                           (assoc  m uid new))
                                         :disconnected))))))]
                         (when disconnected?
                           (when (upd-connected-uid! uid)
                             (receive-event-msg! [:chsk/uidport-close]))))))))

               (when handshake?
                 (handshake! hk-ch) ; Client will immediately repoll
                 ))))))}))

     
(defn- send-buffered-evs>ws-clients!
  "Actually pushes buffered events (as packed-str) to all uid's WebSocket conns."
  [conns_ uid buffered-evs-pstr]
  (doseq [hk-ch (get-in @conns_ [:ws uid])]
    (http-kit/send! hk-ch buffered-evs-pstr)))

     
(defn- send-buffered-evs>ajax-clients!
  "Actually pushes buffered events (as packed-str) to all uid's Ajax conns.
  Allows some time for possible Ajax poller reconnects."
  [conns_ uid buffered-evs-pstr & [{:keys [nmax-attempts ms-base ms-rand]
                                    ;; <= 7 attempts at ~135ms ea = 945ms
                                    :or   {nmax-attempts 7
                                           ms-base       90
                                           ms-rand       90}}]]
  (comment (* 7 (+ 90 (/ 90 2.0))))
  (let [;; All connected/possibly-reconnecting client uuids:
        client-uuids-unsatisfied (keys (get-in @conns_ [:ajax uid]))]
    (when-not (empty? client-uuids-unsatisfied)
      ;; (println "client-uuids-unsatisfied: " client-uuids-unsatisfied)
      (go-loop [n 0 client-uuids-satisfied #{}]
        (let [?pulled ; nil or {<client-uuid> [<?hk-ch> <udt-last-connected>]}
              (encore/swap-in! conns_ [:ajax uid]
                (fn [m] ; {<client-uuid> [<?hk-ch> <udt-last-connected>]}
                  (let [ks-to-pull (remove client-uuids-satisfied (keys m))]
                    ;; (println "ks-to-pull: " ks-to-pull)
                    (if (empty? ks-to-pull)
                      (encore/swapped m nil)
                      (encore/swapped
                        (reduce
                          (fn [m k]
                            (let [[?hk-ch udt-last-connected] (get m k)]
                              (assoc m k [nil udt-last-connected])))
                          m ks-to-pull)
                        (select-keys m ks-to-pull))))))]
          (assert (or (nil? ?pulled) (map? ?pulled)))
          (let [?newly-satisfied
                (when ?pulled
                  (reduce-kv
                   (fn [s client-uuid [?hk-ch _]]
                     (if (or (nil? ?hk-ch)
                             ;; hk-ch may have closed already (`send!` will noop):
                             (not (http-kit/send! ?hk-ch buffered-evs-pstr)))
                       s
                       (conj s client-uuid))) #{} ?pulled))
                now-satisfied (into client-uuids-satisfied ?newly-satisfied)]
            ;; (println "now-satisfied:" now-satisfied)
            (when (and (< n nmax-attempts)
                       (some (complement now-satisfied) client-uuids-unsatisfied))
              ;; Allow some time for possible poller reconnects:
              (<! (async/timeout (+ ms-base (rand-int ms-rand))))
              (recur (inc n) now-satisfied))))))))

;;;; Client API

      
                      
                                                   
                                                                 
                                                                                                                     
                                                      
                                                                             

      
                                           
                  
                                                 
                                                   
                                                                                  
                        
                                                
                                                                            

      
                                                 
                
                              
                                                 
                                                 
                                       

      
                                                                    
                             
                                  
                         
                                                          
                                                                  
                                    
                                                          
                                   
                  

      
                    
                                                                         
                                                             
                           
          
                                    
                            
                         
                          
                        
                     
                                                                     
                                     

      
                                                                 
                         
                            
                       
                          

      
                                        
                                                         
                                             
                                   
                                   
                                                                
                             
                         
                        
                                 
                 

                                                     
                      
                                                                   
                                                   
                                                                         
                           
      

           
                                                     
                                       
                                                                    
                                         
                                        
                                                  
                                                           
                                                
                                      
                                                                       
                                                              
                                                                  
                                                   
                                                    

                        
                               
                                                           
                             
                                                  
                                                                             
                                            

              
                                 
                                       
                             
                             
                                              
                            
                                                                           
                                         
                                        
                         

                                                              
                          
                                                            
                             

                    
                                                         
                                                              
                      
                                        
                       
                       
                                                         
                                                              
                                                                             
                                 
                                                                            

                                                  
                                                   
                                                                              
                                            
                              
                             
                                  
                                                                              

                                                                      
                                
                                                            
                                                             
                                                                                
                                                         
                                                                    
                                                     
                            
                                                                 
                                                     
                                       
                                                                            
                                                        
                                          
                                                                                 
                                                    
                                                                    
                                                    

                                 
                                 
                                            
                                                
                                 
                                                                           
                                                                 
                                                        
                                      
                                                 
                                                                  
                         

                                                                         
                                                                         
                                   

                                             
                            
             

      
                                                                                  
           
                                                     
                                       
                                                                    
                                         
                                        
                                                  
                                                           
                                                
           
                               
                                              
                                                             
                   
                                                       
                                                        
                                                        
                                                              

                                                
                      
                                     
                                                     
                                                            
                                                         

                                      
                                                         
                                             
                                                           
                                                                             
                                                           

                               

                                                                
                          
                                                            
                             

                    
                                          
                                      
                     
                     
                                                
                                
                                                               
                               
                                                   
                                                               
                                 

                                                       
                     
                                  
                                        
                                                   
                                                                          
                                                                        
                                                                   
                                                          
                                
                                                   
                                                                                 
                                                                              
                                                                            
                                                                                
                                                                             
                                                                         
                                  
                                                     
                                                                      
                                         

                                                                                    
                                           
                                                         
                              
                                                              
                                                    
                                                                                  
                                                                       
                                                            

                                      
                                                 
                                                                                  
                                                                                  
                            
       
          

      
                        
                                                                              

                                                                       
                                    
                                                                                
                               
                                      
                                                                  
                                        
                                 
                                    
                              
                            
                           

                                                                            
                                           
                                                                            
                                                                             
                                        

      
                          
                           
                                                                                
                                                                          
                                                                                   
                                                               

                 
                                                                                    
                                                                                 
                                           
                                                                     
                                                                 
                                                                                 
         
                                                                            
                                 
                                                                                 
                                                             
                                      
                                               
                                  
                             

                                    
                                          
                 
                                                                        

                                                
                                                    
                                                              
                                                               
                                                    

                                  
                                 
                                                                    
                               
                                                         
                                                                

                      
                    
                                                                     
                                                                                      
                                                                                      
                                                                                       
                        

            
           
                               
                         
                                 
                                                                        
                                             
                                        
                                            
                                              
                                            
                                             
                                          
                                                 
                                                               
                                                                

                             
                                                                       
                                                       
                           
                                    
                                                                                   
                                                  
                                             
                                                 
                                                       
                                                 
                                                                      
                                                                        

              
                    
                              
                                         
                                  

;;;; Example routers

     
(defn start-chsk-router-loop!
  "Creates a go-loop to call `(event-msg-handler <event-msg> <ch-recv>)` and
  returns a `(fn stop! [])`. Traps & logs errors w/o breaking loop."
  [event-msg-handler ch-recv]
  (let [ch-ctrl (chan)]
    (go-loop []
      (when-not
        (try ; returns nil or ::stop
          (let [[v p] (async/alts! [ch-recv ch-ctrl])]
            (if (identical? p ch-ctrl) ::stop
              (let [event-msg v]
                (try
                  (timbre/tracef "Event-msg: %s" event-msg)
                  (event-msg-handler event-msg ch-recv)
                  nil
                  (catch Throwable t
                    (timbre/errorf t "Chsk-router-loop handling error: %s"
                      event-msg))))))
          (catch Throwable t
            (timbre/errorf t "Chsk-router-loop channel error!")))
        (recur)))
    (fn stop! [] (async/close! ch-ctrl))))

      
                             
                                                                    
                                                                  
                         
                       
               
                                                  
                                         
                                      
                                         
                       
                                          

;;;;;;;;;;;; This file autogenerated from src/taoensso/sente.cljx
