(ns taoensso.sente
  "Channel sockets. Otherwise known as The Shiz.

      Protocol  | client>server | client>server + ack/reply | server>user[1] push
    * WebSockets:       ✓              [2]                          ✓  ; [3]
    * Ajax:            [4]              ✓                          [5] ; [3]

    [1] ALL of a user's connected clients (browser tabs, devices, etc.).
        Note that user > session > client > connection for consistency over time
        + multiple devices.
    [2] Emulate with cb-uuid wrapping.
    [3] By uid only (=> logged-in users only).
    [4] Emulate with dummy-cb wrapping.
    [5] Emulate with long-polling against uid (=> logged-in users only).

  Abbreviations:
    * chsk  - channel socket.
    * hk-ch - Http-kit Channel.
    * uid   - User id. An application-specified identifier unique to each user
              and sessionized under `:uid` key to enable server>user push.
              May have semantic meaning (e.g. username, email address), or not
              (e.g. random uuid) - app's discresion.
    * cb    - callback.
    * tout  - timeout.
    * ws    - WebSocket/s.

  Special messages (implementation detail):
    * cb replies: :chsk/closed, :chsk/timeout, :chsk/error.
    * client-side events:
        [:chsk/handshake <#{:ws :ajax}>],
        [:chsk/ping      <#{:ws :ajax}>], ; Though no :ajax ping
        [:chsk/state [<#{:open :first-open :closed}> <#{:ws :ajax}]],
        [:chsk/recv  <[buffered-evs]>]. ; server>user push

    * server-side events:
       [:chsk/bad-edn <edn>],
       [:chsk/bad-event <chsk-event>].

    * event wrappers: {:chsk/clj <clj> :chsk/dummy-cb? true} (for [2]),
                      {:chsk/clj <clj> :chsk/cb-uuid <uuid>} (for [4]).

  Notable implementation details:
    * Edn is used as a flexible+convenient transfer format, but can be seen as
      an implementation detail. Users may apply additional string encoding (e.g.
      JSON) at will. (This would incur a cost, but it'd be negligable compared
      to even the fastest network transfer times).
    * No server>client (with/without cb) mechanism is provided since:
      - server>user is what people actually want 90% of the time, and is a
        preferable design pattern in general IMO.
      - server>client could be (somewhat inefficiently) simulated with server>user.
    * core.async is used liberally where brute-force core.async allows for
      significant implementation simplifications. We lean on core.async's strong
      efficiency here.

  General-use notes:
    * Single HTTP req+session persists over entire chsk session but cannot
      modify sessions! Use standard a/sync HTTP Ring req/resp for logins, etc.
    * Easy to wrap standard HTTP Ring resps for transport over chsks. Prefer
      this approach to modifying handlers (better portability)."
  {:author "Peter Taoussanis"}

       
  (:require [clojure.string :as str]
            [clojure.set    :as set]
            [clojure.core.async :as async :refer (<! <!! >! >!! put! chan
                                                     go go-loop)]
            [clojure.tools.reader.edn :as edn]
            [org.httpkit.server       :as http-kit]
            [taoensso.encore          :as encore]
            [taoensso.timbre          :as timbre])

        
                                     
                                     
                                                                
                                     
                                                         

        
                                                                           )

;;;; Shared (client+server)

(defn- chan? [x]
         (instance? clojure.core.async.impl.channels.ManyToManyChannel x)
                                                                         )

(defn- validate-event-form [x]
  (cond  (not (vector? x))        :wrong-type
         (not (#{1 2} (count x))) :wrong-length
   :else (let [[ev-id _] x]
           (cond (not (keyword? ev-id))  :wrong-id-type
                 (not (namespace ev-id)) :unnamespaced-id
                 :else nil))))

(defn event? "Valid [ev-id ?ev-data] form?" [x] (nil? (validate-event-form x)))

(defn assert-event [x]
  (when-let [?err (validate-event-form x)]
    (let [err-fmt
          (str
           (case ?err
             :wrong-type   "Malformed event (wrong type)."
             :wrong-length "Malformed event (wrong length)."
             (:wrong-id-type :unnamespaced-id)
             "Malformed event (`ev-id` should be a namespaced keyword)."
             :else "Malformed event (unknown error).")
           " Event should be of `[ev-id ?ev-data]` form: %s")]
      (throw (ex-info (format err-fmt (str x)) {:malformed-event x})))))

(defn cb-success? [cb-reply] ;; Cb reply need _not_ be `event` form!
  (not (#{:chsk/closed :chsk/timeout :chsk/error} cb-reply)))

      ; For #+cljs we'd rather just throw client-side on bad edn from server
(defn- try-read-edn [edn]
  (try (edn/read-string edn)
       (catch Throwable t [:chsk/bad-edn edn])))

(defn- unwrap-edn-msg-with-?cb->clj "edn -> [clj ?cb-uuid]"
  [edn]
  (let [msg             (try-read-edn    edn)
                                             
        ?cb-uuid (and (map? msg) (:chsk/cb-uuid msg))
        clj      (if-not ?cb-uuid msg (:chsk/clj msg))]
    [clj ?cb-uuid]))

;;;; Server

     
(defn event-msg?
  "Valid {:client-uuid _ :ring-req _ :event _ :?reply-fn _} form?"
  [x]
  (and (map? x) (= (count x) 4)
       (every? #{:client-uuid :ring-req :event :?reply-fn} (keys x))
       (let [{:keys [client-uuid hk-ch ring-req event ?reply-fn]} x]
         (and (string? client-uuid) ; Set by client (Ajax) or server (WebSockets)
              (map? ring-req)
              (event? event)
              (or (nil? ?reply-fn) (ifn? ?reply-fn))))))

     
(defn- receive-event-msg!
  [ch-recv {:as ev-msg :keys [client-uuid ring-req event ?reply-fn]}]
  (let [ev-msg*
        {:client-uuid client-uuid ; Browser-tab / device identifier
         :ring-req    ring-req
         :event       (if (event? event) event [:chsk/bad-event event])
         :?reply-fn
         (if (ifn? ?reply-fn) ?reply-fn
           (-> (fn [resp-clj] ; Dummy warn fn
                 (timbre/warnf "Trying to reply to non-cb req: %s" event))
               ;; Useful to distinguish between a real cb reply fn and dummy:
               (with-meta {:dummy-reply-fn? true})))}]

    (if (event-msg? ev-msg*) ; Be conservative about what we put to chan!
      (put! ch-recv ev-msg*)
      (timbre/warnf "Bad ev-msg!: %s (%s)" ev-msg* ev-msg))))

     
(defn- send-buffered-evs>ws-clients!
  "Actually pushes buffered events (edn) to all uid's WebSocket conns."
  [conns_ uid buffered-evs-edn]
  (doseq [hk-ch (get-in @conns_ [:ws uid])]
    (http-kit/send! hk-ch buffered-evs-edn)))

     
(defn- send-buffered-evs>ajax-clients!
  "Actually pushes buffered events (edn) to all uid's Ajax conns. Allows some
  time for possible Ajax poller reconnects."
  [conns_ uid buffered-evs-edn & [{:keys [nmax-attempts ms-base ms-rand]
                                   ;; <= 7 attempts at ~135ms ea = 945ms
                                   :or   {nmax-attempts 5
                                          ms-base       90
                                          ms-rand       90}}]]
  (comment (* 7 (+ 90 (/ 90 2.0))))
  (let [;; All connected/possibly-reconnecting client uuids:
        client-uuids-unsatisfied (keys (get-in @conns_ [:ajax uid]))]
    (when-not (empty? client-uuids-unsatisfied)
      ;; (println "client-uuids-unsatisfied: " client-uuids-unsatisfied)
      (go-loop [n 0 client-uuids-satisfied #{}]
        (let [?pulled ; nil or {<client-uuid> [<?hk-ch> <udt-last-connected>]}
              (encore/swap-in! conns_ [:ajax uid]
                (fn [m] ; {<client-uuid> [<?hk-ch> <udt-last-connected>]}
                  (let [ks-to-pull (remove client-uuids-satisfied (keys m))]
                    (if (empty? ks-to-pull)
                      (encore/swapped m nil)
                      (encore/swapped
                       (reduce-kv (fn [m k [?hk-ch udt-last-connected]]
                                    (assoc m k [nil udt-last-connected]))
                                  m ks-to-pull)
                       (select-keys m ks-to-pull))))))]
          (assert (or (nil? ?pulled) (map? ?pulled)))
          (let [?newly-satisfied
                (when ?pulled
                  (reduce-kv
                   (fn [s client-uuid [?hk-ch _]]
                     (if (or (nil? ?hk-ch)
                             ;; hk-ch may have closed already:
                             (not (http-kit/send! ?hk-ch buffered-evs-edn)))
                       s
                       (conj s client-uuid))) #{} ?pulled))
                now-satisfied (into client-uuids-satisfied ?newly-satisfied)]
            ;; (println "now-satisfied:" now-satisfied)
            (when (and (< n nmax-attempts)
                       (some (complement now-satisfied) client-uuids-unsatisfied))
              ;; Allow some time for possible poller reconnects:
              (<! (async/timeout (+ ms-base (rand-int ms-rand))))
              (recur (inc n) now-satisfied))))))))

     
(defn make-channel-socket!
  "Returns `{:keys [ch-recv send-fn ajax-post-fn ajax-get-or-ws-handshake-fn
                    connected-uids]}`:
    * ch-recv - core.async channel ; For server-side chsk request router, will
                                   ; receive `event-msg`s from clients.
    * send-fn - (fn [user-id ev])   ; For server>user push
    * ajax-post-fn                - (fn [ring-req]) ; For Ring CSRF-POST, chsk URL
    * ajax-get-or-ws-handshake-fn - (fn [ring-req]) ; For Ring GET,       chsk URL
    * connected-uids ; Watchable, read-only (atom {:ws #{_} :ajax #{_} :any #{_}})

  Options:
    * recv-buf-or-n    ; Used for ch-recv buffer
    * user-id-fn       ; (fn [ring-req]) -> unique user-id, as used by
                       ; server>user push.
    * send-buf-ms-ajax ; [1]
    * send-buf-ms-ws   ; [1]

  [1] Optimization to allow transparent batching of rapidly-triggered
      server>user pushes. This is esp. important for Ajax clients which use a
      (slow) reconnecting poller. Actual event dispatch may occur <= given ms
      after send call (larger values => larger batch windows)."
  [& [{:keys [recv-buf-or-n send-buf-ms-ajax send-buf-ms-ws user-id-fn]
       :or   {recv-buf-or-n (async/sliding-buffer 1000)
              send-buf-ms-ajax 100
              send-buf-ms-ws   30
              user-id-fn (fn [ring-req] (get-in ring-req [:session :uid]))}}]]
  {:pre [(encore/pos-int? send-buf-ms-ajax)
         (encore/pos-int? send-buf-ms-ws)]}

  (let [ch-recv (chan recv-buf-or-n)
        conns_  (atom {:ws   {} ; {<uid> <#{hk-chs}>}
                       :ajax {} ; {<uid> {<client-uuid> [<?hk-ch> <udt-last-connected>]}}
                       })
        connected-uids_ (atom {:ws #{} :ajax #{} :any #{}})
        send-buffers_   (atom {:ws  {} :ajax  {}}) ; {<uid> [<buffered-evs> <#{ev-uuids}>]}

        upd-connected-uids!
        (fn []
          (swap! connected-uids_
            (fn [m]
              (let [{:keys [ws ajax]} @conns_
                    ws     (set (keys ws))
                    ajax   (set (keys ajax))]
                {:ws ws :ajax ajax :any (set/union ws ajax)}))))]

    {:ch-recv ch-recv
     :connected-uids connected-uids_
     :send-fn ; server>user (by uid) push
     (fn [uid ev & [{:as _opts :keys [flush-send-buffer?]}]]
       (timbre/tracef "Chsk send: (->uid %s) %s" uid ev)
       (assert-event ev)
       (assert (not (nil? uid))
         "server>user push requires a non-nil user-id (client session :uid by default)")
       (let [ev-uuid (encore/uuid-str)
             flush-buffer!
             (fn [type]
               (when-let [pulled (encore/swap-in! send-buffers_ [type]
                                   (fn [m] (encore/swapped (dissoc m uid)
                                                          (get    m uid))))]
                 (let [[buffered-evs ev-uuids] pulled]
                   (assert (vector? buffered-evs))
                   (assert (set?    ev-uuids))
                   ;; Don't actually flush unless the event buffered with _this_
                   ;; send call is still buffered (awaiting flush). This means
                   ;; that we'll have many (go block) buffer flush calls that'll
                   ;; noop. They're cheap, and this approach is preferable to
                   ;; alternatives like flush workers.
                   (when (contains? ev-uuids ev-uuid)
                     (let [buffered-evs-edn (pr-str buffered-evs)]
                       (case type
                         :ws   (send-buffered-evs>ws-clients!   conns_
                                 uid buffered-evs-edn)
                         :ajax (send-buffered-evs>ajax-clients! conns_
                                 uid buffered-evs-edn)))))))]

         (if (= ev [:chsk/close])
           (do
             (timbre/debugf "Chsk CLOSING: %s" uid)

             (when flush-send-buffer?
               (doseq [type [:ws :ajax]]
                 (flush-buffer! type)))

             (doseq [hk-ch     (get-in @conns_ [:ws   uid])] (http-kit/close hk-ch))
             (doseq [hk-ch (-> (get-in @conns_ [:ajax uid])
                               (vals)
                               (map first)
                               (remove nil?))] (http-kit/close hk-ch)))

           (do
             ;; Buffer event
             (doseq [type [:ws :ajax]]
               (encore/swap-in! send-buffers_ [type uid]
                 (fn [old-v]
                   (if-not old-v [[ev] #{ev-uuid}]
                     (let [[buffered-evs ev-uuids] old-v]
                       [(conj buffered-evs ev)
                        (conj ev-uuids     ev-uuid)])))))

             ;;; Flush event buffers after relevant timeouts:
             ;; * May actually flush earlier due to another timeout.
             ;; * We send to _all_ of a uid's connections.
             ;; * Broadcasting is possible but I'd suggest doing it rarely, and
             ;;   only to users we know/expect are actually online.
             (go (when-not flush-send-buffer? (<! (async/timeout send-buf-ms-ws)))
                 (flush-buffer! :ws))
             (go (when-not flush-send-buffer? (<! (async/timeout send-buf-ms-ajax)))
                 (flush-buffer! :ajax)))))

       nil)

     :ajax-post-fn ; Does not participate in `conns_` (has specific req->resp)
     (fn [ring-req]
       (http-kit/with-channel ring-req hk-ch
         (let [msg       (-> ring-req :params :edn try-read-edn)
               dummy-cb? (and (map? msg) (:chsk/dummy-cb? msg))
               clj       (if-not dummy-cb? msg (:chsk/clj msg))]

           (receive-event-msg! ch-recv
             {;; Currently unused for non-lp POSTs, but necessary for `event-msg?`:
              :client-uuid "degenerate-ajax-post-fn-uuid" ; (encore/uuid-str)
              :ring-req ring-req
              :event clj
              :?reply-fn
              (when-not dummy-cb?
                (fn reply-fn [resp-clj] ; Any clj form
                  (timbre/tracef "Chsk send (ajax reply): %s" resp-clj)
                  (let [resp-edn (pr-str resp-clj)]
                    ;; true iff apparent success:
                    (http-kit/send! hk-ch resp-edn))))})

           (when dummy-cb?
             (timbre/tracef "Chsk send (ajax reply): dummy-200")
             (http-kit/send! hk-ch (pr-str :chsk/dummy-200))))))

     :ajax-get-or-ws-handshake-fn ; ajax-poll or ws-handshake
     (fn [ring-req]
       (http-kit/with-channel ring-req hk-ch
         (let [uid (user-id-fn ring-req)
               client-uuid ; Browser-tab / device identifier
               (str uid "-" ; Security measure (can't be controlled by client)
                 (or (get-in ring-req [:params :ajax-client-uuid])
                     (encore/uuid-str)))

               receive-event-msg!* ; Partial
               (fn [event & [?reply-fn]]
                 (receive-event-msg! ch-recv
                   {:client-uuid  client-uuid ; Fixed (constant) with handshake
                    :ring-req     ring-req    ; ''
                    :event        event
                    :?reply-fn    ?reply-fn}))]

           (if (:websocket? ring-req)
             (do
               (timbre/tracef "New WebSocket channel: %s %s"
                 (or uid "(no uid)") (str hk-ch)) ; _Must_ call `str` on ch
               (when uid
                 (encore/swap-in! conns_ [:ws uid] (fn [s] (conj (or s #{}) hk-ch)))
                 (swap! connected-uids_
                   (fn [{:keys [ws ajax any]}]
                     {:ws (conj ws uid) :ajax ajax :any (conj any uid)})))

               (http-kit/on-receive hk-ch
                 (fn [req-edn]
                   (let [[clj ?cb-uuid] (unwrap-edn-msg-with-?cb->clj req-edn)]
                     (receive-event-msg!* clj
                       (when ?cb-uuid
                         (fn reply-fn [resp-clj] ; Any clj form
                           (timbre/tracef "Chsk send (ws reply): %s" resp-clj)
                           (let [resp-edn (pr-str {:chsk/clj     resp-clj
                                                   :chsk/cb-uuid ?cb-uuid})]
                             ;; true iff apparent success:
                             (http-kit/send! hk-ch resp-edn))))))))

               ;; We rely on `on-close` to trigger for _every_ conn:
               (http-kit/on-close hk-ch
                 (fn [status]
                   (when uid
                     (encore/swap-in! conns_ [:ws]
                       (fn [m] ; {<uid> <#{hk-chs}>
                         (let [new (disj (get m uid #{}) hk-ch)]
                           (if (empty? new)
                             (dissoc m uid) ; gc
                             (assoc  m uid new)))))
                     (upd-connected-uids!))))
               (http-kit/send! hk-ch (pr-str [:chsk/handshake :ws])))

             (when uid ; Server shouldn't attempt a non-uid long-pollling GET anyway
               (encore/swap-in! conns_ [:ajax uid client-uuid]
                 (fn [m] [hk-ch (encore/now-udt)]))
               (swap! connected-uids_
                 (fn [{:keys [ws ajax any]}]
                   {:ws ws :ajax (conj ajax uid) :any (conj any uid)}))

               ;; We rely on `on-close` to trigger for _every_ conn:
               (http-kit/on-close hk-ch
                 (fn [status]
                   (encore/swap-in! conns_ [uid :ajax client-uuid]
                     (fn [[hk-ch udt-last-connected]] [nil udt-last-connected]))

                   (let [udt-disconnected (encore/now-udt)]
                     (go
                      ;; Allow some time for possible poller reconnects:
                      (<! (async/timeout 5000))
                      (let [disconnected?
                            (encore/swap-in! conns_ [:ajax]
                              (fn [m] ; {<uid> {<client-uuid> [<?hk-ch> _]}
                                (let [[_ ?udt-last-connected]
                                      (get-in m [uid client-uuid])
                                      disconnected?
                                      (and ?udt-last-connected ; Not yet gc'd
                                           (>= udt-disconnected
                                               ?udt-last-connected))]
                                  (if-not disconnected?
                                    (encore/swapped m false)
                                    (let [new (dissoc (get m uid) client-uuid)]
                                      (encore/swapped
                                       (if (empty? new)
                                         (dissoc m uid) ; Gc
                                         (assoc  m uid new))
                                       true))))))]
                        (when disconnected?
                          (upd-connected-uids!))))))))))))}))

;;;; Client

      
                                           
                  
                                                 
                                                   
                                                                                  
                        
                                                
                                                                            

      
                                               
               
                             
                                                
                                                
                                       

      
                                                         
                                      
                                                
                                                                             
                                                                               
                             
                  
                        
                                                          
                       
                                            
                                                                        
                                        
                    

      
                      
                                                 
                                                                                    
                                                 
                                                                                  
                                      
                                                                               
                                                

      
                                                                 
                               
                            
                                                 
                                                          
                                   
                  

                                     
                                  
                                    
                            
                         
                          
                        
                     
                                                                     
                                     

      
                            
                                      
                         
                            
                       
                          

                                                     
                                                                          
                                                                     
                         
           
                      
                         
                                                     
                                       
                                                                    
                                         
                                             
                                        
                                                           
                                                
                                                        
                                                                 
              
                                    
                                      
                             
                             
                                              
                            
                                                                          
                                         
                                        
                         

                                        
                                                         
                                                              
                              
                     
                     
                                                
                                                           
                                                                           
                                           
                                                   
                                                              

                                                
                                                 
                                                                            
                                          
                 
                          
                               
                                                                           
                                                                   
                             
                                                 
                                                                             
                                                      
                                                                            
                                                  
                                                       
                                                     
                                    
                                                                                   
                                       
                                                                              
                                                 
                                                                                  
                              
                              
                                        
                                             
                              
                                                                       
                                                               
                                                    
                                   
                                                                            
                      
                                                                      
                                          

                                 

                                           
                        
         
             

      
                                                       
                                             
           
                        
                            
                                                     
                                       
                                                                    
                                         
                                             
                                                        
                            
                                                           
                                                
           
                               
                                              
                                                               
                   
                                        
                                                                    
                                                                        
                                         
                                                        
                                                 

                                                
                      
                                     
                                                     
                                                   
                                                         

                                     
                                                         
                                             
                                                        
                                                                             
                                                  

                               

                                      
                                                                              
                                                                               
                                                                               
                
                    
                                                                             
                                            

                     
                     
                                                
                                
                                                              
                              
                                                   
                                                              
                                

                                                       
                     
                                      
                                                
                                                                      
                                                                     
                                                                
                                                       
                             
                                            
                                                  
                                                          
                                      

                                                                                 
                                                
                                                               
                                                                        
                                                     
                                                        

                                                 
                                                                                  
                                                                                  
                          

                         
                                                              
                                                                        
                                                      
          
          

      
                                     
                                                                      
                                                                             
                                        

      
                          
                                                                                      
                                
                                                                         
                                                                    
                                                                      
                                                

                                                                            
                                           
        
                                 
                                                         
                                 
                                                                                 
                                 
                                   
                             

                                          
                 
                                                                        

                               
                                            

                                                                             
                                                      
                                           
                                            

            
           
                               
                         
                                               
                                                                  
                                
                                          

                             
                                                                       
                                                       
                           
                                                               
                                                        
                                          

                                                    
                                 
                                                                          
                                                                                

              
                 
                                         
               
                   
                                                                             
                                                                                      
                                                                          

;;;; Routers

     
(defn start-chsk-router-loop! [event-msg-handler ch]
  (go-loop []
    (try
      (let [event-msg (<! ch)]
        (try
          (timbre/tracef "Event-msg: %s" event-msg)
          (event-msg-handler event-msg ch)
          (catch Throwable t
            (timbre/errorf t "Chsk-router-loop handling error: %s" event-msg))))
      (catch Throwable t
        (timbre/errorf t "Chsk-router-loop channel error!")))
    (recur)))

      
                                                
             
                                      
                                                                       
                                                      
                

;;;;;;;;;;;; This file autogenerated from src/taoensso/sente.cljx
